function md=meshyams(md,varargin);
%MESHYAMS - Build model of Antarctica by refining according to observed velocity error estimator
%
%   Usage:
%      md=meshyams(md,varargin);
%      where varargin is a lit of paired arguments. 
%      arguments can be: 'domainoutline': Argus file containing the outline of the domain to be meshed
%      arguments can be: 'velocities': matlab file containing the velocities [m/yr]
%      optional arguments: 'groundeddomain': Argus file containing the outline of the grounded ice
%                          this option is used to minimize the metric on water (no refinement)
%      optional arguments: 'resolution': initial mesh resolution [m]
%      optional arguments: 'nsteps': number of steps of mesh adaptation
%      optional arguments: 'epsilon': average interpolation error wished [m/yr]
%      optional arguments: 'hmin': minimum edge length
%      optional arguments: 'hmanx': maximum edge
%      
%
%   Examples:
%      md=meshyams(md,'domainoutline','Domain.exp','velocities','vel.mat');
%      md=meshyams(md,'domainoutline','Domain.exp','velocities','vel.mat','groundeddomain','ground.exp');
%      md=meshyams(md,'domainoutline','Domain.exp','velocities','vel.mat','groundeddomain','ground.exp','nsteps',6,'epsilon',2,'hmin',500,'hmax',30000);

%recover options
yamsoptions=recover_options(varargin{:});

%add default options
yamsoptions=process_mesh_options(yamsoptions);

%recover some fields
domainoutline=yamsoptions.domainoutline;
groundeddomain=yamsoptions.groundeddomain;
velocities=yamsoptions.velocities;
resolution=yamsoptions.resolution;
gradation=yamsoptions.gradation;
nsteps=yamsoptions.nsteps;
epsilon=yamsoptions.epsilon;
hmin=yamsoptions.hmin;
hmax=yamsoptions.hmax;

%mesh with initial resolution
disp('Initial mesh generation...');
md=mesh(md,domainoutline,resolution);

%load velocities 
disp('loading velocities...');
load(velocities);

%start mesh adaptation
for i=1:nsteps,
	disp(['Iteration #' num2str(i) '/' num2str(nsteps)]);

	%interpolate velocities onto mesh
	disp('   interpolating velocities');
	vx_obs=InterpFromGrid(x_m,y_m,vx,md.x,md.y,0);
	vy_obs=InterpFromGrid(x_m,y_m,vy,md.x,md.y,0);
	field=sqrt(vx_obs.^2+vy_obs.^2);

	%set gridonwater field
	if ~strcmp(groundeddomain,'N/A'),
		gridground=ContourToMesh(md.elements,md.x,md.y,expread(groundeddomain,1),'node',2);
		md.gridonwater=ones(md.numberofgrids,1);
		md.gridonwater(find(gridground))=0;
	else
		md.gridonwater=zeros(md.numberofgrids,1);
	end

	%adapt according to velocities
	disp('   adapting');
	md=YamsCall(md,field,hmin,hmax,gradation(i),epsilon);
end
	
disp(['Final mesh, number of elements: ' num2str(md.numberofelements)]);

%Now, build the connectivity tables for this mesh.
md.nodeconnectivity=NodeConnectivity(md.elements,md.numberofgrids);
md.elementconnectivity=ElementConnectivity(md.elements,md.nodeconnectivity);

%Recreate the segments
disp('Creating segments...');
elementconnectivity=md.elementconnectivity;
elementonboundary=double(elementconnectivity(:,end)==0);
pos=find(elementonboundary);
num_segments=length(pos);
segments=zeros(num_segments,3);
for i=1:num_segments,
	el1=pos(i);
	els2=elementconnectivity(el1,find(elementconnectivity(el1,:)));
	flag=intersect(md.elements(els2(1),:),md.elements(els2(2),:));
	nods1=md.elements(el1,:);
	nods1(find(nods1==flag))=[];
	segments(i,:)=[nods1 el1];

	ord1=find(nods1(1)==md.elements(el1,:));
	ord2=find(nods1(2)==md.elements(el1,:));

	%swap segment grids if necessary
	if ( (ord1==1 & ord2==2) | (ord1==2 & ord2==3) | (ord1==3 & ord2==1) ),
		temp=segments(i,1);
		segments(i,1)=segments(i,2);
		segments(i,2)=temp;
	end
	segments(i,1:2)=fliplr(segments(i,1:2));
end
md.segments=segments;

%Fill in rest of fields:
md.z=zeros(md.numberofgrids,1);
md.gridonboundary=zeros(md.numberofgrids,1); md.gridonboundary(md.segments(:,1:2))=1;
md.gridonbed=ones(md.numberofgrids,1);
md.gridonsurface=ones(md.numberofgrids,1);
md.elementonbed=ones(md.numberofelements,1);
md.elementonsurface=ones(md.numberofelements,1);
