function md=YamsCall(md,field,hmin,hmax,gradation,epsilon),
%YAMSCALL - call yams
%
%   build a metric using the Hessian of the given field
%   call Yams and the output mesh is plugged onto the model
%   -hmin = minimum edge length (m)
%   -hmax = maximum edge length (m)
%   -gradation = maximum edge length gradation between 2 elements
%   -epsilon = average error on each element (m/yr)
%
%   Usage:
%      md=YamsCall(md,field,hmin,hmax,epsilon)
%
%   Example:
%      md=YamsCall(md,md.vel_obs,1500,10^8,1.3,0.9);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% Build metric %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

global ISSM_DIR

t1=clock; fprintf('%s','      computing Hessian...');
%initialization
index=md.elements;
numberofgrids=md.numberofgrids;
numberofelements=md.numberofelements;
gradx=zeros(numberofgrids,1);
grady=zeros(numberofgrids,1);
metric=zeros(numberofelements,3);

%2d geometric parameter (do not change)
scale=2/9; 

%build some usefull variables
line=index(:);
summation=1/3*ones(3,1);
linesize=3*numberofelements;

%get areas and  nodal functions coefficients N(x,y)=alpha x + beta y + gamma 
[alpha beta]=GetNodalFunctionsCoeff(index,md.x,md.y);
areas=GetAreas(index,md.x,md.y);

%Compute gradient for each element
grad_elx=sum(field(index).*alpha,2); 
grad_ely=sum(field(index).*beta,2);

%update weights that holds the volume of all the element holding the grid i
weights=sparse(line,ones(linesize,1),repmat(areas,3,1),numberofgrids,1);

%Compute gradient for each grid (average of the elements around)
gradx=sparse(line,ones(linesize,1),repmat(areas.*grad_elx,3,1),numberofgrids,1);
grady=sparse(line,ones(linesize,1),repmat(areas.*grad_ely,3,1),numberofgrids,1);
gradx=gradx./weights;
grady=grady./weights;

%Compute hessian for each element
hessian=[sum(gradx(index).*alpha,2) sum(grady(index).*alpha,2) sum(grady(index).*beta,2)];

%Compute Hessian on the nodes (average of the elements around)
hessian=[sparse(line,ones(linesize,1),repmat(areas.*hessian(:,1),3,1),numberofgrids,1)./weights ...
	sparse(line,ones(linesize,1),repmat(areas.*hessian(:,2),3,1),numberofgrids,1)./weights ...
	sparse(line,ones(linesize,1),repmat(areas.*hessian(:,3),3,1),numberofgrids,1)./weights ];
t2=clock;fprintf('%s\n',[' done (' num2str(etime(t2,t1)) ' seconds)']);

t1=clock; fprintf('%s','      computing metric...');
%first, find the eigen values of eah line of H=[hessian(i,1) hessian(i,2); hessian(i,2)  hessian(i,3)]
a=hessian(:,1); b=hessian(:,2); d=hessian(:,3);
lambda1=0.5*((a+d)+sqrt(4*b.^2+(a-d).^2));
lambda2=0.5*((a+d)-sqrt(4*b.^2+(a-d).^2));
pos1=find(lambda1==0);
pos2=find(lambda2==0);

%Modify the eigen values to control the shape of the elements
lambda1=min(max(abs(lambda1)*scale/epsilon,1/hmax^2),1/hmin^2);
lambda2=min(max(abs(lambda2)*scale/epsilon,1/hmax^2),1/hmin^2);

%compute eigen vectors
norm1=sqrt(8*b.^2+2*(d-a).^2+2*(d-a).*sqrt((a-d).^2+4*b.^2));
v1x=2*b./norm1;
v1y=((d-a)+sqrt((a-d).^2+4*b.^2))./norm1;
norm2=sqrt(8*b.^2+2*(d-a).^2-2*(d-a).*sqrt((a-d).^2+4*b.^2));
v2x=2*b./norm2;
v2y=((d-a)-sqrt((a-d).^2+4*b.^2))./norm2;

%Compute new metric (for each node M=V*Lambda*V^-1)
metric=full([(v1x.*v2y-v1y.*v2x).^(-1).*(lambda1.*v2y.*v1x-lambda2.*v1y.*v2x) ...
	(v1x.*v2y-v1y.*v2x).^(-1).*(lambda1.*v1y.*v2y-lambda2.*v1y.*v2y) ...
	(v1x.*v2y-v1y.*v2x).^(-1).*(-lambda1.*v2x.*v1y+lambda2.*v1x.*v2y)]);

%some corrections for 0 eigen values
metric(pos1,:)=repmat([1/hmin^2 0 1/hmin^2],length(pos1),1);
metric(pos2,:)=repmat([1/hmin^2 0 1/hmin^2],length(pos2),1);

if any(isnan(metric)),
	error('YamsCall error message: NaN in the metric...')
end
t2=clock;fprintf('%s\n',[' done (' num2str(etime(t2,t1)) ' seconds)']);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% Build FILES %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

t1=clock; fprintf('%s','      writing initial mesh files...');
save -ascii carre0.met  metric

fid=fopen('carre0.mesh','w');

%initialiation
fprintf(fid,'\n%s\n%i\n','MeshVersionFormatted',1);

%dimension
fprintf(fid,'\n%s\n%i\n','Dimension',2);

%Vertices
fprintf(fid,'\n%s\n%i\n\n','Vertices',md.numberofgrids);
fprintf(fid,'%8g %8g %i\n',[md.x md.y zeros(md.numberofgrids,1)]');

%Triangles
fprintf(fid,'\n\n%s\n%i\n\n','Triangles',md.numberofelements);
fprintf(fid,'%i %i %i %i\n',[md.elements zeros(md.numberofelements,1)]');

%Corners
pos=find(md.gridonboundary);
fprintf(fid,'\n\n%s\n%i\n\n','Corners',length(pos));
fprintf(fid,'%i\n',pos');

%close
fclose(fid);
t2=clock;fprintf('%s\n',[' done (' num2str(etime(t2,t1)) ' seconds)']);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% call Yams %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%call yams
fprintf('%s\n','      call Yams...');
addpath([ISSM_DIR '/externalpackages/yams']);
system(['yams2-linux -O 1 -v 1 -ecp -hgrad ' num2str(gradation)  ' carre0 carre1']);
rmpath([ISSM_DIR '/externalpackages/yams']);

%plug new mesh
t1=clock; fprintf('\n%s','      reading final mesh files...');
Tria=load('carre1.tria');
Coor=load('carre1.coor');
md.x=Coor(:,1);
md.y=Coor(:,2);
md.z=zeros(size(Coor,1),1);
md.elements=Tria;

md.numberofgrids=size(Coor,1);
md.numberofelements=size(Tria,1);
t2=clock;fprintf('%s\n',[' done (' num2str(etime(t2,t1)) ' seconds)']);

%clean up:
system('rm carre0.mesh carre0.met carre1.tria carre1.coor carre1.meshb');
