/*! \file  ContourtoMesh
    \brief: takes an  contour file, and figures out which nodes or elements from the mesh  
    are inside this contour. 
	usage:
	[in_nod,in_elem]=ContourToMesh(index,x,y,contours,interptype,edgevalue);
	
	input:

		index,x,y: delaunay triangulation.
		contours: structure holding sets of vertices making open contours. 
		interptype: string definining type of interpolation ('element', or 'node', or 'element and node');
		edgevalue: integer (0, 1 or 2) defining the value associated to the nodes on the edges of the polygons

	output:
		in_nod: vector of flags (0 or 1), of size nods if interptype is set to 'node' or 'element and node', 
				or of size 0 otherwise.
		in_elem: vector of flags (0 or 1), of size nel if interptype is set to 'element' or 'element and node', 
				or of size 0 otherwise.
*/
	
#include "./ContourToMesh.h"

WRAPPER(ContourToMesh){

	int i,j;

	/* required input: */
	int     edgevalue;
	double *index      = NULL;
	int     nel;
	double *x          = NULL;
	int     nods;
	double *y          = NULL;
	char   *interptype = NULL;
	char* contourname  = NULL;
	DataSet* contours  = NULL;

	/* output: */
	SeqVec<double> *in_nod  = NULL;
	SeqVec<double> *in_elem = NULL;

	/*Boot module: */
	MODULEBOOT();

	/*checks on output arguments on the matlab side: */
	#ifdef _HAVE_MATLAB_MODULES_
	if(nlhs!=1 && nlhs!=2){
		ContourToMeshUsage();
		_error_("usage. See above");
	}
	#endif
	/*check on input arguments: */
	if(nrhs!=NRHS){
		ContourToMeshUsage();
		_error_("usage. See above");
	}


	/*Fetch inputs: */
	FetchData(&index,&nel,NULL,INDEX);
	FetchData(&x,&nods,NULL,X);
	FetchData(&y,NULL,NULL,Y);
	FetchData(&edgevalue,EDGEVALUE);
	FetchData(&contourname,CONTOURNAME);
	contours=DomainOutlineRead<double>(contourname);

	/*Fetch  interptype: */
	FetchData(&interptype,INTERPTYPE);

	/*Run interpolation routine: */
	ContourToMeshx( &in_nod,&in_elem,index,x,y,contours,interptype,nel,nods,edgevalue);

	/* output: */
	if (strcmp(interptype,"node")==0){
		WriteData(PLHS0,in_nod);
	}
	else if (strcmp(interptype,"element")==0){
		WriteData(PLHS0,in_elem);
	}
	else if (strcmp(interptype,"element and node")==0){
		WriteData(PLHS0,in_nod);
		WriteData(PLHS1,in_elem);
	}
	else _error_("wrong interpolation type");

	/*end module: */
	MODULEEND();
	
}

void ContourToMeshUsage(void)//{{{1
{
	_printLine_("CONTOURTOMESH - Flag the elements or nodes inside a contour");
	_printLine_("");
	_printLine_("      Usage: ");
	_printLine_("         [in_nod,in_elem]=ContourToMesh(index,x,y,contourname,interptype,edgevalue)\n");
	_printLine_("");
	_printLine_("         index,x,y: mesh triangulation.");
	_printLine_("         contourname: name of .exp file containing the contours.");
	_printLine_("         interptype: string definining type of interpolation ('element', or 'node').");
	_printLine_("         edgevalue: integer (0, 1 or 2) defining the value associated to the nodes on the edges of the polygons.");
	_printLine_("         in_nod: vector of flags (0 or 1), of size nods if interptype is set to 'node' or 'element and node', ");
	_printLine_("            or of size 0 otherwise.");
	_printLine_("         in_elem: vector of flags (0 or 1), of size nel if interptype is set to 'element' or 'element and node', ");
	_printLine_("            or of size 0 otherwise.");
	_printLine_("");
	_printLine_("      Example: ");
	_printLine_("         in_nod=ContourToMesh(md.elements,md.x,md.y,'Contour.exp','node',1)");
	_printLine_("         in_elements=ContourToMesh(md.elements,md.x,md.y,'Contour.exp','element',0)");
	_printLine_("         [in_nodes,in_elements]=ContourToMesh(md.elements,md.x,md.y,'Contour.exp','element and node',0)");
	_printLine_("");
}
//}}}
