/*!\file InterpFromGridToMesh.c
 * \brief: data interpolation from a list of (x,y,values) into mesh vertices
 
	InterpFromGridToMesh.c

	usage:
	data_mesh=InterpFromGridToMesh(x,y,data,x_mesh,y_mesh);
	
	where:

		input:
		x,y: coordinates of matrix data
		data - matrix holding the data to be interpolated onto the mesh.
		x_mesh,y_mesh: coordinates of the mesh vertices onto which we interpolate.
		
		output: 
		data_mesh:  vector of mesh interpolated data.
*/
	
#include "./InterpFromGridToMesh.h"

void mexFunction( int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[]) {

	int i,j;

	/*input: */
	double *x = NULL;
	double *y = NULL;
	int     x_rows,y_rows;
	double *data  = NULL;
	int     data_rows,data_cols;
	double *x_mesh = NULL;
	double *y_mesh = NULL;
	int     x_mesh_rows,y_mesh_rows;
	double  default_value;
	int     interpolationenum;

	/* output: */
	SeqVec<double>*  data_mesh=NULL;

	/*Boot module: */
	MODULEBOOT();

	/*checks on arguments on the matlab side: */
	if((nlhs!=NLHS) || (nrhs!=6 && nrhs!=7)){
		InterpFromGridToMeshUsage();
		_error_("usage. See above");
	}

	/*Input datasets: */
	FetchData(&x,&x_rows,NULL,XHANDLE);
	FetchData(&y,&y_rows,NULL,YHANDLE);
	FetchData(&data,&data_rows,&data_cols,DATAHANDLE);
	FetchData(&x_mesh,&x_mesh_rows,NULL,XMESHHANDLE);
	FetchData(&y_mesh,&y_mesh_rows,NULL,YMESHHANDLE);
	FetchData(&default_value,DEFAULTHANDLE);

	/* Run core computations: */
	if(nrhs==7){
		FetchData(&interpolationenum,INTERPENUM);
		InterpFromGridToMeshx(&data_mesh, x, x_rows,  y, y_rows, data, data_rows,data_cols, x_mesh, y_mesh, x_mesh_rows,default_value,interpolationenum);
	}
	else{
		InterpFromGridToMeshx(&data_mesh, x, x_rows,  y, y_rows, data, data_rows,data_cols, x_mesh, y_mesh, x_mesh_rows,default_value);
	}

	/*Write data: */
	WriteData(DATAMESH,data_mesh);

	/*end module: */
	MODULEEND();
}

void InterpFromGridToMeshUsage(void)
{
	_pprintLine_("INTERPFROMGRIDTOMESH - interpolation from a grid onto a list of points");
	_pprintLine_("");
	_pprintLine_("   This function is a multi-threaded mex file that interpolates a field");
	_pprintLine_("   defined on a grid onto a list of points");
	_pprintLine_("");
	_pprintLine_("   Usage:");
	_pprintLine_("      data_mesh=InterpFromGridToMesh(x,y,data,x_mesh,y_mesh,default_value);");
	_pprintLine_("");
	_pprintLine_("      data: matrix holding the data to be interpolated onto the mesh.");
	_pprintLine_("      x,y: coordinates of matrix data. (x and y must be in increasing order)");
	_pprintLine_("      x_mesh,y_mesh: coordinates of the points onto which we interpolate.");
	_pprintLine_("      default_value: default value if no data is found (holes).");
	_pprintLine_("      data_mesh: vector of mesh interpolated data.");
	_pprintLine_("");
	_pprintLine_("   Example:");
	_pprintLine_("      load('velocities.mat');");
	_pprintLine_("      md.inversion.vx_obs=InterpFromGridToMesh(x_n,y_m,vx,md.mesh.x,md.mesh.y,0);");
	_pprintLine_("");
}
