function md=setdefaultparameters(md);
%SETDEFAULTPARAMETERS - plug default parameters onto model
%
%   Although the model parameterization should be done in
%   the parameter file, some parameters are initialized here
%   with a default value.
%   These default values can be changed if necessary.
%
%   Usage:
%      md=setdefaultparameters(md);


%Materials parameters

%ice density (kg/m^3)
md.rho_ice=917;

%water density (kg/m^3)
md.rho_water=1023;

%firn density (kg/m^3)
md.rho_firn=830;

%ice heat capacity cp (J/kg/K)
md.heatcapacity=2093;

%ice latent heat of fusion L (J/kg)
md.latentheat=3.34*10^5;

%ice thermal conductivity lamda (W/m/K)
md.thermalconductivity=2.4;

%the melting point of ice at 1 atmosphere of pressure in K
md.meltingpoint=273.15;

%rate of change of melting point with pressure (K/Pa)
md.beta=9.8*10^-8;

%mixed layer (ice-water interface) heat capacity (J/kg/K)
md.mixed_layer_capacity=3974;

%thermal exchange velocity (ice-water interface) (m/s)
md.thermal_exchange_velocity=1.00*10^-4;

%Physical parameters

%acceleration due to gravity (m/s^2)
md.g=9.81;

%converstion from year to seconds
md.yts=365*24*3600;

%drag type (2=elastic)
md.drag_type=2;

%Solver parameters

%relative convergence criterion ((vel(n-1)-vel(n))/vel(n))
md.eps_rel=0.01;

%absolute convergence criterion (max(vel(n-1)-vel(n)) (m/year)
md.eps_abs=10;

%sparsity
md.sparsity=0.001;

%the connectivity is the avergaded number of nodes linked to a
%given node through an edge. This connectivity is used to initially
%allocate memory to the stiffness matrix. A value of 16 seems to
%give a good memory/time ration. This value can be checked in
%trunk/test/Miscellaneous/runme.m
md.connectivity=25;

%lowmem??
md.lowmem=0;

%coefficient to update the viscosity between each iteration of
%a diagnostic according to the following formula
%viscosity(n)=viscosity(n)+viscosity_overshoot(viscosity(n)-viscosity(n-1))
md.viscosity_overshoot=0.3;

%parameter used to print temporary results (convergence criterion,
%current step,...)
md.debug=0;

%Stokes mesh
md.stokesreconditioning=10^13;

%Penalties

%the penalty offset is the power used to compute the value
%of the penalty as follows
%kappa=max(stiffness matrix)*10^penalty_offset
md.penalty_offset=3;

%penalty_melting is the penalty used to keep T<Tpmp
%kappa=penalty_melting
md.penalty_melting=10^7;

%in some solutions, it might be needed to stop a run when only
%a few constraints remain unstable. For thermal computation, this
%parameter is often used.
md.penalty_lock=10;

%minimum thickness to avoid stiffness singularity, used in
%the parameter file
md.minh=1;

%in some cases, it might be needed to stop a run when only
%a few constraints remain unstable. For thermal computation, this
%parameter is often used.
md.min_thermal_constraints=0;

%Transient parameters

%time between 2 time steps
md.dt=md.yts/2;

%number of time steps = md.ndt / md.dt
md.ndt=10*md.dt;

%the artificial diffusivity is used in prognostic to avoid
%numerical wiggles of the solution.
md.artificial_diffusivity=1;

%do we use an adaptative time stepping?
md.timestepping=0;

%minimum thickness change between 2 time steps (m)
md.deltaH=10^-2;

%maximum thickness change between 2 time steps (m)
md.DeltaH=50;

%minimum temperature change between 2 time steps (K)
md.deltaT=10^-2;

%maximum temperature change between 2 time steps (K)
md.DeltaT=1;

%When the previous criteria are fulfilled, if the time steps
%can be accelerated as follows
%dt = timeacc * dt
md.timeacc=2;

%When the previous criteria are fulfilled, if the time steps
%can be reduced as follows
%dt = timedec * dt
md.timedec=.5;

%Control

%parameter to be inferred by control methods (only
%drag and B are supported yet)
md.control_type='drag';

%number of steps in the control methods
md.nsteps=20;

%maximum number of iteration in the optimization algorithm for
%each step
md.maxiter=20*ones(md.nsteps,1);

%tolerance used by the optimization algorithm
md.tolx=10^-4;

%the inversed parameter is updated as follows:
%new_par=old_par + optscal(n)*C*gradient with C in [0 1];
%usually the optscal must be of the order of magnitude of the 
%inversed parameter (10^8 for B, 50 for drag) and can be decreased
%after the first iterations
md.optscal=50*ones(md.nsteps,1);

%mincontrolconstraint is the minimum acceptable value of the
%inversed parameter (B>0 for example)
md.mincontrolconstraint=0;

%maxcontrolconstraint is the maximum acceptable value of the
%inversed parameter (drag<200 for example)
md.maxcontrolconstraint=200;

%several misfits can be used:
%0 = absolute misfit
%1 = relative misfit
%2 = logarithmic misfit
md.fit=0*ones(md.nsteps,1);

%minimum velocity to avoid the misfit to be singular
md.epsvel=eps;

%averaged velocity used to scale the logarithmic Misfit (000 m/an)
md.meanvel=1000/(365*24*3600);

%the value of plot can be set to 1 if the user wants to see
%the direction and the updated parameter after each control
%method iteration (only for cielo serial)
md.plot=1;

%Parallelisation parameters

%cluster set as 'none' for serial
md.cluster='none';

%this option can be activated (1) to load automatically the results
%onto the model after a parallel run by waiting for the lock file
%that is generated once the solution has converged
md.waitonlock=1;

%number of processors to be used (if parallel)
md.np=8;

%minutes
md.time=60;

%queue name for cluster
md.queue='shortc';

%exculsive mode (share no cpus)
md.exclusive=1;

%?????
md.scheduler_configuration='local';

%prerun allocation cleanup
md.alloc_cleanup=1;

%mesh2grid parameter
md.mesh2grid_parameters={'vel'};

%interpolation type
md.mesh2grid_interpolation={'node'};

%filter
md.mesh2grid_filter={'average'};

%coordinate of the corner east (actually WEST...)
md.mesh2grid_cornereast=0;

%coordinate of the corner north
md.mesh2grid_cornernorth=0;

%x posting
md.mesh2grid_xposting=1000;

%y posting
md.mesh2grid_yposting=1000;

%number of lines
md.mesh2grid_nlines=1000;

%number of columns
md.mesh2grid_ncolumns=1000;

%window size
md.mesh2grid_windowsize=1;

%the string of solverstring is used directly by PETSc to solve finite element
%systems KU=F. By default, we use MUMPS solver
md.solverstring=' -mat_type aijmumps -ksp_type preonly -pc_type lu -mat_mumps_icntl_14 40 ';

%Ice solver: 'general' for Matlab's default solver (or 'lu' or 'sholesky')
md.solver_type='general';
