/*!\file SolverxSeq
 * \brief implementation of sequential solver using the GSL librarie
 */

#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif
#include <cstring>

#include "./Solverx.h"
#include "../../shared/shared.h"
#include "../../include/include.h"
#include "../../io/io.h"

#ifdef _HAVE_GSL_
#include <gsl/gsl_linalg.h>
#endif

#ifdef _HAVE_ADOLC_
#include "../../shared/Numerics/adolc_edf.h"
#endif

void SolverxSeq(SeqVec** puf,SeqMat* Kff, SeqVec* pf, Parameters* parameters){/*{{{*/

	#ifdef _HAVE_GSL_
	/*Intermediary: */
	int M,N,N2,s;
	SeqVec *uf = NULL;
	IssmDouble *x  = NULL;

	Kff->GetSize(&M,&N);
	pf->GetSize(&N2);

	if(N!=N2)_error_("Right hand side vector of size " << N2 << ", when matrix is of size " << M << "-" << N << " !");
	if(M!=N)_error_("Stiffness matrix should be square!");

#ifdef _HAVE_ADOLC_
	SolverxSeq(&x,Kff->matrix,pf->vector,N,parameters);
#else
	SolverxSeq(&x,Kff->matrix,pf->vector,N);
#endif
	uf=new SeqVec(x,N);

	/*Assign output pointers:*/
	*puf=uf;

	#else
		_error_("GSL support not compiled in!");
	#endif

}/*}}}*/
#ifdef _HAVE_ADOLC_

int EDF_for_solverx(int n, IssmPDouble *x, int m, IssmPDouble *y) {
    if(m*(m+1)!=n)_error_("Stiffness matrix should be square!");
    SolverxSeq(&y,x, x+m*m, m);
    return 0;
}

void SolverxSeq(IssmDouble** pX,IssmDouble* A,IssmDouble* B,int n, Parameters* parameters){//{{{
	// pack inputs to conform to the EDF-prescribed interface
        IssmDouble*  adoubleEDF_X=xNew<IssmDouble>(n*(n+1)); // packed inputs, i.e. matrix and right hand side
        for(int i=0; i<n*n;i++)adoubleEDF_X[i]    =A[i]; // pack matrix
        for(int i=0; i<n;  i++)adoubleEDF_X[i+n*n]=B[i]; // pack the right hand side
        IssmPDouble* pdoubleEDF_X=xNew<IssmPDouble>(n*(n+1)); // provide space to transfer inputs during call_ext_fct
	IssmPDouble* pdoubleEDF_Y=xNew<IssmPDouble>(n);       // provide space to transfer outputs during call_ext_fct
	// call the wrapped solver through the registry entry we retrieve from parameters
	call_ext_fct(dynamic_cast<GenericParam<Adolc_edf> * >(parameters->FindParamObject(AdolcParamEnum))->GetParameterValue().myEDF_for_solverx_p,
	             n*(n+1), pdoubleEDF_X, adoubleEDF_X,
	             n, pdoubleEDF_Y, B);
	xDelete(adoubleEDF_X);
	xDelete(pdoubleEDF_X);
	xDelete(pdoubleEDF_Y);
}
/*}}}*/
#endif
void SolverxSeq(IssmPDouble** pX,IssmPDouble* A,IssmPDouble* B,int n){ //{{{
	#ifdef _HAVE_GSL_
	/*GSL Matrices and vectors: */
	int              s;
	gsl_matrix_view  a;
	gsl_vector_view  b;
	gsl_vector      *x = NULL;
	gsl_permutation *p = NULL;
	/*A will be modified by LU decomposition. Use copy*/
	double* Acopy = xNew<double>(n*n);
	xMemCpy<double>(Acopy,A,n*n);

	/*Initialize gsl matrices and vectors: */
	a = gsl_matrix_view_array (Acopy,n,n);
	b = gsl_vector_view_array (B,n);
	x = gsl_vector_alloc (n);

	/*Run LU and solve: */
	p = gsl_permutation_alloc (n);
	gsl_linalg_LU_decomp (&a.matrix, p, &s);
	gsl_linalg_LU_solve (&a.matrix, p, &b.vector, x);

	//printf ("x = \n");
	//gsl_vector_fprintf (stdout, x, "%g");

	/*Copy result*/
	double* X = xNew<double>(n);
	memcpy(X,gsl_vector_ptr(x,0),n*sizeof(double));

	/*Clean up and assign output pointer*/
	xDelete<double>(Acopy);
	gsl_permutation_free(p);
	gsl_vector_free(x);
	*pX=X;
	#endif
}
/*}}}*/
