%MATDAMAGEICE class definition
%
%   Usage:
%      matdamageice=matdamageice();

classdef matdamageice
	properties (SetAccess=public) 
		rho_ice                    = 0.;
		rho_water                  = 0.;
		rho_freshwater             = 0.;
		mu_water                   = 0.;
		heatcapacity               = 0.;
		latentheat                 = 0.;
		thermalconductivity        = 0.;
		meltingpoint               = 0.;
		beta                       = 0.;
		mixed_layer_capacity       = 0.;
		thermal_exchange_velocity  = 0.;
		rheology_B   = NaN;
		rheology_n   = NaN;
		rheology_Z   = NaN;
		rheology_law = '';
	end
	methods
		function obj = matdamageice(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%ice density (kg/m^3)
			obj.rho_ice=917.;

			%ocean water density (kg/m^3)
			obj.rho_water=1023.;

			%fresh water density (kg/m^3)
			obj.rho_freshwater=1000.;

			%water viscosity (N.s/m^2)
			obj.mu_water=0.001787;  

			%ice heat capacity cp (J/kg/K)
			obj.heatcapacity=2093.;

			%ice latent heat of fusion L (J/kg)
			obj.latentheat=3.34*10^5;

			%ice thermal conductivity (W/m/K)
			obj.thermalconductivity=2.4;

			%the melting point of ice at 1 atmosphere of pressure in K
			obj.meltingpoint=273.15;

			%rate of change of melting point with pressure (K/Pa)
			obj.beta=9.8*10^-8;

			%mixed layer (ice-water interface) heat capacity (J/kg/K)
			obj.mixed_layer_capacity=3974.;

			%thermal exchange velocity (ice-water interface) (m/s)
			obj.thermal_exchange_velocity=1.00*10^-4;

			%Rheology law: what is the temperature dependence of B with T
			%available: none, paterson and arrhenius
			obj.rheology_law='Paterson';
		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{
			md = checkfield(md,'materials.rho_ice','>',0);
			md = checkfield(md,'materials.rho_water','>',0);
			md = checkfield(md,'materials.rho_freshwater','>',0);
			md = checkfield(md,'materials.mu_water','>',0);
			md = checkfield(md,'materials.rheology_B','>',0,'size',[md.mesh.numberofvertices 1]);
			md = checkfield(md,'materials.rheology_n','>',0,'size',[md.mesh.numberofelements 1]);
			md = checkfield(md,'materials.rheology_Z','>',0,'size',[md.mesh.numberofvertices 1]);
			md = checkfield(md,'materials.rheology_law','values',{'None' 'Paterson' 'Arrhenius'});
		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   Materials:\n'));

			fielddisplay(obj,'rho_ice','ice density [kg/m^3]');
			fielddisplay(obj,'rho_water','ocean water density [kg/m^3]');
			fielddisplay(obj,'rho_freshwater','fresh water density [kg/m^3]');
			fielddisplay(obj,'mu_water','water viscosity [N s/m^2]');
			fielddisplay(obj,'heatcapacity','heat capacity [J/kg/K]');
			fielddisplay(obj,'thermalconductivity','ice thermal conductivity [W/m/K]');
			fielddisplay(obj,'meltingpoint','melting point of ice at 1atm in K');
			fielddisplay(obj,'latentheat','latent heat of fusion [J/m^3]');
			fielddisplay(obj,'beta','rate of change of melting point with pressure [K/Pa]');
			fielddisplay(obj,'mixed_layer_capacity','mixed layer capacity [W/kg/K]');
			fielddisplay(obj,'thermal_exchange_velocity','thermal exchange velocity [m/s]');
			fielddisplay(obj,'rheology_B','flow law parameter [Pa/s^(1/n)]');
			fielddisplay(obj,'rheology_n','Glen''s flow law exponent');
			fielddisplay(obj,'rheology_Z','rheology multiplier');
			fielddisplay(obj,'rheology_law','law for the temperature dependance of the rheology: ''None'', ''Paterson'' or ''Arrhenius''');
		end % }}}
		function marshall(obj,fid) % {{{
			WriteData(fid,'enum',MaterialsEnum(),'data',MatdamageiceEnum(),'format','Integer');
			WriteData(fid,'object',obj,'class','materials','fieldname','rho_ice','format','Double');
			WriteData(fid,'object',obj,'class','materials','fieldname','rho_water','format','Double');
			WriteData(fid,'object',obj,'class','materials','fieldname','rho_freshwater','format','Double');
			WriteData(fid,'object',obj,'class','materials','fieldname','mu_water','format','Double');
			WriteData(fid,'object',obj,'class','materials','fieldname','heatcapacity','format','Double');
			WriteData(fid,'object',obj,'class','materials','fieldname','latentheat','format','Double');
			WriteData(fid,'object',obj,'class','materials','fieldname','thermalconductivity','format','Double');
			WriteData(fid,'object',obj,'class','materials','fieldname','meltingpoint','format','Double');
			WriteData(fid,'object',obj,'class','materials','fieldname','beta','format','Double');
			WriteData(fid,'object',obj,'class','materials','fieldname','mixed_layer_capacity','format','Double');
			WriteData(fid,'object',obj,'class','materials','fieldname','thermal_exchange_velocity','format','Double');
			WriteData(fid,'object',obj,'class','materials','fieldname','rheology_B','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'class','materials','fieldname','rheology_n','format','DoubleMat','mattype',2);
			WriteData(fid,'object',obj,'class','materials','fieldname','rheology_Z','format','DoubleMat','mattype',1);
			WriteData(fid,'data',StringToEnum(obj.rheology_law),'enum',MaterialsRheologyLawEnum(),'format','Integer');
		end % }}}
	end
end
