/*!\file Matpar.c
 * \brief: implementation of the Matpar object
 */

#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <stdio.h>
#include <string.h>
#include "../../classes.h"
#include "../../../shared/shared.h"
#include "../../../include/include.h"
#include "../../../EnumDefinitions/EnumDefinitions.h"
		
/*Matpar constructors and destructor*/
/*FUNCTION Matpar::Matpar() {{{*/
Matpar::Matpar(){
	return;
}
/*}}}*/
/*FUNCTION Matpar::Matpar(int matpar_mid,IoModel* iomodel){{{*/
Matpar::Matpar(int matpar_mid, IoModel* iomodel){

	this->mid = matpar_mid;
	iomodel->Constant(&this->rho_ice,MaterialsRhoIceEnum);
	iomodel->Constant(&this->rho_water,MaterialsRhoWaterEnum);
	iomodel->Constant(&this->rho_freshwater,MaterialsRhoFreshwaterEnum);
	iomodel->Constant(&this->mu_water,MaterialsMuWaterEnum);
	iomodel->Constant(&this->heatcapacity,MaterialsHeatcapacityEnum);
	iomodel->Constant(&this->thermalconductivity,MaterialsThermalconductivityEnum);
	iomodel->Constant(&this->latentheat,MaterialsLatentheatEnum);
	iomodel->Constant(&this->beta,MaterialsBetaEnum);
	iomodel->Constant(&this->meltingpoint,MaterialsMeltingpointEnum);
	iomodel->Constant(&this->referencetemperature,ConstantsReferencetemperatureEnum);
	iomodel->Constant(&this->mixed_layer_capacity,MaterialsMixedLayerCapacityEnum);
	iomodel->Constant(&this->thermal_exchange_velocity,MaterialsThermalExchangeVelocityEnum);
	iomodel->Constant(&this->g,ConstantsGEnum);
	
	iomodel->Constant(&this->hydro_CR,HydrologyCREnum);
	iomodel->Constant(&this->kn,HydrologyKnEnum);
	iomodel->Constant(&this->hydro_n,HydrologyNEnum);
	iomodel->Constant(&this->hydro_p,HydrologyPEnum);
	iomodel->Constant(&this->hydro_q,HydrologyQEnum);
	this->inputs=NULL; /*not used here*/
}
/*}}}*/
/*FUNCTION Matpar::~Matpar() {{{*/
Matpar::~Matpar(){
	return;
}
/*}}}*/

/*Object virtual functions definitions:*/
/*FUNCTION Matpar::Echo {{{*/
void Matpar::Echo(void){

	_printLine_("Matpar:");
	_printLine_("   mid: " << mid);
	_printLine_("   rho_ice: " << rho_ice);
	_printLine_("   rho_water: " << rho_water);
	_printLine_("   rho_freshwater: " << rho_freshwater);
	_printLine_("   mu_water: " << mu_water);
	_printLine_("   heatcapacity: " << heatcapacity);
	_printLine_("   thermalconductivity: " << thermalconductivity);
	_printLine_("   latentheat: " << latentheat);
	_printLine_("   beta: " << beta);
	_printLine_("   meltingpoint: " << meltingpoint);
	_printLine_("   referencetemperature: " << referencetemperature);
	_printLine_("   mixed_layer_capacity: " << mixed_layer_capacity);
	_printLine_("   thermal_exchange_velocity: " << thermal_exchange_velocity);
	_printLine_("   g: " << g);
	return;
}
/*}}}*/
/*FUNCTION Matpar::DeepEcho {{{*/
void Matpar::DeepEcho(void){

	this->Echo();
}		
/*}}}*/
/*FUNCTION Matpar::Id {{{*/
int    Matpar::Id(void){ return mid; }
/*}}}*/
/*FUNCTION Matpar::MyRank {{{*/
int    Matpar::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/
/*FUNCTION Matpar::ObjectEnum{{{*/
int Matpar::ObjectEnum(void){

	return MatparEnum;

}
/*}}}*/
/*FUNCTION Matpar::copy {{{*/
Object* Matpar::copy() {
	return new Matpar(*this); 
}
/*}}}*/

/*Update virtual functions definitions:*/
/*FUNCTION Matpar::InputUpdateFromVector(IssmDouble* vector, int name, int type) {{{*/
void   Matpar::InputUpdateFromVector(IssmDouble* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matpar::InputUpdateFromVector(int* vector, int name, int type) {{{*/
void   Matpar::InputUpdateFromVector(int* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matpar::InputUpdateFromVector(bool* vector, int name, int type) {{{*/
void   Matpar::InputUpdateFromVector(bool* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matpar::InputUpdateFromVectorDakota(IssmDouble* vector, int name, int type) {{{*/
void   Matpar::InputUpdateFromVectorDakota(IssmDouble* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matpar::InputUpdateFromVectorDakota(int* vector, int name, int type) {{{*/
void   Matpar::InputUpdateFromVectorDakota(int* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matpar::InputUpdateFromVectorDakota(bool* vector, int name, int type) {{{*/
void   Matpar::InputUpdateFromVectorDakota(bool* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matpar::InputUpdateFromMatrixDakota(int* vector, int name, int type) {{{*/
void  Matpar::InputUpdateFromMatrixDakota(IssmDouble* matrix, int nrows, int ncols,int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matpar::InputUpdateFromConstant(IssmDouble constant, int name) {{{*/
void   Matpar::InputUpdateFromConstant(IssmDouble constant, int name){

	switch(name){
		case MaterialsRhoIceEnum:
			this->rho_ice=constant;
			break;
		case MaterialsRhoWaterEnum:
			this->rho_water=constant;
			break;
		case MaterialsRhoFreshwaterEnum:
			this->rho_freshwater=constant;
			break;
		case MaterialsMuWaterEnum:
			this->mu_water=constant;
			break;
		case MaterialsHeatcapacityEnum:
			this->heatcapacity=constant;
			break;
		case MaterialsThermalconductivityEnum:
			this->thermalconductivity=constant;
			break;
		case  MaterialsLatentheatEnum:
			this->latentheat=constant;
			break;
		case  MaterialsBetaEnum:
			this->beta=constant;
			break;
		case  MaterialsMeltingpointEnum:
			this->meltingpoint=constant;
			break;
		case  ConstantsReferencetemperatureEnum:
			this->referencetemperature=constant;
			break;
		case  MaterialsMixedLayerCapacityEnum:
			this->mixed_layer_capacity=constant;
			break;
		case  MaterialsThermalExchangeVelocityEnum:
			this->thermalconductivity=constant;
			break;
		case  ConstantsGEnum:
			this->g=constant;
			break;
		default: 
			break;
	}

}
/*}}}*/
/*FUNCTION Matpar::InputUpdateFromConstant(int constant, int name) {{{*/
void   Matpar::InputUpdateFromConstant(int constant, int name){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matpar::InputUpdateFromConstant(bool constant, int name) {{{*/
void   Matpar::InputUpdateFromConstant(bool constant, int name){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matpar::InputUpdateFromSolution{{{*/
void   Matpar::InputUpdateFromSolution(IssmDouble* solution){
	/*Nothing updated yet*/
}
/*}}}*/

/*Matpar management: */
/*FUNCTION Matpar::Configure {{{*/
void  Matpar::Configure(Elements* elementsin){

	/*nothing done yet!*/

}
/*}}}*/
/*FUNCTION Matpar::GetBeta {{{*/
IssmDouble Matpar::GetBeta(){
	return beta;
}
/*}}}*/
/*FUNCTION Matpar::GetG {{{*/
IssmDouble Matpar::GetG(){
	return g;
}
/*}}}*/
/*FUNCTION Matpar::GetHeatCapacity {{{*/
IssmDouble Matpar::GetHeatCapacity(){
	return heatcapacity;
}
/*}}}*/
/*FUNCTION Matpar::GetLatentHeat {{{*/
IssmDouble Matpar::GetLatentHeat(){
	return latentheat;
}
/*}}}*/
/*FUNCTION Matpar::GetMeltingPoint {{{*/
IssmDouble Matpar::GetMeltingPoint(){
	return meltingpoint;
}
/*}}}*/
/*FUNCTION Matpar::GetReferenceTemperature {{{*/
IssmDouble Matpar::GetReferenceTemperature(){
	return referencetemperature;
}
/*}}}*/
/*FUNCTION Matpar::GetMixedLayerCapacity {{{*/
IssmDouble Matpar::GetMixedLayerCapacity(){
	return mixed_layer_capacity;
}
/*}}}*/
/*FUNCTION Matpar::GetRhoIce {{{*/
IssmDouble Matpar::GetRhoIce(){
	
	return rho_ice;
}
/*}}}*/
/*FUNCTION Matpar::GetRhoWater {{{*/
IssmDouble Matpar::GetRhoWater(){
	return rho_water;
}
/*}}}*/
/*FUNCTION Matpar::GetRhoFreshwater {{{*/
IssmDouble Matpar::GetRhoFreshwater(){
	return rho_freshwater;
}
/*}}}*/
/*FUNCTION Matpar::GetMuWater {{{*/
IssmDouble Matpar::GetMuWater(){
	return mu_water;
}
/*}}}*/
/*FUNCTION Matpar::GetThermalConductivity {{{*/
IssmDouble Matpar::GetThermalConductivity(){
	return thermalconductivity;
}
/*}}}*/
/*FUNCTION Matpar::GetThermalExchangeVelocity {{{*/
IssmDouble Matpar::GetThermalExchangeVelocity(){
	return thermal_exchange_velocity;
}
/*}}}*/
/*FUNCTION Matpar::GetKn {{{*/		 
IssmDouble Matpar::GetKn(){			 
	return kn;		 
}		 
/*}}}*/			 
/*FUNCTION Matpar::GetHydrologyP {{{*/			 
IssmDouble Matpar::GetHydrologyP(){		 
	return hydro_p;			 
}		 
/*}}}*/			 
/*FUNCTION Matqar::GetHydrologyQ {{{*/			 
IssmDouble Matpar::GetHydrologyQ(){		 
	return hydro_q;			 
}		 
/*}}}*/			 
/*FUNCTION Matpar::GetHydrologyCR {{{*/		 
IssmDouble Matpar::GetHydrologyCR(){		 
	return hydro_CR;		 
}		 
/*}}}*/			 
/*FUNCTION Matpar::GetHydrologyN {{{*/			 
IssmDouble Matpar::GetHydrologyN(){		 
	return hydro_n;			 
}		 
/*}}}*/ 
/*FUNCTION Matpar::TMeltingPoint {{{*/
IssmDouble Matpar::TMeltingPoint(IssmDouble pressure){
	return meltingpoint-beta*pressure;
}
/*}}}*/
/*FUNCTION Matpar::PureIceEnthalpy{{{*/
IssmDouble Matpar::PureIceEnthalpy(IssmDouble pressure){
	return heatcapacity*(TMeltingPoint(pressure)-referencetemperature);
}
/*}}}*/
/*FUNCTION Matpar::GetEnthalpyDiffusionParameter{{{*/
IssmDouble Matpar::GetEnthalpyDiffusionParameter(IssmDouble enthalpy,IssmDouble pressure){
	if(enthalpy<PureIceEnthalpy(pressure)){
		return thermalconductivity/(rho_ice*heatcapacity);
	}
	else{
		return 0.1*thermalconductivity/(rho_ice*heatcapacity);
	}
}
/*}}}*/
/*FUNCTION Matpar::EnthalpyToThermal {{{*/
void Matpar::EnthalpyToThermal(IssmDouble* ptemperature,IssmDouble* pwaterfraction,IssmDouble enthalpy,IssmDouble pressure){

	/*Ouput*/
	IssmDouble temperature,waterfraction;
	
	if(enthalpy<PureIceEnthalpy(pressure)){
		temperature=referencetemperature+enthalpy/heatcapacity;
		waterfraction=0;
	}
	else{
		temperature=TMeltingPoint(pressure);
		waterfraction=(enthalpy-PureIceEnthalpy(pressure))/latentheat;
	}

	/*Assign output pointers:*/
	*pwaterfraction=waterfraction;
	*ptemperature=temperature;
}
/*}}}*/
/*FUNCTION Matpar::ThermalToEnthalpy {{{*/
void Matpar::ThermalToEnthalpy(IssmDouble * penthalpy,IssmDouble temperature,IssmDouble waterfraction,IssmDouble pressure){

	/*Ouput*/
	IssmDouble enthalpy;
	
	if(temperature<TMeltingPoint(pressure)){
		enthalpy=heatcapacity*(temperature-referencetemperature);
	}
	else{
		enthalpy=PureIceEnthalpy(pressure)+latentheat*waterfraction;
	}

	/*Assign output pointers:*/
	*penthalpy=enthalpy;
}
/*}}}*/
