%HYDROLOGY class definition
%
%   Usage:
%      hydrology=hydrology();

classdef hydrology
	properties (SetAccess=public) 
		spcwatercolumn = NaN;
		n              = 0;
		CR             = 0;
		p              = 0;
		q              = 0;
		kn             = 0;
		stabilization  = 0;
	end
	methods
		function obj = hydrology(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%Parameters from Johnson's 2002 thesis, section 3.5.4			 
			obj.n=.02;			 
			obj.CR=.01;
			obj.p=2;		 
			obj.q=1;		 
			obj.kn=0;

			%Type of stabilization to use 0:nothing 1:artificial_diffusivity
			obj.stabilization=1;
		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{

			%Early return
			if ~ismember(HydrologyAnalysisEnum(),analyses), return; end

			md = checkfield(md,'hydrology.spcwatercolumn','forcing',1);
			md = checkfield(md,'hydrology.stabilization','>=',0);
		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   hydrology solution parameters:'));

			fielddisplay(obj,'spcwatercolumn','water thickness constraints (NaN means no constraint)');
			fielddisplay(obj,'n','Manning roughness coefficient');
			fielddisplay(obj,'CR','tortuosity parameter');
			fielddisplay(obj,'p','dimensionless exponent in Manning velocity formula');
			fielddisplay(obj,'q','dimensionless exponent in Manning velocity formula');
			fielddisplay(obj,'kn','parameter in effective pressure formula');
			fielddisplay(obj,'stabilization','artificial diffusivity (default is 1). can be more than 1 to increase diffusivity.');

		end % }}}
		function marshall(obj,fid) % {{{
			WriteData(fid,'object',obj,'fieldname','spcwatercolumn','format','DoubleMat','mattype',1);
			WriteData(fid,'object',obj,'fieldname','n','format','Double');
			WriteData(fid,'object',obj,'fieldname','CR','format','Double');
			WriteData(fid,'object',obj,'fieldname','p','format','Double');
			WriteData(fid,'object',obj,'fieldname','q','format','Double');
			WriteData(fid,'object',obj,'fieldname','kn','format','Double');
			WriteData(fid,'object',obj,'fieldname','stabilization','format','Double');
		end % }}}
	end
end
