#module imports
from fielddisplay import fielddisplay
from EnumDefinitions import *
from StringToEnum import StringToEnum
from checkfield import *
from WriteData import *

class materials(object):
	"""
	MATERIALS class definition

	   Usage:
	      materials=materials();
	"""

	#properties
	def __init__(self):
		# {{{ Properties
		self.rho_ice                    = 0
		self.rho_water                  = 0
		self.rho_freshwater             = 0
		self.mu_water                   = 0
		self.heatcapacity               = 0
		self.latentheat                 = 0
		self.thermalconductivity        = 0
		self.meltingpoint               = 0
		self.beta                       = 0
		self.mixed_layer_capacity       = 0
		self.thermal_exchange_velocity  = 0
		self.rheology_B   = float('NaN')
		self.rheology_n   = float('NaN')
		self.rheology_law = ''

		self.setdefaultparameters()
		#}}}
	def __repr__(self):
		# {{{ Display
		string="   Materials:"

		string="%s\n\n%s"%(string,fielddisplay(self,"rho_ice","ice density [kg/m^3]"))
		string="%s\n%s"%(string,fielddisplay(self,"rho_water","water density [kg/m^3]"))
		string="%s\n%s"%(string,fielddisplay(self,"rho_freshwater","fresh water density [kg/m^3]"))
		string="%s\n%s"%(string,fielddisplay(self,"mu_water","water viscosity [N s/m^2]"))
		string="%s\n%s"%(string,fielddisplay(self,"heatcapacity","heat capacity [J/kg/K]"))
		string="%s\n%s"%(string,fielddisplay(self,"thermalconductivity","ice thermal conductivity [W/m/K]"))
		string="%s\n%s"%(string,fielddisplay(self,"meltingpoint","melting point of ice at 1atm in K"))
		string="%s\n%s"%(string,fielddisplay(self,"latentheat","latent heat of fusion [J/m^3]"))
		string="%s\n%s"%(string,fielddisplay(self,"beta","rate of change of melting point with pressure [K/Pa]"))
		string="%s\n%s"%(string,fielddisplay(self,"mixed_layer_capacity","mixed layer capacity [W/kg/K]"))
		string="%s\n%s"%(string,fielddisplay(self,"thermal_exchange_velocity","thermal exchange velocity [m/s]"))
		string="%s\n%s"%(string,fielddisplay(self,"rheology_B","flow law parameter [Pa/s^(1/n)]"))
		string="%s\n%s"%(string,fielddisplay(self,"rheology_n","Glen's flow law exponent"))
		string="%s\n%s"%(string,fielddisplay(self,"rheology_law","law for the temperature dependance of the rheology: 'None', 'Paterson' or 'Arrhenius'"))

		return string
		#}}}
	def setdefaultparameters(self):
		# {{{setdefaultparameters
		#ice density (kg/m^3)
		self.rho_ice=917

		#ocean water density (kg/m^3)
		self.rho_water=1023

		#fresh water density (kg/m^3)
		self.rho_freshwater=1000

		#water viscosity (N.s/m^2)
		self.mu_water=0.001787  

		#ice heat capacity cp (J/kg/K)
		self.heatcapacity=2093

		#ice latent heat of fusion L (J/kg)
		self.latentheat=3.34*10**5

		#ice thermal conductivity (W/m/K)
		self.thermalconductivity=2.4

		#the melting point of ice at 1 atmosphere of pressure in K
		self.meltingpoint=273.15

		#rate of change of melting point with pressure (K/Pa)
		self.beta=9.8*10**-8

		#mixed layer (ice-water interface) heat capacity (J/kg/K)
		self.mixed_layer_capacity=3974

		#thermal exchange velocity (ice-water interface) (m/s)
		self.thermal_exchange_velocity=1.00*10**-4

		#Rheology law: what is the temperature dependence of B with T
		#available: none, paterson and arrhenius
		self.rheology_law='Paterson'
		return self
		#}}}

	def checkconsistency(self,md,solution,analyses):    # {{{
		md = checkfield(md,'materials.rho_ice','>',0)
		md = checkfield(md,'materials.rho_water','>',0)
		md = checkfield(md,'materials.rho_freshwater','>',0)
		md = checkfield(md,'materials.mu_water','>',0)
		md = checkfield(md,'materials.rheology_B','>',0,'size',[md.mesh.numberofvertices])
		md = checkfield(md,'materials.rheology_n','>',0,'size',[md.mesh.numberofelements])
		md = checkfield(md,'materials.rheology_law','values',['None','Paterson','Arrhenius'])
		return md
	# }}}

	def marshall(self,fid):    # {{{
		WriteData(fid,'object',self,'fieldname','rho_ice','format','Double')
		WriteData(fid,'object',self,'fieldname','rho_water','format','Double')
		WriteData(fid,'object',self,'fieldname','rho_freshwater','format','Double')
		WriteData(fid,'object',self,'fieldname','mu_water','format','Double')
		WriteData(fid,'object',self,'fieldname','heatcapacity','format','Double')
		WriteData(fid,'object',self,'fieldname','latentheat','format','Double')
		WriteData(fid,'object',self,'fieldname','thermalconductivity','format','Double')
		WriteData(fid,'object',self,'fieldname','meltingpoint','format','Double')
		WriteData(fid,'object',self,'fieldname','beta','format','Double')
		WriteData(fid,'object',self,'fieldname','mixed_layer_capacity','format','Double')
		WriteData(fid,'object',self,'fieldname','thermal_exchange_velocity','format','Double')
		WriteData(fid,'object',self,'fieldname','rheology_B','format','DoubleMat','mattype',1)
		WriteData(fid,'object',self,'fieldname','rheology_n','format','DoubleMat','mattype',2)
		WriteData(fid,'data',StringToEnum(self.rheology_law)[0],'enum',MaterialsRheologyLawEnum(),'format','Integer')
	# }}}

