/*!\file FemModel.c
 * \brief: implementation of the FemModel object
 */


#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <stdio.h>
#include "../Container/Container.h"
#include "../modules/ModelProcessorx/ModelProcessorx.h"
#include "../io/io.h"
#include "./classes.h"
#include "../include/include.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../modules/modules.h"

/*Object constructors and destructor*/
/*FUNCTION FemModel::constructor {{{*/
FemModel::FemModel(char* inputfilename, char* outputfilename, const int in_solution_type,const int* analyses,const int nummodels){

	/*intermediary*/
	int         i;
	int         analysis_type;
	FILE       *IOMODEL = NULL;
	extern int  my_rank;

	/*Open input file on cpu 0: */
	if(my_rank==0) IOMODEL = pfopen(inputfilename ,"rb");

	/*Initialize internal data: */
	this->nummodels=nummodels;
	this->solution_type=in_solution_type;
	this->analysis_counter=nummodels-1; //point to last analysis_type carried out.
	this->results=new Results(); //not initialized by CreateDataSets
	
	/*Dynamically allocate whatever is a list of length nummodels: */
	analysis_type_list=xNew<int>(nummodels);

	/*Initialize: */
	for(i=0;i<nummodels;i++)analysis_type_list[i]=analyses[i];

	/*create datasets for all analyses*/
	ModelProcessorx(&this->elements,&this->nodes,&this->vertices,&this->materials,&this->constraints,&this->loads,&this->parameters,IOMODEL,this->solution_type,nummodels,analyses);

	/*do the post-processing of the datasets to get an FemModel that can actually run analyses: */
	for(i=0;i<nummodels;i++){

		if(VerboseMProcessor()) _pprintLine_("   Processing finite element model of analysis " << EnumToStringx(analysis_type_list[i]) << ":");
		analysis_type=analysis_type_list[i];
		this->SetCurrentConfiguration(analysis_type);
	
		if(i==0){
			if(VerboseMProcessor()) _pprintLine_("      creating vertex degrees of freedom");
			VerticesDofx(vertices,parameters); //only call once, we only have one set of vertices
		}

		if(VerboseMProcessor()) _pprintLine_("      resolving node constraints");
		SpcNodesx(nodes,constraints,parameters,analysis_type); 

		if(VerboseMProcessor()) _pprintLine_("      creating nodal degrees of freedom");
		NodesDofx(nodes,parameters,analysis_type);
	
		if(VerboseMProcessor()) _pprintLine_("      configuring element and loads");
		ConfigureObjectsx(elements, loads, nodes, vertices, materials,parameters);
	}
	
	/*Close input file descriptors: */
	if(my_rank==0) pfclose(IOMODEL,inputfilename);

	/*Add output file name to parameters: */
	this->parameters->AddObject(new StringParam(OutputfilenameEnum,outputfilename));

}

/*}}}*/
/*FUNCTION FemModel::destructor {{{*/
FemModel::~FemModel(){

	/*Intermediary*/
	int i;

	/*Delete all the datasets: */
	xDelete<int>(analysis_type_list);
	delete elements;
	delete nodes;
	delete vertices;
	delete constraints;
	delete loads;
	delete materials;
	delete parameters;
	delete results;

}
/*}}}*/

/*Object management*/
/*FUNCTION FemModel::Echo {{{*/
void FemModel::Echo(void){

	_printLine_("FemModel echo: ");
	_printLine_("   number of fem models: " << nummodels);
	_printLine_("   analysis_type_list: ");
	for(int i=0;i<nummodels;i++)_printLine_("     " << i << ": " << EnumToStringx(analysis_type_list[i]));
	_printLine_("   current analysis_type: ");
	_printLine_("     " << analysis_counter << ": " << EnumToStringx(analysis_type_list[analysis_counter]));

}
/*}}}*/

/*Numerics: */
/*FUNCTION FemModel::SetCurrentConfiguration(int configuration_type,int analysis_type){{{*/
void FemModel::SetCurrentConfiguration(int configuration_type,int analysis_type){

	/*Use configuration_type to setup the analysis counter, the configurations of objects etc ... but use 
	 * analysis_type to drive the element numerics. This allows for use of 1 configuration_type for several 
	 * analyses. For example: do a SurfaceSlopeX, SurfaceSlopeY, BedSlopeX and BedSlopeY analysis using the 
	 * Slope configuration.*/

	int found=-1;
	for(int i=0;i<nummodels;i++){
		if (analysis_type_list[i]==configuration_type){
			found=i;
			break;
		}
	}
	if(found!=-1) analysis_counter=found;
	else _error_("Could not find alias for analysis_type " << EnumToStringx(configuration_type) << " in list of FemModel analyses");

	/*Now, plug analysis_counter and analysis_type inside the parameters: */
	this->parameters->SetParam(analysis_counter,AnalysisCounterEnum);
	this->parameters->SetParam(analysis_type,AnalysisTypeEnum);
	this->parameters->SetParam(configuration_type,ConfigurationTypeEnum);

	/*configure elements, loads and nodes, for this new analysis: */
	this->elements->SetCurrentConfiguration(elements,loads, nodes,vertices, materials,parameters);
	this->nodes->SetCurrentConfiguration(elements,loads, nodes,vertices, materials,parameters);
	this->loads->SetCurrentConfiguration(elements, loads, nodes,vertices, materials,parameters);

	#ifdef _HAVE_PETSC_
	/*take care of petsc options, that depend on this analysis type (present only after model processor)*/
	if(this->parameters->Exist(PetscOptionsStringsEnum)){
		PetscOptionsFromAnalysis(this->parameters,analysis_type);
		if(VerboseSolver()) _pprintLine_("      petsc Options set for analysis type: " << EnumToStringx(analysis_type));
	}
	#endif

}
/*}}}*/
/*FUNCTION FemModel::SetCurrentConfiguration(int configuration_type){{{*/
void FemModel::SetCurrentConfiguration(int configuration_type){

	/*overload: analysis_type = configuration_type: */
	this->SetCurrentConfiguration(configuration_type,configuration_type);
}
/*}}}*/
