%INITIALIZATION class definition
%
%   Usage:
%      initialization=initialization();

classdef initialization
	properties (SetAccess=public) 
		vx            = NaN;
		vy            = NaN;
		vz            = NaN;
		vel           = NaN;
		pressure      = NaN;
		temperature   = NaN;
		watercolumn   = NaN;
		waterfraction = NaN;
	end
	methods
		function obj = initialization(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

		end % }}}
		function md = checkconsistency(obj,md,solution,analyses) % {{{
			if ismember(DiagnosticHorizAnalysisEnum,analyses)
				if ~(isnan(md.initialization.vx) | isnan(md.initialization.vy)),
					md = checkfield(md,'initialization.vx','NaN',1,'size',[md.mesh.numberofvertices 1]);
					md = checkfield(md,'initialization.vy','NaN',1,'size',[md.mesh.numberofvertices 1]);
				end
			end
			if ismember(PrognosticAnalysisEnum,analyses),
				md = checkfield(md,'initialization.vx','NaN',1,'size',[md.mesh.numberofvertices 1]);
				md = checkfield(md,'initialization.vy','NaN',1,'size',[md.mesh.numberofvertices 1]);
			end
			if ismember(HydrologyAnalysisEnum,analyses),
				md = checkfield(md,'initialization.watercolumn','NaN',1,'size',[md.mesh.numberofvertices 1]);
			end
			if ismember(BalancethicknessAnalysisEnum,analyses),
				md = checkfield(md,'initialization.vx','NaN',1,'size',[md.mesh.numberofvertices 1]);
				md = checkfield(md,'initialization.vy','NaN',1,'size',[md.mesh.numberofvertices 1]);
				%Triangle with zero velocity
				if any(sum(abs(md.initialization.vx(md.mesh.elements)),2)==0 & sum(abs(md.initialization.vy(md.mesh.elements)),2)==0)
					md = checkmessage(md,'at least one triangle has all its vertices with a zero velocity');
				end
			end
			if ismember(ThermalAnalysisEnum,analyses),
				md = checkfield(md,'initialization.vx','NaN',1,'size',[md.mesh.numberofvertices 1]);
				md = checkfield(md,'initialization.vy','NaN',1,'size',[md.mesh.numberofvertices 1]);
				md = checkfield(md,'initialization.vz','NaN',1,'size',[md.mesh.numberofvertices 1]);
				md = checkfield(md,'initialization.pressure','NaN',1,'size',[md.mesh.numberofvertices 1]);
			end
			if (ismember(EnthalpyAnalysisEnum,analyses) & md.thermal.isenthalpy) | solution==EnthalpySolutionEnum,
				md = checkfield(md,'initialization.waterfraction','>=',0,'size',[md.mesh.numberofvertices 1]);
			end
		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   initial field values:'));

			fielddisplay(obj,'vx','x component of velocity');
			fielddisplay(obj,'vy','y component of velocity');
			fielddisplay(obj,'vz','z component of velocity');
			fielddisplay(obj,'vel','velocity norm');
			fielddisplay(obj,'pressure','pressure field');
			fielddisplay(obj,'temperature','temperature in Kelvins');
			fielddisplay(obj,'watercolumn','thickness of subglacial water');
			fielddisplay(obj,'waterfraction','fraction of water in the ice');

		end % }}}
		function marshall(obj,fid) % {{{
			WriteData(fid,'data',obj.vx,'format','DoubleMat','mattype',1,'enum',VxEnum);
			WriteData(fid,'data',obj.vy,'format','DoubleMat','mattype',1,'enum',VyEnum);
			WriteData(fid,'data',obj.vz,'format','DoubleMat','mattype',1,'enum',VzEnum);
			WriteData(fid,'data',obj.pressure,'format','DoubleMat','mattype',1,'enum',PressureEnum);
			WriteData(fid,'data',obj.temperature,'format','DoubleMat','mattype',1,'enum',TemperatureEnum);
			WriteData(fid,'data',obj.watercolumn,'format','DoubleMat','mattype',1,'enum',WatercolumnEnum);
			WriteData(fid,'data',obj.waterfraction,'format','DoubleMat','mattype',1,'enum',WaterfractionEnum);
		end % }}}
	end
end
