/* \file Mode.h
 * \brief  Header file defining the Model structure and processing of input data.
 * \sa Model.cpp
 */

#ifndef _MODEL_H
#define _MODEL_H

#include "../io/io.h"
#include "../DataSet/DataSet.h"
#include "../toolkits/toolkits.h"

struct Model {

	char*   repository;
	char*   meshtype;
	char*   analysis_type;
	char*   solverstring;

	/*2d mesh: */
	int     numberofelements;
	int     numberofnodes;
	double* x;
	double* y;
	double* z;
	double* elements;
	double* elements_type;

	/*3d mesh: */
	int     numberofnodes2d;
	int     numberofelements2d;
	double* elements2d;
	double* deadgrids;
	int     numlayers;
	double* uppernodes;


	/*results: */
	double* vx;
	double* vy;
	double* vz;
	double* pressure;

	/*observations: */
	double*  vx_obs;
	double*  vy_obs;

	/*geometry: */
	double* elementonbed;
	double* elementonsurface;
	double* gridonbed;
	double* gridonsurface;
	double* thickness;
	double* surface;
	double* bed;
	double* elementoniceshelf;

	/*friction: */
	int     drag_type;
	double* drag;
	double* p;
	double* q;

	/*boundary conditions: */
	//diagnostic
	int     numberofsegs_diag;
	double* segmentonneumann_diag;
	double* neumannvalues_diag;
	double* gridondirichlet_diag;
	double* dirichletvalues_diag;
	//prognostic
	double* segmentonneumann_prog;
	double* neumannvalues_prog;
	double* gridondirichlet_prog;
	double* dirichletvalues_prog;
	//prognostic2
	double* segmentonneumann_prog2;
	double* neumannvalues_prog2;
	//thermal
	double* gridondirichlet_thermal;
	double* dirichletvalues_thermal;
	double* geothermalflux;
	
	/*materials: */
	double  rho_water,rho_ice;
	double  g;
	double* B;
	double* n;

	/*control methods: */
	char*	control_type;

	/*solution parameters: */
	double* fit;
	double  meanvel,epsvel;
	int     artificial_diffusivity;
	int     nsteps;
	double  tolx;
	double* maxiter;
	double  mincontrolconstraint;
	double  maxcontrolconstraint;
	int     debug;
	int     plot;
	double  eps_rel;
	double  eps_abs;
	double  dt,ndt;
	double  penalty_offset;
	double  penalty_melting;
	int     penalty_lock;
	double  sparsity;
	int     connectivity;
	int     lowmem;
	double* optscal;
	double  yts;
	double  viscosity_overshoot;
	int     waitonlock;

	/*thermal parameters: */
	double beta;
	double meltingpoint;
	double latentheat;
	double  heatcapacity,thermalconductivity;
	int    min_thermal_constraints;
	double mixed_layer_capacity;
	double thermal_exchange_velocity;

	/*rifts: */
	int      numrifts;
	int*     riftsnumpenaltypairs;
	double** riftspenaltypairs;
	int*     riftsfill;
	double*  riftsfriction;

	/*penalties: */
	int      numpenalties;
	double*  penalties;
	
	/*basal: */
	double*  melting;
	double*  accumulation;


	/*exterior data: */
	int* epart; /*!element partitioning.*/
	int* my_grids; /*! grids that belong to this cpu*/
	double* my_bordergrids; /*! grids that belong to this cpu, and some other cpu also*/
	int*     penaltypartitioning;

};



	/*constructor and destructor: */
	Model*	NewModel(void);
	void    DeleteModel( Model** pthis);

	/*Echo: */
	void    ModelEcho(Model* model,int which_part,int rank);

	/*Initialization with matlab workspace data, or marshall binary data: */
	int	    ModelInit(Model** pmodel,ConstDataHandle model_handle);

	/*Creation of fem datasets: general drivers*/
	void	CreateElementsNodesAndMaterials(DataSet** pelements,DataSet** pnodes, DataSet** pmaterials, Model* model,ConstDataHandle model_handle);
	void    CreateConstraints(DataSet** pconstraints, Model* model,ConstDataHandle model_handle);
	void    CreateLoads(DataSet** ploads, Model* model, ConstDataHandle model_handle);
	void    CreateParameters(DataSet** pparameters,Model* model,ConstDataHandle model_handle);

	
	/*Create of fem datasets: specialised drivers: */
	
	/*diagnostic horizontal*/
	void	CreateElementsNodesAndMaterialsDiagnosticHoriz(DataSet** pelements,DataSet** pnodes, DataSet** pmaterials, Model* model,ConstDataHandle model_handle);
	void	CreateConstraintsDiagnosticHoriz(DataSet** pconstraints,Model* model,ConstDataHandle model_handle);
	void    CreateLoadsDiagnosticHoriz(DataSet** ploads, Model* model, ConstDataHandle model_handle);

	/*diagnostic vertical*/
	void	CreateElementsNodesAndMaterialsDiagnosticVert(DataSet** pelements,DataSet** pnodes, DataSet** pmaterials, Model* model,ConstDataHandle model_handle);
	void	CreateConstraintsDiagnosticVert(DataSet** pconstraints,Model* model,ConstDataHandle model_handle);
	void    CreateLoadsDiagnosticVert(DataSet** ploads, Model* model, ConstDataHandle model_handle);
	
	/*control:*/
	void    CreateParametersControl(DataSet* parameters,Model* model,ConstDataHandle model_handle,int* pcount);


#endif  /* _MODEL_H */
