%PFE class definition
%
%   Usage:
%      cluster=discover();
%      cluster=discover('np',3);
%      cluster=discover('np',3,'login','username');

classdef discover 
    properties (SetAccess=public)  
		 % {{{
		 name=oshostname();
		 login='';
		 numnodes=20;
		 cpuspernode=8; 
		 port=1025;
		 queue='general';
		 time=12*60;
		 processor='west';
		 codepath='';
		 executionpath='';
		 interactive=0;
		 bbftp=0;
		 numstreams=8;
		 hyperthreading=0;
	 end
	 properties (SetAccess=private) 
		 np=20*8;
		 % }}}
	 end
	 methods
		 function cluster=discover(varargin) % {{{

			 %initialize cluster using default settings if provided
			 if (exist('discover_settings')==2), discover_settings; end

			 %use provided options to change fields
			 cluster=AssignObjectFields(pairoptions(varargin{:}),cluster);
		 end
		 %}}}
		 function disp(cluster) % {{{
			 %  display the object
			 disp(sprintf('class ''%s'' object ''%s'' = ',class(cluster),inputname(1)));
			 disp(sprintf('    name: %s',cluster.name));
			 disp(sprintf('    login: %s',cluster.login));
			 disp(sprintf('    port: %i',cluster.port));
			 disp(sprintf('    numnodes: %i',cluster.numnodes));
			 disp(sprintf('    cpuspernode: %i',cluster.cpuspernode));
			 disp(sprintf('    np: %i',cluster.cpuspernode*cluster.numnodes));
			 disp(sprintf('    queue: %s',cluster.queue));
			 disp(sprintf('    time: %i',cluster.time));
			 disp(sprintf('    processor: %s',cluster.processor));
			 disp(sprintf('    codepath: %s',cluster.codepath));
			 disp(sprintf('    executionpath: %s',cluster.executionpath));
			 disp(sprintf('    interactive: %i',cluster.interactive));
			 disp(sprintf('    hyperthreading: %i',cluster.hyperthreading));
		 end
		 %}}}
		 function md = checkconsistency(cluster,md,solution,analyses) % {{{

			 available_queues={'general_long','general','general_small','debug'};
			 queue_requirements_time=[24*60 12*60 12*60 60];
			 queue_requirements_np=[516 1024 16 32];

			 QueueRequirements(available_queues,queue_requirements_time,queue_requirements_np,cluster.queue,cluster.np,cluster.time)

			 if ( strcmpi(cluster.queue,'general') & cluster.cpuspernode*cluster.numnodes < 17)
				 md = checkmessage(md,'cpus must be great than 17 for general queue');
			 end
			 %now, check cluster.cpuspernode according to processor type
			 if ( strcmpi(cluster.processor,'neha')),
				 if ((cluster.cpuspernode>8 ) | (cluster.cpuspernode<1)),
					 md = checkmessage(md,'cpuspernode should be between 1 and 8 for ''neha'' processors');
				 end
			 elseif strcmpi(cluster.processor,'west'),
				 if ((cluster.cpuspernode>12 ) | (cluster.cpuspernode<1)),
					 md = checkmessage(md,'cpuspernode should be between 1 and 12 for ''west'' processors');
				 end
			 else
				 md = checkmessage(md,'unknown processor type, should be ''neha'' or ''west'' ');
			 end

			 %Miscelaneous
			 if isempty(cluster.login), md = checkmessage(md,'login empty'); end
			 if isempty(cluster.codepath), md = checkmessage(md,'codepath empty'); end
			 if isempty(cluster.executionpath), md = checkmessage(md,'executionpath empty'); end

		 end
		 %}}}
		 function BuildQueueScript(cluster,dirname,modelname,solution,io_gather,isvalgrind,isgprof) % {{{

			 if(isgprof),    disp('gprof not supported by cluster, ignoring...'); end

			 %compute number of processors
			 cluster.np=cluster.numnodes*cluster.cpuspernode;

			 %write queuing script 
			 fid=fopen([modelname '.queue'],'w');
			 fprintf(fid,'#PBS -S /bin/bash\n');
%			 fprintf(fid,'#PBS -N %s\n',modelname);
			 fprintf(fid,'#PBS -l select=%i:ncpus=%i:proc=%s\n',cluster.numnodes,cluster.cpuspernode,cluster.processor);
			 fprintf(fid,'#PBS -l walltime=%i\n',cluster.time*60); %walltime is in seconds.
			 fprintf(fid,'#PBS -q %s \n',cluster.queue);
			 fprintf(fid,'#PBS -W group_list=s1010\n');
			 fprintf(fid,'#PBS -m e\n');
			 fprintf(fid,'#PBS -o %s.outlog \n',modelname);
			 fprintf(fid,'#PBS -e %s.errlog \n\n',modelname);
			 fprintf(fid,'. /usr/share/modules/init/bash\n\n');
			 fprintf(fid,'module load comp/intel-10.1.023\n');
			 fprintf(fid,'module load mpi/impi-3.2.2.006\n');
			 fprintf(fid,'module load lib/mkl-10.1.2.024\n\n');
			 fprintf(fid,'export PATH="$PATH:."\n\n');
			 fprintf(fid,'export MPI_GROUP_MAX=64\n\n');
			 fprintf(fid,'cd $PBS_O_WORKDIR\n\n');
			 fprintf(fid,'mpirun -np %i %s/issm.exe %s $PBS_O_WORKDIR %s\n',cluster.np,cluster.codepath,EnumToString(solution),modelname);
			 if ~io_gather, %concatenate the output files:
				 fprintf(fid,'cat %s.outbin.* > %s.outbin',modelname,modelname);
			 end
			 fclose(fid);

			 %in interactive mode, create a run file, and errlog and outlog file
			 if cluster.interactive,
				 fid=fopen([modelname '.run'],'w');
				 if ~isvalgrind,
					 fprintf(fid,'mpirun -np %i %s/issm.exe %s $PBS_O_WORKDIR %s\n',cluster.np,cluster.codepath,EnumToString(solution),modelname);
				 else
					 fprintf(fid,'mpirun -np %i valgrind --leak-check=full %s/issm.exe %s $PBS_O_WORKDIR %s\n',cluster.np,cluster.codepath,EnumToString(solution),modelname);
				 end
				 if ~io_gather, %concatenate the output files:
					 fprintf(fid,'cat %s.outbin.* > %s.outbin',modelname,modelname);
				 end
				 fclose(fid);
				 fid=fopen([modelname '.errlog'],'w');
				 fclose(fid);
				 fid=fopen([modelname '.outlog'],'w');
				 fclose(fid);
			 end
		 end %}}}
		 function LaunchQueueJob(cluster,modelname,dirname,filelist)% {{{

			 %compress the files into one zip.
			 compressstring=['tar -zcf ' dirname '.tar.gz '];
			 for i=1:numel(filelist),
				 compressstring = [compressstring ' ' filelist{i}];
			 end
			 if cluster.interactive,
				 compressstring = [compressstring ' ' modelname '.run '  modelname '.errlog ' modelname '.outlog '];
			 end
			 system(compressstring);

			 disp('uploading input file and queueing script');
			 if cluster.interactive,
				 directory=[cluster.executionpath '/Interactive' num2str(cluster.interactive)];
			 else 
				 directory=cluster.executionpath;
			 end

			 if ~cluster.bbftp,
				 issmscpout(cluster.name,directory,cluster.login,cluster.port,{[dirname '.tar.gz']});
			 else
				 issmbbftpout(cluster.name,directory,cluster.login,cluster.port,cluster.numstreams,{[dirname '.tar.gz']});
			 end

			 %lauch command, to be executed via ssh
			 if ~cluster.interactive, 
				 launchcommand=['cd ' cluster.executionpath ' && rm -rf ./' dirname ' && mkdir ' dirname ...
					 ' && cd ' dirname ' && mv ../' dirname '.tar.gz ./ && tar -zxf ' dirname '.tar.gz  && qsub ' modelname '.queue '];
			 else
				 launchcommand=['cd ' cluster.executionpath '/Interactive' num2str(cluster.interactive) ' && tar -zxf ' dirname '.tar.gz'];
			 end

			 disp('launching solution sequence on remote cluster');
			 issmssh(cluster.name,cluster.login,cluster.port,launchcommand);
		 end
		 %}}}
		 function Download(cluster,dirname,filelist)% {{{

			 %copy files from cluster to current directory
			 if ~cluster.interactive,
				 directory=[cluster.executionpath '/' dirname '/'];
			 else
				 directory=[cluster.executionpath '/Interactive' num2str(cluster.interactive) '/'];
			 end

			 if ~cluster.bbftp,
				 issmscpin(cluster.name,cluster.login,cluster.port,directory,filelist);
			 else
				 issmbbftpin(cluster.name, cluster.login, cluster.port, cluster.numstreams, directory, filelist);
			 end

		 end %}}}
	end
end
