/*!\file:  control.cpp
 * \brief: control solution
 */ 

#include "../issm.h"
#include "./parallel.h"

#undef __FUNCT__ 
#define __FUNCT__ "control"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

int main(int argc,char* *argv){

	/*I/O: */
	FILE* fid=NULL;
	char* inputfilename=NULL;
	char* outputfilename=NULL;
	char* lockname=NULL;
	int   numberofnodes;
	int   qmu_analysis=0;
	char* control_type=NULL;

	/*Fem models : */
	FemModel femmodels[5];

	/*Results: */
	DataSet* results=NULL;
	Result*  result=NULL;

	ParameterInputs* inputs=NULL;
	int waitonlock=0;

	/*Intermediary: */
	double* u_g_initial=NULL;
	double* u_g_obs=NULL;
	Param*  param=NULL;
	int      count;

	MODULEBOOT();

	#if !defined(_PARALLEL_) || (defined(_PARALLEL_) && !defined(_HAVE_PETSC_))
	throw ErrorException(__FUNCT__," parallel executable was compiled without support of parallel libraries!");
	#endif

	PetscInitialize(&argc,&argv,(char *)0,"");  

	/*Size and rank: */
	MPI_Comm_rank(MPI_COMM_WORLD,&my_rank);  
	MPI_Comm_size(MPI_COMM_WORLD,&num_procs); 

	_printf_("recover , input file name and output file name:\n");
	inputfilename=argv[2];
	outputfilename=argv[3];
	lockname=argv[4];

	/*Open handle to data on disk: */
	fid=pfopen(inputfilename,"rb");

	_printf_("read and create finite element model:\n");
	_printf_("\n   reading control horiz model data:\n");
	CreateFemModel(&femmodels[0],fid,"control","horiz");
	_printf_("\n   reading control vert model data:\n");
	CreateFemModel(&femmodels[1],fid,"control","vert");
	_printf_("\n   reading control stokes model data:\n");
	CreateFemModel(&femmodels[2],fid,"control","stokes");
	_printf_("\n   reading control hutter model data:\n");
	CreateFemModel(&femmodels[3],fid,"control","hutter");
	_printf_("\n   reading surface and bed slope computation model data:\n");
	CreateFemModel(&femmodels[4],fid,"slope_compute","");

	_printf_("initialize inputs:\n");
	femmodels[0].parameters->FindParam((void*)&u_g_initial,"u_g");
	femmodels[0].parameters->FindParam((void*)&u_g_obs,"u_g_obs");
	femmodels[0].parameters->FindParam((void*)&numberofnodes,"numberofnodes");

	inputs=new ParameterInputs;
	inputs->Add("velocity",u_g_initial,3,numberofnodes);
	inputs->Add("velocity_obs",u_g_obs,2,numberofnodes);

	/*erase velocities: */
	param=(Param*)femmodels[0].parameters->FindParamObject("u_g");
	femmodels[0].parameters->DeleteObject((Object*)param);

	param=(Param*)femmodels[0].parameters->FindParamObject("u_g_obs");
	femmodels[0].parameters->DeleteObject((Object*)param);

	_printf_("initialize results:\n");
	results=new DataSet(ResultsEnum());

	//Add output file name to parameters of femmodels[0]
	count=femmodels[0].parameters->Size()+1;
	param= new Param(count,"outputfilename",STRING);
	param->SetString(outputfilename);
	femmodels[0].parameters->AddObject(param);

	/*are we running the solution sequence, or a qmu wrapper around it? : */
	femmodels[0].parameters->FindParam((void*)&qmu_analysis,"qmu_analysis");
	if(!qmu_analysis){

		/*run control analysis: */
		_printf_("call computational core:\n");
		control_core(results,femmodels,inputs);

	}
	else{
		/*run qmu analysis: */
		_printf_("calling qmu analysis on control core:\n");

#ifdef _HAVE_DAKOTA_ 
		Qmux(&femmodels[0],inputs,DiagnosticAnalysisEnum(),NoneAnalysisEnum());
#else
		throw ErrorException(__FUNCT__," Dakota not present, cannot do qmu!");
#endif
	}

	/*Add analysis_type and control_type to results: */
	result=new Result(results->Size()+1,0,1,"analysis_type","control");
	results->AddObject(result);
	
	femmodels[0].parameters->FindParam((void*)&control_type,"control_type");
	result=new Result(results->Size()+1,0,1,"control_type",control_type);
	results->AddObject(result);

	_printf_("process results:\n");
	ProcessResults(&results,&femmodels[0],ControlAnalysisEnum());

	_printf_("write results to disk:\n");
	OutputResults(results,outputfilename);

	_printf_("write lock file:\n");
	femmodels[0].parameters->FindParam((void*)&waitonlock,"waitonlock");
	if (waitonlock){
		WriteLockFile(lockname);
	}

	_printf_("closing MPI and Petsc\n");
	PetscFinalize(); 

	/*end module: */
	MODULEEND();
	
	return 0; //unix success return;
}
