/*\file KMLOverlay.c
 *\brief: KML file overlay mex module.
 */
#include "./KMLOverlay.h"

void mexFunction( int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[]){

	int i,verbose=1;

	/*input: */
	char*   filkml=NULL;
	char*   filkmz=NULL;

	FILE*   fid=NULL;

	Options* options=NULL;
	int      nlat=0,nlong=0;
	double*  lataxis =NULL;
	double*  longaxis=NULL;
	int      nimages=0;
	char**   pimages=NULL;
	double   dzip=0;
	char*    czip=NULL;

	/* output: */
	int     ierror=0;

	/*Boot module: */
	MODULEBOOT();

	/*checks on arguments on the matlab side: */
	if(nlhs>NLHS){
		KMLOverlayUsage(); _error2_("KMLOverlay usage error");
	}
	if(nrhs<NRHS){
		KMLOverlayUsage(); _error2_("KMLOverlay usage error");
	}

	/*Input datasets: */
	FetchData(&filkml,FILENAME);
	FetchData(&options,NRHS,nrhs,prhs);

	options->Get(&lataxis ,&nlat ,"lataxis" );
	if (verbose && lataxis) for (i=0; i<nlat; i++) _printLine_("  lataxis [" << i << "]=" << lataxis[i]);
	options->Get(&longaxis,&nlong,"longaxis");
	if (verbose && longaxis) for (i=0; i<nlong; i++) _printLine_("  longaxis[" << i << "]=" << longaxis[i]);
	options->Get(&pimages,&nimages,"images");
	if (verbose && pimages) for (i=0; i<nimages; i++) _printLine_("  pimages[" << i << "]=\"" << pimages[i] << "\"");
	options->Get(&dzip,"zip",0.);
	if (verbose) _printLine_("  dzip=" << dzip);

	/*some checks*/
	if (nlat !=2) _error2_("Latitudinal axes \"lataxis\" require two double values, not " << nlat << ".");
	if (nlong!=2) _error2_("Longitudinal axes \"longaxis\" require two double values, not " << nlong << ".");
	if (!nimages) _error2_("No image files provided.");

	if ((int)dzip){
		filkmz=filkml;
		filkml=(char*)mxMalloc(8*sizeof(char));
		strcpy(filkml,"doc.kml");
	}

	if(!strlen(filkml)) strcpy(filkml,"stdout");

	if(verbose) _printLine_("Opening kml overlay file \"" << filkml << "\".");
	fid=fopen(filkml,"w");

	/* Run core computations: */
	if (verbose) _printLine_("Calling core:");
	KMLOverlayx(&ierror,lataxis,longaxis,nimages,pimages,fid);

	if (verbose) _printLine_("Closing file \"" << filkml << "\".");
	fclose(fid);

	/* Create kmz file, if specified: */
	if ((int)dzip) {
		czip=(char*)mxMalloc((5+strlen(filkmz)+1+strlen(filkml)+1)*sizeof(char));
		czip[0]='\0';
		strcat(czip,"!zip ");
		strcat(czip,filkmz);
		strcat(czip," ");
		strcat(czip,filkml);
		for (i=0; i<nimages; i++)
			if (strlen(pimages[i]) && strncmp(pimages[i],"http",4)) {
				czip=(char*)mxRealloc(czip,(strlen(czip)+1+strlen(pimages[i])+1)*sizeof(char));
				strcat(czip," ");
				strcat(czip,pimages[i]);
			}
		if (verbose) _printLine_("Zipping file \"" << filkmz << "\".");
		if (verbose) _printLine_(czip);

		if (mexEvalString(czip)) _error2_("Error zipping file \"" << filkmz << "\".");
		xfree((void**)&czip);
		xfree((void**)&filkmz);
	}

	/*Write data: */
	WriteData(ERRORFLAG,ierror);

	/*Clean-up*/
	if (pimages) {
		for (i=nimages; i>0; i--) xfree((void**)&(pimages[i-1]));
		xfree((void**)&pimages);
	}
	xfree((void**)&longaxis);
	xfree((void**)&lataxis);
	delete options;
	xfree((void**)&filkml);

	/*end module: */
	MODULEEND();
}

void KMLOverlayUsage(void){
	_pprintLine_("KMLOverlay - KML file overlay module:");
	_pprintLine_("");
	_pprintLine_("   This module reads a list of image files and writes a KML or KMZ overlay file.");
	_pprintLine_("");
	_pprintLine_("   Usage:");
	_pprintLine_("      ierror=KMLOverlay(kmlfile,'param name',param,...);");
	_pprintLine_("");
	_pprintLine_("      kmlfile     KML or KMZ file name (string)");
	_pprintLine_("");
	_pprintLine_("      lataxis     latitude axis (double vector [south north], required)");
	_pprintLine_("      longaxis    longitude axis (double vector [west east], required)");
	_pprintLine_("      images      relative or http image file names (string or array of strings or cell array of strings, required)");
	_pprintLine_("      zip         flag to zip the doc.kml and image files into kmzfile (double, non-zero for kmz)");
	_pprintLine_("");
	_pprintLine_("      ierror     error flag (double, non-zero for error)");
	_pprintLine_("");
	_pprintLine_("   Example:");
	_pprintLine_("      KMLOverlay(kmlfile,'lataxis',[south north],'longaxis',[west east],'images',{'file1.png','http://issm/file2.png'},'zip',1);");
	_pprintLine_("");
}
