/*\file Kml2Exp.c
 *\brief: kml to exp file conversion mex module.
 */
#include "./Kml2Exp.h"

void mexFunction( int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[]){

	int i,verbose=1;

	/*input: */
	char    *filkml=NULL,*filexp=NULL;
	int     sgn;
	Options* options=NULL;
	double   cm=0.,sp=0.;

	/* output: */
	int     iret=0;

	/*Boot module: */
	MODULEBOOT();

	/*checks on arguments on the matlab side: */
	if (nlhs > NLHS) {
		Kml2ExpUsage(); _error2_("Kml2Exp usage error");
	}
	if (nrhs < NRHS) {
		Kml2ExpUsage(); _error2_("Kml2Exp usage error");
	}

	/*Input datasets: */
	FetchData(&filkml,KML_IN);
	FetchData(&filexp,EXP_IN);
	FetchData(&sgn,SGN_IN);
	FetchData(&options,NRHS,nrhs,prhs);

	/*  defaults are in Ll2xydef, so don't duplicate them here, and only use user values if both have been specified  */
	if (options->GetOption("central_meridian") || options->GetOption("standard_parallel")) {
		options->Get(&cm,"central_meridian");
		if (verbose) _printLine_("  cm=" << cm);
		options->Get(&sp,"standard_parallel");
		if (verbose) _printLine_("  sp=" << sp);
	}

	/*some checks*/
	if (sgn !=+1 && sgn!= -1) _error2_("Hemisphere sgn=" << sgn << " must be +1 (north) or -1 (south).");
	if (fabs(cm)      > 180.) _error2_("Central meridian cm=" << cm << " must be between -180 (west) and +180 (east) degrees.");
	if (sp < 0. || sp >  90.) _error2_("Standard parallel sp=" << sp << " must be between 0 and 90 degrees (in specified hemisphere).");

	/* Run core computations: */
	if (options->GetOption("central_meridian") && options->GetOption("standard_parallel"))
		iret=Kml2Expx(filkml,filexp,sgn,cm,sp);
	else
		iret=Kml2Expx(filkml,filexp,sgn);

	/*Write data: */
	WriteData(RET_OUT,iret);

	/*Clean-up*/
	delete options;
	xfree((void**)&filexp);
	xfree((void**)&filkml);

	/*end module: */
	MODULEEND();
}

void Kml2ExpUsage(void){
	if(true) _pprintLine_("Kml2Exp - kml to exp file conversion module:");
	if(true) _pprintLine_("");
	if(true) _pprintLine_("   This module converts a file from kml to exp format.");
	if(true) _pprintLine_("");
	if(true) _pprintLine_("   Usage:");
	if(true) _pprintLine_("      [ret]=Kml2Exp(filexp,filkml,sgn,'param name',param,...);");
	if(true) _pprintLine_("");
	if(true) _pprintLine_("      filkml      file name of kml file to be read (char)");
	if(true) _pprintLine_("      filexp      file name of exp file to be written (char)");
	if(true) _pprintLine_("      sgn         sign for hemisphere (double, +1 (north) or -1 (south))");
	if(true) _pprintLine_("");
	if(true) _pprintLine_("      central_meridian     central meridian (double, optional, but must specify with sp)");
	if(true) _pprintLine_("      standard_parallel    standard parallel (double, optional, but must specify with cm)");
	if(true) _pprintLine_("");
	if(true) _pprintLine_("      ret         return code (non-zero for warning)");
	if(true) _pprintLine_("");
	if(true) _pprintLine_("   Examples:");
	if(true) _pprintLine_("      [ret]=Kml2Exp('file.kml','file.exp', 1);");
	if(true) _pprintLine_("      [ret]=Kml2Exp('file.kml','file.exp', 1,'central_meridian',45,'standard_parallel',70);");
	if(true) _pprintLine_("      [ret]=Kml2Exp('file.kml','file.exp',-1,'central_meridian', 0,'standard_parallel',71);");
	if(true) _pprintLine_("");
}

