/*!\file InputUpdateFromDakotax
 * \brief: update datasets using  parameter inputs
 */

#include "./InputUpdateFromDakotax.h"
#include "../../shared/shared.h"
#include "../../include/include.h"
#include "../../toolkits/toolkits.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../modules.h"

void  InputUpdateFromDakotax(Elements* elements,Nodes* nodes,Vertices* vertices,Loads* loads,Materials*  materials,Parameters* parameters,double* variables,char* *variables_descriptors,int numvariables){

	int     i,j,k,l;
	int     dummy;
	
	int     numberofvertices;
	int     nrows;
	int     ncols;
	int     npart;
	double *qmu_part  = NULL;

	double *distributed_values = NULL;
	double *parameter          = NULL;
	char   *descriptor         = NULL;
	char    root[50]; //root name of variable, ex: DragCoefficent, RhoIce, etc ...

	/*retrieve parameters: */
	parameters->FindParam(&npart,QmuNumberofpartitionsEnum);
	parameters->FindParam(&qmu_part,&dummy,QmuPartitionEnum);
	numberofvertices=vertices->NumberOfVertices();

	/*Go through all dakota descriptors, ex: "rho_ice","thermal_conductivity","thickness1","thickness2", etc ..., and 
	 * for each descriptor, take the variable value and plug it into the inputs: */

	for(i=0;i<numvariables;i++){
	
		descriptor=variables_descriptors[i];

		/*From descriptor, figure out if the variable is scaled, indexed, nodal, or just a simple variable: */
		if (strncmp(descriptor,"scaled_",7)==0){
			
			/*Variable is scaled. Determine root name of variable (ex: scaled_DragCoefficient_1 -> DragCoefficient). Allocate distributed_values and fill the 
			 * distributed_values with the next npart variables: */
			
			//strcpy(root,strstr(descriptor,"_")+1); *strstr(root,"_")='\0';
			memcpy(root,strstr(descriptor,"_")+1,(strlen(strstr(descriptor,"_")+1)+1)*sizeof(char));
			*strstr(root,"_")='\0';


			distributed_values=xNew<double>(npart);
			for(j=0;j<npart;j++){
				distributed_values[j]=variables[i+j];
			}

			/*Now, pick up the parameter corresponding to root: */
			parameters->FindParam(&parameter,&nrows,&ncols,StringToEnumx(root));

			/*We've got the parameter, we need to update it using qmu_part (a partitioning vector), 
			 * and the distributed_values. Two cases: we either have a nrows=numberofvertices, in 
			 * which case our parameter is a vector, or nrows=numberofvertices+1, in which case, 
			 * our parameter is a transient vector. Deal with both cases accordingly: */
			for(k=0;k<numberofvertices;k++){
				for(l=0;l<ncols;l++){
					*(parameter+ncols*k+l)=*(parameter+ncols*k+l)*distributed_values[(int)qmu_part[k]];
				}
			}

			#ifdef _DEBUG_
				PetscSynchronizedPrintf(MPI_COMM_WORLD,"Parameter matrix:");
				PetscSynchronizedFlush(MPI_COMM_WORLD);
				for(l=0;l<ncols;l++){
					PetscSynchronizedPrintf(MPI_COMM_WORLD," time %i\n",l);
					PetscSynchronizedFlush(MPI_COMM_WORLD);

					for(k=0;k<numberofvertices;k++){
						PetscSynchronizedPrintf(MPI_COMM_WORLD," node %i value %g\n",k+1,*(parameter+k*ncols+l));
						PetscSynchronizedFlush(MPI_COMM_WORLD);
					}
				}
				PetscSynchronizedPrintf(MPI_COMM_WORLD," descriptor: %s root %s enum: %i\n",descriptor,root,StringToEnumx(root));
				PetscSynchronizedFlush(MPI_COMM_WORLD);
			#endif
			  

			/*Update inputs using the parameter matrix: */
			InputUpdateFromMatrixDakotax( elements,nodes, vertices,loads, materials,  parameters, parameter, nrows,ncols,StringToEnumx(root), VertexEnum);

			/*increment i to skip the distributed values just collected: */
			i+=npart-1; //careful, the for loop will add 1.

			/*Free allocations: */
			xDelete<double>(parameter);
			xDelete<double>(distributed_values);
		}
		else if (strncmp(descriptor,"indexed_",8)==0){
			_error2_("indexed variables not supported yet!");
		}
		else if (strncmp(descriptor,"nodal_",8)==0){
			_error2_("nodal variables not supported yet!");
		}
		else{
			/*Ok, standard variable, just update inputs using the variable: */
			InputUpdateFromConstantx( elements,nodes, vertices,loads, materials,  parameters, variables[i],StringToEnumx(descriptor));
		}
	}

	/*Free ressources:*/
	xDelete<double>(qmu_part);
}
