/*!\file:  OptimalSearch.cpp
 * \brief optimization algorithm
 */ 

#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "./numerics.h"
#include "../../objects/objects.h"
#include "../../io/io.h"
#include "../../include/include.h"
#include "../../shared/shared.h"
#include <float.h>

void OptimalSearch(IssmDouble* psearch_scalar,IssmDouble* pJ,OptPars* optpars,IssmDouble (*f)(IssmDouble,OptArgs*), OptArgs* optargs){

	/* This routine is optimizing a given function*/

	/*function values: */
	IssmDouble fx1,fx2,fxbest;
	IssmDouble x1,x2,xmin,xbest;

	/*tolerances: */
	IssmDouble seps;
	IssmDouble tolerance=1.e-4;
	int    maxiter;

	/*counters: */
	int  iter=0;
	bool loop;

	/*Recover parameters:*/
	xmin     =optpars->xmin;
	x1       =optpars->xmin;
	x2       =optpars->xmax;
	maxiter  =optpars->maxiter;
	
	//get the value of the function at the first boundary
	fx1= (*f)(x1,optargs);
	if (isnan(fx1)) _error2_("Function evaluation returned NaN");
	_printf_(VerboseControl(),"\n        Iteration         x           f(x)       Tolerance\n\n");
	_printf_(VerboseControl(),"        %s    %12.6g  %12.6g  %s","   N/A",x1,fx1,"         N/A\n");

	//update tolerances
	seps=sqrt(DBL_EPSILON);

	loop=true;
	while(loop){

		/*get f(x2)*/
		iter++;
		fx2 = (*f)(x2,optargs);
		if (isnan(fx2)) _error2_("Function evaluation returned NaN");
		_printf_(VerboseControl(),"         %5i    %12.6g  %12.6g  %12.6g\n",iter,x2,fx2,fabs(x2-x1)>fabs(fx2-fx1)?fabs(fx2-fx1):fabs(x2-x1));

		//Stop the optimization?
		if ((fabs(x2-x1)+seps)<tolerance || (fabs(fx2-fx1)+seps)<tolerance){
			_printf_(VerboseControl(),"      %s%g\n","optimization terminated: the current x satisfies the termination criteria using 'tolx' of " ,tolerance);
			loop=false;
		}
		else if (iter>=maxiter){
			_printf_(VerboseControl(),"      %s\n","exiting: Maximum number of iterations has been exceeded  - increase 'maxiter'\n");
			loop=false;
		}
		else{
			//continue
			loop=true;
		}

		/*Compute x2*/
		if(fx2<fx1){
			xbest=x2; fxbest=fx2;
			x1=x2;
			x2=xmin+1.1*(x2-xmin); 
			fx1=fx2;
		}
		else{
			xbest=x1; fxbest=fx1;
			x2=xmin+0.5*(x2-xmin);
		}
	}//end while

	/*Assign output pointers: */
	*psearch_scalar=xbest;
	*pJ=fxbest;
}
