/*!\file:  Kriging.cpp
 * \brief  "c" core code for Kriging
 */ 

#include "./Krigingx.h"
#include "../../shared/shared.h"
#include "../../include/include.h"
#include "../../toolkits/toolkits.h"
#include "../../objects/objects.h"
#include "../../Container/Container.h"
#include "../modules.h"

#ifdef _HAVE_GSL_
#include <gsl/gsl_linalg.h>
#endif

/*FUNCTION pKrigingx{{{*/
int pKrigingx(double** ppredictions,double **perror,double* obs_x, double* obs_y, double* obs_list, int obs_length,double* x_interp,double* y_interp,int n_interp,Options* options){

	extern int num_procs;
	extern int my_rank;

	/*output*/
	double *predictions = NULL;
	double *error       = NULL;

	/*Intermediaries*/
	int           mindata,maxdata;
	double        radius;
	char         *output       = NULL;
	Variogram    *variogram    = NULL;
	Observations *observations = NULL;

	/*Get Variogram from Options*/
	ProcessVariogram(&variogram,options);
	options->Get(&radius,"searchradius",0.);
	options->Get(&mindata,"mindata",1);
	options->Get(&maxdata,"maxdata",50);

	/*Process observation dataset*/
	observations=new Observations(obs_list,obs_x,obs_y,obs_length,options);

	/*Allocate output*/
	predictions =(double*)xcalloc(n_interp,sizeof(double));
	error       =(double*)xcalloc(n_interp,sizeof(double));

	/*Get output*/
	options->Get(&output,"output","prediction");

	if(strcmp(output,"quadtree")==0){
		observations->QuadtreeColoring(predictions,x_interp,y_interp,n_interp);
	}
	else if(strcmp(output,"variomap")==0){
		observations->Variomap(predictions,x_interp,n_interp);
	}
	else if(strcmp(output,"prediction")==0){

		/*partition loop across threads: */
		for(int idx=my_rank;idx<n_interp;idx+=num_procs){
			_printf_(true,"\r      interpolation progress: %5.2lf%%",double(idx)/double(n_interp)*100);
			observations->InterpolationKriging(&predictions[idx],&error[idx],x_interp[idx],y_interp[idx],radius,mindata,maxdata,variogram);
		}
		_printf_(true,"\r      interpolation progress: %5.2lf%%\n",100.);

#ifdef _HAVE_MPI_
		double *sumpredictions =(double*)xmalloc(n_interp*sizeof(double));
		double *sumerror       =(double*)xmalloc(n_interp*sizeof(double));
		MPI_Allreduce(predictions,sumpredictions,n_interp,MPI_DOUBLE,MPI_SUM,MPI_COMM_WORLD);
		MPI_Allreduce(error,sumerror,n_interp,MPI_DOUBLE,MPI_SUM,MPI_COMM_WORLD);
		xfree((void**)&error); error=sumerror;
		xfree((void**)&predictions); predictions=sumpredictions;
#endif
	}
	else{
		_error_("output '%s' not supported yet",output);
	}
	_printf_(true,"\r      interpolation progress: %5.2lf%%\n",100.);

	/*clean-up and Assign output pointer*/
	delete variogram;
	delete observations;
	xfree((void**)&output);
	*ppredictions = predictions;
	*perror       = error;
	return 1;
}/*}}}*/
void ProcessVariogram(Variogram **pvariogram,Options* options){/*{{{*/

	/*Intermediaries*/
	Variogram* variogram = NULL;
	char      *model     = NULL;

	if(options->GetOption("model")){
		options->Get(&model,"model");
		if     (strcmp(model,"gaussian")==0)    variogram = new GaussianVariogram(options);
		else if(strcmp(model,"exponential")==0) variogram = new ExponentialVariogram(options);
		else if(strcmp(model,"spherical")==0)   variogram = new SphericalVariogram(options);
		else if(strcmp(model,"power")==0)       variogram = new PowerVariogram(options);
		else _error_("variogram %s not supported yet (list of supported variogram: gaussian, exponential, spherical and power)",model);
	}
	else variogram = new GaussianVariogram(options);

	/*Assign output pointer*/
	xfree((void**)&model);
	*pvariogram = variogram;
}/*}}}*/
void GslSolve(double** pX,double* A,double* B,int n){/*{{{*/
#ifdef _HAVE_GSL_

		/*GSL Matrices and vectors: */
		int              s;
		gsl_matrix_view  a;
		gsl_vector_view  b;
		gsl_vector      *x = NULL;
		gsl_permutation *p = NULL;

		/*A will be modified by LU decomposition. Use copy*/
		double* Acopy = (double*)xmalloc(n*n*sizeof(double));
		memcpy(Acopy,A,n*n*sizeof(double));

		/*Initialize gsl matrices and vectors: */
		a = gsl_matrix_view_array (Acopy,n,n);
		b = gsl_vector_view_array (B,n);
		x = gsl_vector_alloc (n);

		/*Run LU and solve: */
		p = gsl_permutation_alloc (n);
		gsl_linalg_LU_decomp (&a.matrix, p, &s);
		gsl_linalg_LU_solve (&a.matrix, p, &b.vector, x);

		//printf ("x = \n");
		//gsl_vector_fprintf (stdout, x, "%g");

		/*Copy result*/
		double* X = (double*)xmalloc(n*sizeof(double));
		memcpy(X,gsl_vector_ptr(x,0),n*sizeof(double));

		/*Clean up and assign output pointer*/
		xfree((void**)&Acopy);
		gsl_permutation_free(p);
		gsl_vector_free(x);
		*pX=X;
#else
		_error_("GSL support required");
#endif
	}/*}}}*/
