/*!\file:  Kriging.cpp
 * \brief  "c" core code for Kriging
 */ 

#include "./Krigingx.h"
#include "../../shared/shared.h"
#include "../../include/include.h"
#include "../../toolkits/toolkits.h"
#include "../../objects/objects.h"
#include "../../Container/Observations.h"
#include "../modules.h"
#ifdef _HAVE_GSL_
#include <gsl/gsl_linalg.h>
#endif
/*FUNCTION Krigingx{{{*/
int Krigingx(double** ppredictions,double **perror,double* obs_x, double* obs_y, double* obs_list, int obs_length,double* x_interp,double* y_interp,int n_interp,Options* options){

	/*output*/
	double *predictions = NULL;
	double *error       = NULL;

	/*Intermediaries*/
	int           mindata,maxdata;
	double        radius;
	char         *output       = NULL;
	Variogram    *variogram    = NULL;
	Observations *observations = NULL;

	/*threading: */
	KrigingxThreadStruct gate;
	int num=1;
#ifdef _MULTITHREADING_
	num=_NUMTHREADS_;
#endif

	/*Get Variogram from Options*/
	ProcessVariogram(&variogram,options);
	options->Get(&radius,"searchradius",0.);
	options->Get(&mindata,"mindata",1);
	options->Get(&maxdata,"maxdata",50);

	/*Process observation dataset*/
	observations=new Observations(obs_list,obs_x,obs_y,obs_length,options);

	/*Allocate output*/
	predictions =(double*)xcalloc(n_interp,sizeof(double));
	error       =(double*)xcalloc(n_interp,sizeof(double));

	/*Get output*/
	options->Get(&output,"output","prediction");

	if(strcmp(output,"quadtree")==0){
		observations->QuadtreeColoring(predictions,x_interp,y_interp,n_interp);
	}
	else if(strcmp(output,"variomap")==0){
		observations->Variomap(predictions,x_interp,n_interp);
	}
	else if(strcmp(output,"prediction")==0){

		/*initialize thread parameters: */
		gate.n_interp     = n_interp;
		gate.x_interp     = x_interp;
		gate.y_interp     = y_interp;
		gate.radius       = radius;
		gate.mindata      = mindata;
		gate.maxdata      = maxdata;
		gate.variogram    = variogram;
		gate.observations = observations;
		gate.predictions  = predictions;
		gate.error        = error;
		gate.percent      = (double*)xcalloc(num,sizeof(double));

		/*launch the thread manager with Krigingxt as a core: */
		LaunchThread(Krigingxt,(void*)&gate,num);
		printf("\r      interpolation progress:  100.00%%\n");
		xfree((void**)&gate.percent);
	}
	else{
		_error_("output '%s' not supported yet",output);
	}

	/*clean-up and Assign output pointer*/
	delete variogram;
	delete observations;
	xfree((void**)&output);
	*ppredictions = predictions;
	*perror       = error;
	return 1;
}/*}}}*/
/*FUNCTION Krigingxt{{{*/
void* Krigingxt(void* vpthread_handle){

	/*gate variables :*/
	KrigingxThreadStruct *gate        = NULL;
	pthread_handle       *handle      = NULL;
	int my_thread;
	int num_threads;
	int i0,i1;

	/*recover handle and gate: */
	handle      = (pthread_handle*)vpthread_handle;
	gate        = (KrigingxThreadStruct*)handle->gate;
	my_thread   = handle->id;
	num_threads = handle->num;

	/*recover parameters :*/
	int           n_interp     = gate->n_interp;
	double       *x_interp     = gate->x_interp;
	double       *y_interp     = gate->y_interp;
	double        radius       = gate->radius;
	int           mindata      = gate->mindata;
	int           maxdata      = gate->maxdata;
	Variogram    *variogram    = gate->variogram;
	Observations *observations = gate->observations;
	double       *predictions  = gate->predictions;
	double       *error        = gate->error;
	double       *percent      = gate->percent;

	/*Intermediaries*/
	int           i,j,n_obs;
	double        numerator,denominator,ratio,localpercent;
	double       *x            = NULL;
	double       *y            = NULL;
	double       *obs          = NULL;
	double       *Gamma        = NULL;
	double       *GinvG0       = NULL;
	double       *Ginv1        = NULL;
	double       *GinvZ        = NULL;
	double       *gamma0       = NULL;
	double       *ones         = NULL;

	/*partition loop across threads: */
	PartitionRange(&i0,&i1,n_interp,num_threads,my_thread);
	for(int idx=i0;idx<i1;idx++){

		/*Print info*/
		percent[my_thread]=double(idx-i0)/double(i1-i0)*100.;
		localpercent=percent[0];
		for(i=1;i<num_threads;i++) localpercent=min(localpercent,percent[i]);
		printf("\r      interpolation progress: %5.2lf%%",localpercent);

		/*Get list of observations for current point*/
		observations->ObservationList(&x,&y,&obs,&n_obs,x_interp[idx],y_interp[idx],radius,maxdata);
		if(n_obs<mindata){
			predictions[idx] = -999.0; 
			error[idx]       = -999.0; 
			continue;
		}

		/*Allocate intermediary matrix and vectors*/

		Gamma  = (double*)xmalloc(n_obs*n_obs*sizeof(double));
		gamma0 = (double*)xmalloc(n_obs*sizeof(double));
		ones   = (double*)xmalloc(n_obs*sizeof(double));

		/*First: Create semivariogram matrix for observations*/
		for(i=0;i<n_obs;i++){
			for(j=0;j<=i;j++){
				//Gamma[i*n_obs+j] = variogram->SemiVariogram(x[i]-x[j],y[i]-y[j]);
				Gamma[i*n_obs+j] = variogram->Covariance(x[i]-x[j],y[i]-y[j]);
				Gamma[j*n_obs+i] = Gamma[i*n_obs+j];
			}
		}
		for(i=0;i<n_obs;i++) ones[i]=1;

		/*Get semivariogram vector associated to this location*/
		//for(i=0;i<n_obs;i++) gamma0[i] = variogram->SemiVariogram(x[i]-x_interp[idx],y[i]-y_interp[idx]);
		for(i=0;i<n_obs;i++) gamma0[i] = variogram->Covariance(x[i]-x_interp[idx],y[i]-y_interp[idx]);

		/*Solve the three linear systems*/
		GslSolve(&GinvG0,Gamma,gamma0,n_obs); // Gamma^-1 gamma0
		GslSolve(&Ginv1, Gamma,ones,n_obs);   // Gamma^-1 ones
		GslSolve(&GinvZ, Gamma,obs,n_obs);    // Gamma^-1 Z

		/*Prepare predictor*/
		numerator=-1.; denominator=0.;
		for(i=0;i<n_obs;i++) numerator  +=GinvG0[i];
		for(i=0;i<n_obs;i++) denominator+=Ginv1[i];
		ratio=numerator/denominator;

		predictions[idx] = 0.;
		error[idx]       = - numerator*numerator/denominator;
		for(i=0;i<n_obs;i++) predictions[idx] += (gamma0[i]-ratio)*GinvZ[i];
		for(i=0;i<n_obs;i++) error[idx] += gamma0[i]*GinvG0[i];

		/*clean-up*/
		xfree((void**)&x);
		xfree((void**)&y);
		xfree((void**)&obs);
		xfree((void**)&Gamma);
		xfree((void**)&gamma0);
		xfree((void**)&ones);
		xfree((void**)&GinvG0);
		xfree((void**)&Ginv1);
		xfree((void**)&GinvZ);
	}

	return NULL;
}/*}}}*/

void ProcessVariogram(Variogram **pvariogram,Options* options){/*{{{*/

	/*Intermediaries*/
	Variogram* variogram = NULL;
	char      *model     = NULL;

	if(options->GetOption("model")){
		options->Get(&model,"model");
		if     (strcmp(model,"gaussian")==0)    variogram = new GaussianVariogram(options);
		else if(strcmp(model,"exponential")==0) variogram = new ExponentialVariogram(options);
		else if(strcmp(model,"spherical")==0)   variogram = new SphericalVariogram(options);
		else if(strcmp(model,"power")==0)       variogram = new PowerVariogram(options);
		else _error_("variogram %s not supported yet (list of supported variogram: gaussian, exponential, spherical and power)",model);
	}
	else variogram = new GaussianVariogram(options);

	/*Assign output pointer*/
	xfree((void**)&model);
	*pvariogram = variogram;
}/*}}}*/
void GslSolve(double** pX,double* A,double* B,int n){/*{{{*/
#ifdef _HAVE_GSL_

		/*GSL Matrices and vectors: */
		int              s;
		gsl_matrix_view  a;
		gsl_vector_view  b;
		gsl_vector      *x = NULL;
		gsl_permutation *p = NULL;

		/*A will be modified by LU decomposition. Use copy*/
		double* Acopy = (double*)xmalloc(n*n*sizeof(double));
		memcpy(Acopy,A,n*n*sizeof(double));

		/*Initialize gsl matrices and vectors: */
		a = gsl_matrix_view_array (Acopy,n,n);
		b = gsl_vector_view_array (B,n);
		x = gsl_vector_alloc (n);

		/*Run LU and solve: */
		p = gsl_permutation_alloc (n);
		gsl_linalg_LU_decomp (&a.matrix, p, &s);
		gsl_linalg_LU_solve (&a.matrix, p, &b.vector, x);

		//printf ("x = \n");
		//gsl_vector_fprintf (stdout, x, "%g");

		/*Copy result*/
		double* X = (double*)xmalloc(n*sizeof(double));
		memcpy(X,gsl_vector_ptr(x,0),n*sizeof(double));

		/*Clean up and assign output pointer*/
		xfree((void**)&Acopy);
		gsl_permutation_free(p);
		gsl_vector_free(x);
		*pX=X;
#else
		_error_("GSL support required");
#endif
	}/*}}}*/
