%MODELLIST class definition
%
%   Usage:
%      modellist=modellist({md1 md2 md3});

classdef modellist
	properties (SetAccess=public) 
		models  = cell(0,1);
		cluster = generic();
	end
	methods
		function obj = modellist(varargin) % {{{

			%initialize list
			if nargin==0,
				%Do nothing,
			elseif nargin==1,
				if ~isa(varargin{1},'cell'),
					error('not supported yet');
				end

				celllist=varargin{1};

				%check on size of cell list: 
				if (size(celllist,2)~=1),
					error('modellist constructor error message: list of models should be a cell list of column size 1');
				end

				%check that only models are in the celllist: 
				for i=1:size(celllist,1),
					if ~isa(celllist{i},'model')
						error(['modellist constructor error message: element ' num2str(i) ' of cell list is not a model!']);
					end
				end

				obj.models  = celllist;
				obj.cluster = obj.models{1}.cluster;
			end
		end % }}}
		function val = get(obj, propName)% {{{
		%GET - gets model propertie from a specified object ans returns the value
		% 
		%   Usage:
		%      val = get(a, propName)

			switch propName
				case 'numberofelements'
					val = obj.numberofelements;
				case 'numberofnodes'
					val = obj.numberofnodes;
				case 'elements' 
					val = obj.elements;
				case 'x' 
					val = obj.x;
				case 'y' 
					val = obj.y;
				case 'z' 
					val = obj.z;
				otherwise
					error(['get error message: ' propName,' is not a valid model property'])
			end
		end % }}}
		function obj = loadmultipleresultsfromcluster(obj) % {{{
			%LOADMULTIPLERESULTSFROMCLUSTER - load multiple results of solution sequences from cluster
			%
			%   Usage:
			%      obj=loadresultsfromcluster(obj);

			nummodels=length(obj.models);

			%Get cluster settings
			cluster=obj.cluster;
			name=obj.name;
			cluster_rc_location=which('cluster.rc');
			[codepath,executionpath]=ClusterParameters(cluster,cluster_rc_location);

			%Remote tar: 
			disp('tarring results');
			issmssh(cluster,['"cd ' executionpath '/' name ' && rm -rf file_list.txt ModelResults.tar.gz && find -iname ''*-*vs*.outbin'' > file_list.txt && tar zcvf ModelResults.tar.gz --files-from file_list.txt  && rm -rf file_list.txt "']);

			%copy results from cluster to present directory
			scpin(cluster, [executionpath '/' name], {'ModelResults.tar.gz'});

			%untar:
			!tar -zxvf ModelResults.tar.gz

			%ok, go through list and load results from disk: 
			for i=1:nummodels,
				%load  results for this model
				obj.models{i}=loadresultsfromdisk(obj.models{i},[name '-' num2str(i) 'vs' num2str(nummodels) '.outbin']);

				delete([name '-' num2str(i) 'vs' num2str(nummodels) '.outbin']);
			end

			%erase files 
			delete('ModelResults.tar.gz');
		end % }}}
		function obj = solve(obj,varargin)% {{{
			%SOLVE - apply solution sequence for  a list of models. Used in batch mode.
			%
			%   Usage:
			%      obj=solve(obj,varargin)
			%      where varargin is a lit of paired arguments. 
			%      arguments can be: 'analysis_type': 'diagnostic','thermal','prognostic','transient'
			%
			%   Examples:
			%      obj=solve(obj,'analysis_type','diagnostic');

			%recover options
			options=pairoptions(varargin{:});

			%add default options
			options=process_solve_options(options);

			%length of list
			nummodels=length(obj.models);

			%name of queue: to make it unique, add a time stamp
			name=[obj.name '-' datestr(now,1) '-' datestr(now,'HH-MM-SS') ];

			%name of cluster will be first name of list
			cluster=obj.cluster;

			%Figure out parameters for this particular cluster
			cluster_rc_location=which('cluster.rc');
			[codepath,executionpath]=ClusterParameters(cluster,cluster_rc_location);

			%solve in batch mode: 
			for i=1:nummodels,

				%model
				mdex=obj.models{i};

				%recover some fields
				mdex.analysis_type=options.analysis_type;

				mdex.name=[name '-' num2str(i) 'vs' num2str(nummodels)];
				mdex.time=obj.time;
				mdex.queue=obj.queue;
				mdex.cluster=obj.cluster;
				if ~isnan(obj.np),
					mdex.np=obj.np;
				end

				%call solve in batch mode:
				if strcmpi(cluster,oshostname),
					mdex=solve(mdex,varargin{:});
				else
					mdex=solve(mdex,varargin{:},'batch','yes','directory',name);
				end

				%feed back
				obj.models{i}=mdex;
			end

			%locally, we are done.
			if strcmpi(cluster,oshostname),
				return
			end


			%now, tar all the files and then erase them.
			eval(['!find -iname ''' name '-*'' > file_list.txt']);
			!tar zcvf ModelList.tar.gz --files-from file_list.txt
			!rm -rf *.bin *.queue file_list.txt

			%still have to build a launching script.
			BuildMultipleQueueingScript(cluster,name,executionpath,codepath);

			%launch jobs on remote cluster
			LaunchMultipleQueueJob(cluster,name,executionpath);

			%erase files: 
			delete([name '.queue']);
			delete('ModelList.tar.gz');

			%save name: 
			obj.name=name;
		end % }}}
	end
end
