/*
 * \file Observations.c
 * \brief: implementation of the Observations class, derived from DataSet class
 */

/*Headers: {{{*/
#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <vector>
#include <functional>
#include <algorithm>
#include <iostream>

#include "./DataSet.h"
#include "./Observations.h"
#include "../shared/shared.h"
#include "../include/include.h"
#include "../EnumDefinitions/EnumDefinitions.h"

using namespace std;
/*}}}*/

/*Object constructors and destructor*/
/*FUNCTION Observations::Observations(){{{*/
Observations::Observations(){
	this->quadtree = NULL;
	return;
}
/*}}}*/
/*FUNCTION Observations::Observations(double* observations_list,double* x,double* y,int n,Options* options){{{*/
Observations::Observations(double* observations_list,double* x,double* y,int n,Options* options){

	/*Intermediaries*/
	int          i,maxdepth,level,counter;
	int          xi,yi;
	double       xmin,xmax,ymin,ymax;
	double       offset,minlength;
	Observation *observation = NULL;

	/*Get extrema*/
	xmin=x[0]; ymin=y[0];
	xmax=x[0]; ymax=y[0];
	for(i=1;i<n;i++){
		xmin=min(xmin,x[i]); ymin=min(ymin,y[i]);
		xmax=max(xmax,x[i]); ymax=max(ymax,y[i]);
	}
	offset=0.05*(xmax-xmin); xmin-=offset; xmax+=offset;
	offset=0.05*(ymax-ymin); ymin-=offset; ymax+=offset;

	/*Get Minimum box size*/
	if(options->GetOption("boxlength")){
		options->Get(&minlength,"boxlength");
		if(minlength<=0)_error_("boxlength should be a positive number");
		maxdepth=int(log(max(xmax-xmin,ymax-ymin)/minlength +1)/log(2.0));
	}
	else{
		maxdepth = 30;
		minlength=max(xmax-xmin,ymax-ymin)/double((1L<<maxdepth)-1);
	}

	/*Initialize Quadtree*/
	printf("Generating quadtree with a maximum box size %g (depth=%i)... ",minlength,maxdepth);
	this->quadtree = new Quadtree(xmin,xmax,ymin,ymax,maxdepth);

	/*Add observations one by one*/
	counter = 0;
	for(i=0;i<n;i++){
		this->quadtree->IntergerCoordinates(&xi,&yi,x[i],y[i]);
		this->quadtree->QuadtreeDepth2(&level,xi,yi);
		if((int)level <= maxdepth){
			observation = new Observation(x[i],y[i],xi,yi,counter++,observations_list[i]);
			this->quadtree->Add(observation);
			this->AddObject(observation);
		}
		else{
			/*We need to average with the current observations*/
			this->quadtree->AddAndAverage(x[i],y[i],observations_list[i]);
		}
	}
	printf("done\n");
}
/*}}}*/
/*FUNCTION Observations::~Observations(){{{*/
Observations::~Observations(){
	delete quadtree;
	return;
}
/*}}}*/

/*Methods*/
/*FUNCTION Observations::ObservationList{{{*/
void Observations::ObservationList(double **px,double **py,double **pobs,int* pnobs,double x_interp,double y_interp,double radius,int maxdata){

	/*Output and Intermediaries*/
	bool         stop;
	int          nobs,i,j,k,n,counter;
	double       h2,radius2;
	int         *indices      = NULL;
	double      *dists        = NULL;
	double      *x            = NULL;
	double      *y            = NULL;
	double      *obs          = NULL;
	Observation *observation  = NULL;

	/*Compute radius square*/
	if(radius==0) radius=this->quadtree->root->length;
	radius2 = radius*radius;

	/*Find all observations that are in radius*/
	indices = (int*)xmalloc(this->Size()*sizeof(int));
	dists   = (double*)xmalloc(this->Size()*sizeof(double));
	nobs    = 0;

	for (i=0;i<this->Size();i++){
		observation=(Observation*)this->GetObjectByOffset(i);
		h2 = (observation->x-x_interp)*(observation->x-x_interp) + (observation->y-y_interp)*(observation->y-y_interp);

		if(nobs==maxdata && h2>radius2) continue;
		if(nobs<=maxdata){
			indices[nobs]   = i;
			dists[nobs]     = h2;
			nobs++;
		}
		if(nobs==1) continue;

		/*Sort all dists up to now*/
		n=nobs-1;
		stop = false;
		for(k=0;k<n-1;k++){
			if(h2<dists[k]){
				counter=1;
				for(int jj=k;jj<n;jj++){
					j  = n-counter;
					dists[j+1]   = dists[j];
					indices[j+1] = indices[j];
					counter++;
				}
				dists[k]   = h2;
				indices[k] = i;
				stop = true;
				break;
			}
			if(stop) break;
		}
	}  
	xfree((void**)&dists);

	if(nobs){
		/*Allocate vectors*/
		x   = (double*)xmalloc(nobs*sizeof(double));
		y   = (double*)xmalloc(nobs*sizeof(double));
		obs = (double*)xmalloc(nobs*sizeof(double));

		/*Loop over all observations and fill in x, y and obs*/
		for (i=0;i<nobs;i++){
			observation=(Observation*)this->GetObjectByOffset(indices[i]);
			observation->WriteXYObs(&x[i],&y[i],&obs[i]);
		}
	}

	/*Assign output pointer*/
	xfree((void**)&indices);
	*px=x;
	*py=y;
	*pobs=obs;
	*pnobs=nobs;
}/*}}}*/
/*FUNCTION Observations::QuadtreeColoring{{{*/
void Observations::QuadtreeColoring(double* A,double *x,double *y,int n){

	int xi,yi,level;

	for(int i=0;i<n;i++){
		this->quadtree->IntergerCoordinates(&xi,&yi,x[i],y[i]);
		this->quadtree->QuadtreeDepth(&level,xi,yi);
		A[i]=(double)level;
	}

}/*}}}*/
/*FUNCTION Observations::Variomap{{{*/
void Observations::Variomap(double* gamma,double *x,int n){

	/*Output and Intermediaries*/
	int          i,j,k;
	double       distance;
	Observation *observation1 = NULL;
	Observation *observation2 = NULL;

	int *counter= (int*)xmalloc(n*sizeof(int));
	for(j=0;j<n;j++) counter[j] = 0;
	for(j=0;j<n;j++) gamma[j]   = 0.0;

	for(i=0;i<this->Size();i++){
		observation1=(Observation*)this->GetObjectByOffset(i);

		for(j=i+1;j<this->Size();j++){
			observation2=(Observation*)this->GetObjectByOffset(j);

			distance=sqrt(pow(observation1->x - observation2->x,2.) + pow(observation1->y - observation2->y,2.));
			if(distance>x[n-1]) continue;

			int index = int(distance/(x[1]-x[0]));
			if(index>n-1) index = n-1;
			if(index<0)   index = 0;

			gamma[index]   += 1./2.*pow(observation1->value - observation2->value,2.);
			counter[index] += 1;
		}
	}

	/*Normalize semivariogram*/
	gamma[0]=0;
	for(k=0;k<n;k++){
		if(counter[k]) gamma[k] = gamma[k]/double(counter[k]);
	}

	/*Assign output pointer*/
	xfree((void**)&counter);
}/*}}}*/
