/*
 * \file Observations.c
 * \brief: implementation of the Observations class, derived from DataSet class
 */

/*Headers: {{{*/
#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <vector>
#include <functional>
#include <algorithm>
#include <iostream>

#include "./DataSet.h"
#include "./Observations.h"
#include "../shared/shared.h"
#include "../include/include.h"
#include "../EnumDefinitions/EnumDefinitions.h"

using namespace std;
/*}}}*/

/*Object constructors and destructor*/
/*FUNCTION Observations::Observations(){{{*/
Observations::Observations(){
	this->quadtree = NULL;
	return;
}
/*}}}*/
/*FUNCTION Observations::Observations(double* observations_list,double* x,double* y,int n,Options* options){{{*/
Observations::Observations(double* observations_list,double* x,double* y,int n,Options* options){

	/*Intermediaries*/
	int          i,maxdepth,level,counter;
	int          xi,yi;
	double       xmin,xmax,ymin,ymax;
	double       offset,minlength;
	Observation *observation = NULL;

	/*Get extrema*/
	xmin=x[0]; ymin=y[0];
	xmax=x[0]; ymax=y[0];
	for(i=1;i<n;i++){
		xmin=min(xmin,x[i]); ymin=min(ymin,y[i]);
		xmax=max(xmax,x[i]); ymax=max(ymax,y[i]);
	}
	offset=0.05*(xmax-xmin); xmin-=offset; xmax+=offset;
	offset=0.05*(ymax-ymin); ymin-=offset; ymax+=offset;

	/*Get Minimum box size*/
	if(options->GetOption("boxlength")){
		options->Get(&minlength,"boxlength");
		if(minlength<=0)_error_("boxlength should be a positive number");
		maxdepth=int(log(max(xmax-xmin,ymax-ymin)/minlength +1)/log(2.0));
	}
	else{
		maxdepth = 30;
		minlength=max(xmax-xmin,ymax-ymin)/double((1L<<maxdepth)-1);
	}

	/*Initialize Quadtree*/
	printf("Generating quadtree with a maximum box size %g (depth=%i)... ",minlength,maxdepth);
	this->quadtree = new Quadtree(xmin,xmax,ymin,ymax,maxdepth);

	/*Add observations one by one*/
	counter = 0;
	for(i=0;i<n;i++){
		this->quadtree->IntergerCoordinates(&xi,&yi,x[i],y[i]);
		this->quadtree->QuadtreeDepth2(&level,xi,yi);
		if((int)level <= maxdepth){
			observation = new Observation(x[i],y[i],xi,yi,counter++,observations_list[i]);
			this->quadtree->Add(observation);
			this->AddObject(observation);
		}
		else{
			/*We need to average with the current observations*/
			this->quadtree->AddAndAverage(x[i],y[i],observations_list[i]);
		}
	}
	printf("done\n");
}
/*}}}*/
/*FUNCTION Observations::~Observations(){{{*/
Observations::~Observations(){
	delete quadtree;
	return;
}
/*}}}*/

/*Methods*/
/*FUNCTION Observations::ObservationList{{{*/
void Observations::ObservationList(double **px,double **py,double **pobs,int* pnobs,double x_interp,double y_interp,double range){

	/*Output and Intermediaries*/
	int          nobs,i,index;
	double      *x            = NULL;
	double      *y            = NULL;
	double      *obs          = NULL;
	Observation *observation  = NULL;
	int         *indices      = NULL;

	/*Treat range*/
	if(range==0){
		range=this->quadtree->root->length;
	}

	/*Find all observations that are in range*/
	nobs=0;
	while(nobs==0){
		xfree((void**)&indices);
		this->quadtree->RangeSearch(&indices,&nobs,x_interp,y_interp,range);
		if(!nobs){
			/*No observation found, double range*/
			range=2*range;
		}
	}
	/*Check that we do not have more than 50 observations*/
	//while(nobs>50){
	//	range=range/2;
	//	xfree((void**)&indices);
	//	this->quadtree->RangeSearch(&indices,&nobs,x_interp,y_interp,range);
	//}

	if(!nobs) _error_("Unexpected error: no observation within current range and more than 50 observations if doubling range");

	/*Allocate vectors*/
	x   = (double*)xmalloc(nobs*sizeof(double));
	y   = (double*)xmalloc(nobs*sizeof(double));
	obs = (double*)xmalloc(nobs*sizeof(double));

	/*Loop over all observations and fill in x, y and obs*/
	for (i=0;i<nobs;i++){
		observation=(Observation*)this->GetObjectByOffset(indices[i]);
		observation->WriteXYObs(&x[i],&y[i],&obs[i]);
	}

	/*Assign output pointer*/
	xfree((void**)&indices);
	*px=x;
	*py=y;
	*pobs=obs;
	*pnobs=nobs;
}/*}}}*/
/*FUNCTION Observations::QuadtreeColoring{{{*/
void Observations::QuadtreeColoring(double* A,double *x,double *y,int n){

	int xi,yi,level;

	for(int i=0;i<n;i++){
		this->quadtree->IntergerCoordinates(&xi,&yi,x[i],y[i]);
		this->quadtree->QuadtreeDepth(&level,xi,yi);
		A[i]=(double)level;
	}

}/*}}}*/
/*FUNCTION Observations::Variomap{{{*/
void Observations::Variomap(double* gamma,double *x,int n){

	/*Output and Intermediaries*/
	int          nobs,i,j,k;
	double       range;
	Observation *observation1 = NULL;
	Observation *observation2 = NULL;
	int         *indices      = NULL;

	int *counter= (int*)xmalloc(n*sizeof(int));
	for(j=0;j<n;j++) counter[j] = 0;
	for(j=0;j<n;j++) gamma[j]   = 0.0;

	for(i=0;i<this->Size();i++){
		observation1=(Observation*)this->GetObjectByOffset(i);

		for(j=0;j<n;j++){
			range=x[j]; _assert_(range>=0.);

			/*Find all observations that are in range*/
			this->quadtree->RangeSearch(&indices,&nobs,observation1->x,observation1->y,range);

			for (k=0;k<nobs;k++){
				observation2 = (Observation*)this->GetObjectByOffset(indices[k]);
				gamma[j]    += 1./2.*pow(observation1->value - observation2->value,2.);
			}

			counter[j] += nobs;
			xfree((void**)&indices);
		}
	}

	/*Normalize semivariogram*/
	for(j=0;j<n;j++){
		if(counter[j]) gamma[j] = gamma[j]/double(counter[j]);
	}

	/*Assign output pointer*/
	xfree((void**)&counter);
}/*}}}*/
