#include "../objects.h"
/*DOCUMENTATION What is a Quadtree? {{{1
 * A Quadtree is a very simple way to group vertices according
 * to their locations. A square that holds all the points of the mesh
 * (or the geometry) is divided into 4 boxes. As soon as one box
 * hold more than 4 vertices, it is divided into 4 new boxes, etc...
 * There cannot be more than MAXDEEP (=30) subdivision.
 * This process is like a Dichotomy in dimension 2
 *
 *  + - -  -    - -    -    - - + -   - + - + - + - -     - - +
 *  |                           |       |   | X |             |
 *                                      + - + - +
 *  |                           |       |   |   |             |
 *                              + -   - + - + - +             +
 *  |                           |       |       |             |
 *                         
 *  |                           |       |       |             |
 *  + - -  -    - -    -    - - + -   - + -   - + - -     - - +
 *  |                           |               |             |
 *                         
 *  |                           |               |             |
 *                         
 *  |                           |               |             |
 *  |                           |               |             |
 *  + - -  -    - -    -    - - + -   -   -   - + - -     - - +
 *  |                           |                             |
 *                         
 *  |                           |                             |
 *                         
 *  |                           |                             |
 *                         
 *  |                           |                             |
 *  |                           |                             |
 *  |                           |                             |
 *  |                           |                             |
 *  |                           |                             |
 *  + - -  -    - -    -    - - + -   -   -   -   - -     - - +
 *
 * The coordinate system used in a quadtree are integers to avoid
 * round-off errors. The vertex in the lower left box has the coordinates
 * (0 0) 
 * The upper right vertex has the follwing coordinates:
 * 2^30 -1           2^30 -1        in decimal
 * 0 1 1 1 .... 1    0 1 1 1 .... 1 in binary
 *  \--   29  --/     \--   29  --/
 * Using binaries is therefore very easy to locate a vertex in a box:
 * we just need to look at the bits from the left to the right (See ::Add)
 }}}1*/
/*MACROS {{{1*/
/* 
 * 
 *    J    j
 *    ^    ^
 *    |    | +--------+--------+
 *    |    | |        |        |
 * 1X |    | |   2    |   3    |
 *    |    | |        |        |
 *    |    | +--------+--------+
 *    |    | |        |        |
 * 0X |    | |   0    |   1    |
 *    |    | |        |        |
 *    |    | +--------+--------+
 *    |    +-----------------------> i
 *    |         
 *    |----------------------------> I
 *              X0        X1  
 *
 * box 0 -> I=0 J=0 IJ=00  = 0
 * box 1 -> I=1 J=0 IJ=01  = 1
 * box 2 -> I=0 J=1 IJ=10  = 2
 * box 3 -> I=1 J=1 IJ=11  = 3
 */
//IJ(i,j,l) returns the box number of i and j with respect to l
//if !j&l and !i&l -> 0 (box zero: lower left )
//if !j&l and  i&l -> 1 (box one:  lower right)
//if  j&l and !i&l -> 2 (box two:  upper left )
//if  j&l and  i&l -> 3 (box three:upper right)
#define IJ(i,j,l)  ((j&l) ? ((i&l) ? 3:2 ) :((i&l) ? 1:0 ))
/*}}}*/

	/*Constructors/Destructors*/
	/*FUNCTION Quadtree::Quadtree(){{{1*/
	Quadtree::Quadtree(){

		/*Initialize fields*/
		this->MaxDepth=30;
		this->NbQuadtreeBox=0;
		this->NbObs=0;

		/*Create container*/
		this->boxcontainer=new DataSet();

		/*Create Root, pointer toward the main box*/
		this->root=NewQuadtreeBox();

		}
	/*}}}1*/
	/*FUNCTION Quadtree::~Quadtree(){{{1*/
	Quadtree::~Quadtree(){

		delete boxcontainer;
		root=NULL;

	}
	/*}}}1*/

	/*Methods*/
/*FUNCTION Quadtree::Add{{{1*/
void  Quadtree::Add(Observation* observation){

	/*Intermediaries*/
	int          xi,yi,ij,level;
	QuadtreeBox **pb = NULL;
	QuadtreeBox  *b  = NULL;
	QuadtreeBox  *bb = NULL;
	Observation  *obs[4];

	/*Get integer coodinates*/
	xi = observation->xi;
	yi = observation->yi;

	/*Initialize level*/
	level=(1L<<this->MaxDepth);// = 2^30

	/*Get inital box (the largest)*/
	pb=&root;

	/*Find the smallest box where the observation is located*/
	while((b=*pb) && (b->nbitems<0)){ 

		/*Go down one level = 00100 -> 00010*/
		level>>=1;

		/*Get next subbox according to the bit value (level)*/
		pb = &b->box[IJ(xi,yi,level)];
	}
	_assert_(level>0);

	/*Now, try to add the vertex, if the box is full (nbitems=4), we have to divide it in 4 new boxes*/
	while((b=*pb) && (b->nbitems==4)){

		/*Copy the 4 observation in the current Quadtreebox*/
		obs[0] = b->obs[0];
		obs[1] = b->obs[1];
		obs[2] = b->obs[2];
		obs[3] = b->obs[3];

		/*set nbitems as negative (now holding boxes instead of observations)*/
		b->nbitems = -b->nbitems;

		/*Initialize the 4 pointers toward the 4 subboxes*/
		b->box[0]=NULL;
		b->box[1]=NULL;
		b->box[2]=NULL;
		b->box[3]=NULL;

		/*level = 00100 -> 00010*/
		level>>=1;

		/*Put the four observations in the new boxes*/
		for (int k=0;k<4;k++){

			/*Get box for observation number k*/
			ij = IJ(obs[k]->xi,obs[k]->yi,level);
			bb = b->box[ij];
			if(!bb){
				b->box[ij]=NewQuadtreeBox();
				bb=b->box[ij];
			}
			bb->obs[bb->nbitems++] = obs[k];
		}

		/*Get the subbox where the current observation is located*/
		ij=IJ(xi,yi,level);
		pb=&b->box[ij];
	}

	/*alloc the QuadtreeBox if necessary and add current observation*/
	b=*pb;
	if(!b){
		b=*pb=NewQuadtreeBox();
	}
	b->obs[b->nbitems++]=observation;
	NbObs++;

}/*}}}*/
/*FUNCTION Quadtree::Echo{{{1*/
void  Quadtree::Echo(void){

	printf("Quadtree:\n");
	printf("   MaxDepth      = %i\n",MaxDepth);
	printf("   NbQuadtreeBox = %i\n",NbQuadtreeBox);
	printf("   NbObs         = %i\n",NbObs);
	printf("   root          = %p\n",root);

}/*}}}*/
/*FUNCTION Quadtree::NewQuadtreeBox {{{1*/
Quadtree::QuadtreeBox* Quadtree::NewQuadtreeBox(void){

	/*Output*/
	QuadtreeBox* newbox=NULL;

	/*Create and initialize a new box*/
	newbox=new QuadtreeBox();
	newbox->nbitems=0;
	newbox->box[0]=NULL;
	newbox->box[1]=NULL;
	newbox->box[2]=NULL;
	newbox->box[3]=NULL;

	/*Add to container*/
	this->boxcontainer->AddObject(newbox);
	NbQuadtreeBox++;

	/*currentbox now points toward next quadtree box*/
	return newbox;
}/*}}}*/
/*FUNCTION Quadtree::QuadtreeColoring{{{1*/
void Quadtree::QuadtreeColoring(double* A,int *xi,int *yi,int n){

	QuadtreeBox **pb = NULL;
	QuadtreeBox  *b  = NULL;
	int          level;

	for(int i=0;i<n;i++){

		/*Initialize level*/
		level=(1L<<this->MaxDepth);// = 2^30

		/*Get inital box (the largest)*/
		pb=&root;

		/*Find the smallest box where the observation is located*/
		while((b=*pb) && (b->nbitems<0)){ 

			/*Color matrix onces more*/
			A[i]+=1;

			/*Go to one box deeper*/
			level>>=1;
			pb = &b->box[IJ(xi[i],yi[i],level)];
		}
	}
}/*}}}*/
