/*!\file:  Kriging.cpp
 * \brief  "c" core code for Kriging
 */ 

#include "./Krigingx.h"
#include "../../shared/shared.h"
#include "../../include/include.h"
#include "../../toolkits/toolkits.h"
#include "../../objects/objects.h"
#include "../modules.h"
#include <gsl/gsl_linalg.h>

int Krigingx(double** ppredictions,double* x, double* y, double* observations, int n_obs,double* x_interp,double* y_interp,int n_interp,Options* options){

	/*output*/
	double *predictions = NULL;

	/*Intermediaries*/
	int     i,j;
	double  numerator,denominator,ratio;
	double *Gamma     = NULL;
	double *GinvG0    = NULL;
	double *Ginv1     = NULL;
	double *GinvZ     = NULL;
	double *gamma0    = NULL;
	double *ones      = NULL;

	/*Memory allocation*/
	predictions =(double*)xmalloc(n_interp*sizeof(double));
	Gamma       =(double*)xmalloc(n_obs*n_obs*sizeof(double));
	gamma0      =(double*)xmalloc(n_obs*sizeof(double));
	ones        =(double*)xmalloc(n_obs*sizeof(double));

	/*First: Create semivariogram matrix for observations*/
	for(i=0;i<n_obs;i++){
		for(j=0;j<=i;j++){
			SemiVariogram(&Gamma[i*n_obs+j],x[i],y[i],x[j],y[j]);
			Gamma[j*n_obs+i] = Gamma[i*n_obs+j];
		}
	}
	for(i=0;i<n_obs;i++) ones[i]=1;

	/*Loop over all interpolations*/
	printf("      interpolation progress:  %5.2lf %%",0.0);
	for(int idx=0;idx<n_interp;idx++){
		if(idx%100==0) printf("\b\b\b\b\b\b\b%5.2lf %%",(double)idx/n_interp*100);

		/*Get semivariogram vector associated to this location*/
		for(i=0;i<n_obs;i++) SemiVariogram(&gamma0[i],x[i],y[i],x_interp[idx],y_interp[idx]);

		/*Solve the three linear systems*/
		GslSolve(&GinvG0,Gamma,gamma0,n_obs);       // Gamma^-1 gamma0
		GslSolve(&Ginv1, Gamma,ones,n_obs);         // Gamma^-1 ones
		GslSolve(&GinvZ, Gamma,observations,n_obs); // Gamma^-1 Z

		/*Prepare predictor*/
		numerator=-1.; denominator=0.;
		for(i=0;i<n_obs;i++) numerator  +=GinvG0[i];
		for(i=0;i<n_obs;i++) denominator+=Ginv1[i];
		ratio=numerator/denominator;

		predictions[idx]=0;
		for(i=0;i<n_obs;i++) predictions[idx] += (gamma0[i]-ratio)*GinvZ[i];

		/*clean-up*/
		xfree((void**)&GinvG0);
		xfree((void**)&Ginv1);
		xfree((void**)&GinvZ);
	}
	printf("\b\b\b\b\b\b\b\b%5.2lf %%\n",100.0);

	/*clean-up and Assign output pointer*/
	xfree((void**)&Gamma);
	xfree((void**)&gamma0);
	xfree((void**)&ones);
	*ppredictions=predictions;
}

void SemiVariogram(double* gamma,double x1,double y1,double x2,double y2){

	/*Intermediaries*/
	double a,c0,c1;

	/*Calculate distance*/
	double r=sqrt(pow(x1-x2,2.)+pow(y1-y2,2.));

	/*Switch between variogram models*/
	switch(2){
		case 1:/*Exponential*/
			c0=0.2;
			c1=0.8;
			a =1;
			*gamma = c0 + c1*(1-exp(-r/a));
			return;
		case 2:/*Gaussian*/
			c0=0.2;
			c1=0.8;
			a =1;
			*gamma = c0 + c1*(1-exp(-pow(r,2.)/pow(a,2.)));
			return;
		default:
			_error_("Not implemented yet");
	}
}

void GslSolve(double** pX,double* A,double* B,int n){
#ifdef _HAVE_GSL_

	/*GSL Matrices and vectors: */
	int              s;
	gsl_matrix_view  a;
	gsl_vector_view  b;
	gsl_vector      *x = NULL;
	gsl_permutation *p = NULL;

	/*A will be modified by LU decomposition. Use copy*/
	double* Acopy = (double*)xmalloc(n*n*sizeof(double));
	memcpy(Acopy,A,n*n*sizeof(double));

	/*Initialize gsl matrices and vectors: */
	a = gsl_matrix_view_array (Acopy,n,n);
	b = gsl_vector_view_array (B,n);
	x = gsl_vector_alloc (n);

	/*Run LU and solve: */
	p = gsl_permutation_alloc (n);
	gsl_linalg_LU_decomp (&a.matrix, p, &s);
	gsl_linalg_LU_solve (&a.matrix, p, &b.vector, x);

	//printf ("x = \n");
	//gsl_vector_fprintf (stdout, x, "%g");

	/*Copy result*/
	double* X = (double*)xmalloc(n*sizeof(double));
	memcpy(X,gsl_vector_ptr(x,0),n*sizeof(double));

	/*Clean up and assign output pointer*/
	xfree((void**)&Acopy);
	gsl_permutation_free(p);
	gsl_vector_free(x);
	*pX=X;
#else
	error_("GSL support required");
#endif
}
