/*\file OptionParse.c
 *\brief: functions to parse the mex options.
 */
#ifdef HAVE_CONFIG_H
    #include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../../shared/shared.h"
#include "../../io/io.h"
#include "../../include/include.h"
#include "./matlabio.h"

#include <mex.h>

/*FUNCTION OptionDoubleParse {{{1*/
OptionDouble* OptionDoubleParse( char* name, const mxArray* prhs[]){

	OptionDouble *odouble = NULL;

	/*check and parse the name  */
	odouble=new OptionDouble;
	odouble->name =(char *) xmalloc((strlen(name)+1)*sizeof(char));
	memcpy(odouble->name,name,(strlen(name)+1)*sizeof(char));

	/*check and parse the value  */
	if (!mxIsClass(prhs[0],"double")){
		_error_("Value of option \"%s\" must be class \"%s\", not class \"%s\".",odouble->name,"double",odouble->name,mxGetClassName(prhs[0]));
	}

	FetchData(&odouble->values,&odouble->numel,&odouble->ndims,&odouble->size,prhs[0]);

	return(odouble);
}/*}}}*/
/*FUNCTION OptionLogicalParse {{{1*/
OptionLogical* OptionLogicalParse( char* name, const mxArray* prhs[]){

	OptionLogical *ological = NULL;

	/*check and parse the name  */
	ological=new OptionLogical;
	ological->name =(char *) xmalloc((strlen(name)+1)*sizeof(char));
	memcpy(ological->name,name,(strlen(name)+1)*sizeof(char));

	/*check and parse the value  */
	if (!mxIsClass(prhs[0],"logical")){
		_error_("Value of option \"%s\" must be class \"%s\", not class \"%s\".",ological->name,"logical",ological->name,mxGetClassName(prhs[0]));
	}

	FetchData(&ological->values,&ological->numel,&ological->ndims,&ological->size,prhs[0]);

	return(ological);
}/*}}}*/
/*FUNCTION OptionCharParse {{{1*/
OptionChar* OptionCharParse( char* name, const mxArray* prhs[]){

	OptionChar  *ochar = NULL;

	/*check and parse the name  */
	ochar=new OptionChar;
	ochar->name =(char *) xmalloc((strlen(name)+1)*sizeof(char));
	memcpy(ochar->name,name,(strlen(name)+1)*sizeof(char));

	/*check and parse the value  */
	if (!mxIsClass(prhs[0],"char")){
		_error_("Value of option \"%s\" must be class \"%s\", not class \"%s\".",ochar->name,"char",ochar->name,mxGetClassName(prhs[0]));
	}

	FetchData(&ochar->values,&ochar->numel,&ochar->ndims,&ochar->size,prhs[0]);

	return(ochar);
}/*}}}*/
/*FUNCTION OptionStructParse {{{1*/
OptionStruct* OptionStructParse( char* name, const mxArray* prhs[]){

	int            i;
	char           namei[161];
	OptionStruct  *ostruct    = NULL;
	Option        *option     = NULL;
	const mwSize  *ipt        = NULL;
	const mxArray *structi;
	mwIndex        sindex;

	/*check and parse the name  */
	ostruct=new OptionStruct;
	ostruct->name =(char*)xmalloc((strlen(name)+1)*sizeof(char));
	memcpy(ostruct->name,name,(strlen(name)+1)*sizeof(char));

	/*check and parse the value  */
	if (!mxIsClass(prhs[0],"struct")){
		_error_("Value of option \"%s\" must be class \"%s\", not class \"%s\".",ostruct->name,"struct",ostruct->name,mxGetClassName(prhs[0]));
	}

	ostruct->numel=mxGetNumberOfElements(prhs[0]);
	ostruct->ndims=mxGetNumberOfDimensions(prhs[0]);
	ipt           =mxGetDimensions(prhs[0]);
	ostruct->size =(int *) xmalloc(ostruct->ndims*sizeof(int));
	for (i=0; i<ostruct->ndims; i++) ostruct->size[i]=(int)ipt[i];
	if (ostruct->numel) ostruct->values=(Options**) xmalloc(ostruct->numel*sizeof(Options *));

	/*loop through and process each element of the struct array  */
	for (sindex=0; sindex<ostruct->numel; sindex++) {
		ostruct->values[sindex]=new Options;

		/*loop through and process each field for the element  */
		for (i=0; i<mxGetNumberOfFields(prhs[0]); i++) {
			sprintf(namei,"%s.%s",name,mxGetFieldNameByNumber(prhs[0],i));
			structi=mxGetFieldByNumber(prhs[0],sindex,i);

			option=(Option*)OptionParse(namei,&structi);
			ostruct->values[sindex]->AddObject((Object*)option);
			option=NULL;
		}
	}

	return(ostruct);
}/*}}}*/
/*FUNCTION OptionCellParse {{{1*/
OptionCell* OptionCellParse( char* name, const mxArray* prhs[]){

	int            i;
	int           *dims;
	char           namei[161];
	char           cstr[81];
	OptionCell    *ocell      = NULL;
	Option        *option     = NULL;
	const mwSize  *ipt        = NULL;
	const mxArray *celli;
	mwIndex        cindex;

	/*check and parse the name  */
	ocell=new OptionCell;
	ocell->name =(char *) xmalloc((strlen(name)+1)*sizeof(char));
	memcpy(ocell->name,name,(strlen(name)+1)*sizeof(char));

	/*check and parse the value  */
	if (!mxIsClass(prhs[0],"cell")){
		_error_("Value of option \"%s\" must be class \"%s\", not class \"%s\".",ocell->name,"cell",ocell->name,mxGetClassName(prhs[0]));
	}

	ocell->numel=mxGetNumberOfElements(prhs[0]);
	ocell->ndims=mxGetNumberOfDimensions(prhs[0]);
	ipt         =mxGetDimensions(prhs[0]);
	ocell->size =(int *) xmalloc(ocell->ndims*sizeof(int));
	for (i=0; i<ocell->ndims; i++) ocell->size[i]=(int)ipt[i];
	ocell->values=new Options;

	/*loop through and process each element of the cell array  */
	dims=(int *) xmalloc(ocell->ndims*sizeof(int));
	for (cindex=0; cindex<ocell->numel; cindex++) {
		ColumnWiseDimsFromIndex(dims,(int)cindex,ocell->size,ocell->ndims);
		StringFromDims(cstr,dims,ocell->ndims);
		#ifdef _INTEL_WIN_
			_snprintf(namei,161,"%s%s",name,cstr);
		#else
			snprintf(namei,161,"%s%s",name,cstr);
		#endif
		celli=mxGetCell(prhs[0],cindex);

		option=(Option*)OptionParse(namei,&celli);
		ocell->values->AddObject((Object*)option);
		option=NULL;
	}
	xfree((void**)&dims);

	return(ocell);
}/*}}}*/
/*FUNCTION OptionParse{{{1*/
Option* OptionParse(char* name, const mxArray* prhs[]){

	Option *option = NULL;
	mxArray       *lhs[1];

	/*parse the value according to the matlab data type  */
	if     (mxIsClass(prhs[0],"double"))  option=(Option*)OptionDoubleParse(name,prhs);
	else if(mxIsClass(prhs[0],"logical")) option=(Option*)OptionLogicalParse(name,prhs);
	else if(mxIsClass(prhs[0],"char"))    option=(Option*)OptionCharParse(name,prhs);
	else if(mxIsClass(prhs[0],"struct"))  option=(Option*)OptionStructParse(name,prhs);
	else if(mxIsClass(prhs[0],"cell"))    option=(Option*)OptionCellParse(name,prhs);
	else {
		_printf_(true,"  Converting value of option \"%s\" from unrecognized class \"%s\" to class \"%s\".\n",name,mxGetClassName(prhs[0]),"struct");
		if (!mexCallMATLAB(1,lhs,1,(mxArray**)prhs,"struct")) {
			option=(Option*)OptionStructParse(name,(const mxArray**)lhs);
			mxDestroyArray(lhs[0]);
		}
		else _error_("Second argument value of option \"%s\" is of unrecognized class \"%s\".",name,mxGetClassName(prhs[0]));
	}

	return(option);
}/*}}}*/
