/*\file KMLOverlay.c
 *\brief: KML file overlay mex module.
 */
#include "./KMLOverlay.h"

void mexFunction( int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[]){

	int i,verbose=1;

	/*input: */
	char*   filkml=NULL;
	char*   filkmz=NULL;

	FILE*   fid=NULL;

	Options* options=NULL;
	int      nlat=0,nlong=0;
	double*  lataxis =NULL;
	double*  longaxis=NULL;
	int      nimages=0;
	char**   pimages=NULL;
	double   dzip=0;
	char*    czip=NULL;

	/* output: */
	int     ierror=0;

	/*Boot module: */
	MODULEBOOT();

	/*checks on arguments on the matlab side: */
	if(nlhs>NLHS){
		KMLOverlayUsage(); _error_("KMLOverlay usage error");
	}
	if(nrhs<NRHS){
		KMLOverlayUsage(); _error_("KMLOverlay usage error");
	}

	/*Input datasets: */
	FetchData(&filkml,FILENAME);
	FetchData(&options,NRHS,nrhs,prhs);

	options->Get(&lataxis ,&nlat ,"lataxis" );
	if (verbose && lataxis) for (i=0; i<nlat; i++) printf("  lataxis [%d]=%g\n",i,lataxis[i]);
	options->Get(&longaxis,&nlong,"longaxis");
	if (verbose && longaxis) for (i=0; i<nlong; i++) printf("  longaxis[%d]=%g\n",i,longaxis[i]);
	options->Get(&pimages,&nimages,"images");
	if (verbose && pimages) for (i=0; i<nimages; i++) printf("  pimages[%d]=\"%s\"\n",i,pimages[i]);
	options->Get(&dzip,"zip",0.);
	if (verbose) printf("  dzip=%g\n",dzip);

	/*some checks*/
	if (nlat !=2) _error_("Latitudinal axes \"lataxis\" require two double values, not %d.",nlat);
	if (nlong!=2) _error_("Longitudinal axes \"longaxis\" require two double values, not %d.",nlong);
	if (!nimages) _error_("No image files provided.");

	if ((int)dzip){
		filkmz=filkml;
		filkml=(char*)mxMalloc(8*sizeof(char));
		strcpy(filkml,"doc.kml");
	}

	if(!strlen(filkml)) strcpy(filkml,"stdout");

	if(verbose) printf("Opening kml overlay file \"%s\".\n",filkml);
	fid=fopen(filkml,"w");

	/* Run core computations: */
	if (verbose) printf("Calling core:\n");
	KMLOverlayx(&ierror,lataxis,longaxis,nimages,pimages,fid);

	if (verbose) printf("Closing file \"%s\".\n",filkml);
	fclose(fid);

	/* Create kmz file, if specified: */
	if ((int)dzip) {
		czip=(char*)mxMalloc((5+strlen(filkmz)+1+strlen(filkml)+1)*sizeof(char));
		czip[0]='\0';
		strcat(czip,"!zip ");
		strcat(czip,filkmz);
		strcat(czip," ");
		strcat(czip,filkml);
		for (i=0; i<nimages; i++)
			if (strlen(pimages[i]) && strncmp(pimages[i],"http",4)) {
				czip=(char*)mxRealloc(czip,(strlen(czip)+1+strlen(pimages[i])+1)*sizeof(char));
				strcat(czip," ");
				strcat(czip,pimages[i]);
			}
		if (verbose) printf("Zipping file \"%s\".\n",filkmz);
		if (verbose) printf("%s\n",czip);

		if (mexEvalString(czip)) _error_("Error zipping file \"%s\".",filkmz);
		xfree((void**)&czip);
		xfree((void**)&filkmz);
	}

	/*Write data: */
	WriteData(ERRORFLAG,ierror);

	/*Clean-up*/
	if (pimages) {
		for (i=nimages; i>0; i--) xfree((void**)&(pimages[i-1]));
		xfree((void**)&pimages);
	}
	xfree((void**)&longaxis);
	xfree((void**)&lataxis);
	delete options;
	xfree((void**)&filkml);

	/*end module: */
	MODULEEND();
}

void KMLOverlayUsage(void){
	_printf_(true,"KMLOverlay - KML file overlay module:\n");
	_printf_(true,"\n");
	_printf_(true,"   This module reads a list of image files and writes a KML or KMZ overlay file.\n");
	_printf_(true,"\n");
	_printf_(true,"   Usage:\n");
	_printf_(true,"      ierror=KMLOverlay(kmlfile,'param name',param,...);\n");
	_printf_(true,"\n");
	_printf_(true,"      kmlfile     KML or KMZ file name (string)\n");
	_printf_(true,"\n");
	_printf_(true,"      lataxis     latitude axis (double vector [south north], required)\n");
	_printf_(true,"      longaxis    longitude axis (double vector [west east], required)\n");
	_printf_(true,"      images      relative or http image file names (string or array of strings or cell array of strings, required)\n");
	_printf_(true,"      zip         flag to zip the doc.kml and image files into kmzfile (double, non-zero for kmz)\n");
	_printf_(true,"\n");
	_printf_(true,"      ierror     error flag (double, non-zero for error)\n");
	_printf_(true,"\n");
	_printf_(true,"   Example:\n");
	_printf_(true,"      KMLOverlay(kmlfile,'lataxis',[south north],'longaxis',[west east],'images',{'file1.png','http://issm/file2.png'},'zip',1);\n");
	_printf_(true,"\n");
}
