/*\file FetchData.cpp:
 * \brief: general I/O interface to fetch data in python
 */

#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#define PY_ARRAY_UNIQUE_SYMBOL PythonIOSymbol
#define NO_IMPORT

#include "../../toolkits/toolkits.h"
#include "../../include/include.h"
#include "../../shared/shared.h"

/*Primitive data types*/
/*FUNCTION FetchData(double* pscalar,PyObject* py_float){{{1*/
void FetchData(double* pscalar,PyObject* py_float){

	double scalar;

	/*return internal value: */
	scalar=PyFloat_AsDouble(py_float);

	/*output: */
	*pscalar=scalar;
}
/*}}}*/
/*FUNCTION FetchData(int* pinteger,PyObject* py_long){{{1*/
void FetchData(int* pinteger, PyObject* py_long){

	int integer;

	/*return internal value: */
	integer=(int)PyLong_AsLong(py_long);

	/*output: */
	*pinteger=integer;
}
/*}}}*/
/*FUNCTION FetchData(bool* pboolean,PyObject* py_boolean){{{1*/
void FetchData(bool* pboolean,PyObject* py_boolean){

	bool boolean;
	
	/*check this is indeed a subtype of long type: */
	if(!PyBool_Check(py_boolean))_error_("expecting a boolean in input!");

	/*extract boolean: */
	boolean=(bool)PyLong_AsLong(py_boolean);

	/*simple copy: */
	*pboolean=boolean;
	
}
/*}}}*/
/*FUNCTION FetchData(double** pmatrix,int* pM, int* pN, PyObject* py_matrix){{{1*/
void FetchData(double** pmatrix,int* pM,int *pN,PyObject* py_matrix){

	/*output: */
	double* matrix=NULL;
	int M,N;
	int ndim;
	npy_intp*  dims=NULL;

	/*retrive dimensions: */
	ndim=PyArray_NDIM((const PyArrayObject*)py_matrix);
	if(ndim!=2)_error_("expecting an MxN matrix in input!");
	dims=PyArray_DIMS((PyArrayObject*)py_matrix);
	M=dims[0]; N=dims[1];
	
	/*retrieve internal value: */
	matrix=(double*)PyArray_DATA((PyArrayObject*)py_matrix);

	/*output: */
	if(pM)*pM=M;
	if(pN)*pN=N;
	if(pmatrix)*pmatrix=matrix;
}
/*}}}*/

/*Python version dependent: */
#if _PYTHON_MAJOR_ >= 3 
/*FUNCTION FetchData(char** pstring,PyObject* py_unicode){{{1*/
void FetchData(char** pstring,PyObject* py_unicode){

	PyObject* py_bytes;
	char* string=NULL;

	
	/*convert to bytes format: */
	PyUnicode_FSConverter(py_unicode,&py_bytes);

	/*convert from bytes to string: */
	string=PyBytes_AS_STRING(py_bytes);
	
	*pstring=string;
}
/*}}}*/
#else
/*FUNCTION FetchData(char** pstring,PyObject* py_string){{{1*/
void FetchData(char** pstring,PyObject* py_string){

	char* string=NULL;

	/*extract internal string: */
	string=PyString_AsString(py_string);
	
	*pstring=string;
}
/*}}}*/
#endif
