function results=control_core(models,inputs)
%CONTROL_CORE - compute the core inversion
%
%   Usage:
%      results=control_core(models,inputs);
%

%recover models
m_dh=models.dh;
m_dv=models.dv;
m_ds=models.ds;
m_sl=models.sl;

%recover parameters common to all solutions
debug=m_dh.parameters.debug;
dim=m_dh.parameters.dim;
ishutter=m_dh.parameters.ishutter;
ismacayealpattyn=m_dh.parameters.ismacayealpattyn;
isstokes=m_dh.parameters.isstokes;

%initialize control parameters
param_g=models.dh.parameters.param_g;

%set optimization options.
options=ControlOptions(m_dh.parameters);

%Take car of Stokes : compute slope and get spc once for all
if isstokes,

	%bed slope
	displaystring(debug,'\n%s',['      computing bed slope (x and y derivatives)...']);
	slopex=diagnostic_core_linear(m_sl,inputs,'slope_compute','bedx');
	slopey=diagnostic_core_linear(m_sl,inputs,'slope_compute','bedy');
	slopex=FieldExtrude(m_sl.elements,m_sl.nodes,m_sl.loads,m_sl.materials,slopex,'slopex',0);
	slopey=FieldExtrude(m_sl.elements,m_sl.nodes,m_sl.loads,m_sl.materials,slopey,'slopey',0);
	inputs=add(inputs,'bedslopex',slopex,'doublevec',m_sl.parameters.numberofdofspernode,m_sl.parameters.numberofnodes);
	inputs=add(inputs,'bedslopey',slopey,'doublevec',m_sl.parameters.numberofdofspernode,m_sl.parameters.numberofnodes);

	%3d velocity
	displaystring(debug,'\n%s',['      get vertical spcs: launch diagnostic...']);
	displaystring(debug,'\n%s',['         computing horizontal velocities...']);
	u_g=diagnostic_core_nonlinear(m_dh,inputs,'diagnostic','horiz');
	displaystring(debug,'\n%s',['         extruding horizontal velocities...']);
	u_g_horiz=FieldExtrude(m_dh.elements,m_dh.nodes,m_dh.loads,m_dh.materials,u_g,'velocity',1);
	displaystring(debug,'\n%s',['         computing vertical velocities...']);
	inputs=add(inputs,'velocity',u_g_horiz,'doublevec',m_dh.parameters.numberofdofspernode,m_dh.parameters.numberofnodes);
	u_g_vert=diagnostic_core_linear(m_dv,inputs,'diagnostic','vert');
	displaystring(debug,'\n%s',['         combining horizontal and vertical velocities...']);
	u_g=zeros(m_dh.parameters.numberofnodes*3,1);
	u_g(dofsetgen([1,2],3,m_dh.parameters.numberofnodes*3))=u_g_horiz;
	u_g(dofsetgen([3],3,m_dh.parameters.numberofnodes*3))=u_g_vert;
	displaystring(debug,'\n%s',['         computing pressure according to MacAyeal...']);
	p_g=ComputePressure(m_dh.elements,m_dh.nodes,m_dh.loads,m_dh.materials,m_dh.parameters,inputs);
	p_g=p_g/m_ds.parameters.stokesreconditioning;
	%recombine u_g and p_g: 
	u_g_stokes=zeros(m_ds.nodesets.gsize,1);
	u_g_stokes(dofsetgen([1,2,3],4,m_ds.nodesets.gsize))=u_g;
	u_g_stokes(dofsetgen([4],4,m_ds.nodesets.gsize))=p_g;
	inputs=add(inputs,'velocity',u_g_stokes,'doublevec',4,m_ds.parameters.numberofnodes);
	displaystring(debug,'\n%s',['         update boundary conditions for stokes using velocities previously computed...']);
	m_ds.y_g=zeros(m_ds.nodesets.gsize,1);
	m_ds.y_g(dofsetgen([1,2,3],4,m_ds.nodesets.gsize))=u_g;
	[m_ds.ys m_ds.ys0]=Reducevectorgtos(m_ds.y_g,m_ds.nodesets);
	displaystring(debug,'\n%s',['         computing stokes velocities and pressure ...']);
	u_g=diagnostic_core_nonlinear(m_ds,inputs,'diagnostic','stokes');
	inputs=add(inputs,'velocity',u_g,'doublevec',4,m_ds.parameters.numberofnodes);
end

for n=1:m_dh.parameters.nsteps,

	%set options
	options=optimset(options,'MaxFunEvals',m_dh.parameters.maxiter(n));

	disp(sprintf('\n%s%s%s%s\n',['   control method step ' num2str(n) '/' num2str(m_dh.parameters.nsteps)]));

	%update inputs with new fit
	inputs=add(inputs,'fit',m_dh.parameters.fit(n),'double');
	inputs=add(inputs,m_dh.parameters.control_type,param_g,'doublevec',1,m_dh.parameters.numberofnodes);

	%Update inputs in datasets
	if isstokes,
		[m_ds.elements,m_ds.nodes,m_ds.loads,m_ds.materials]=UpdateFromInputs(m_ds.elements,m_ds.nodes,m_ds.loads,m_ds.materials,inputs);
	else
		[m_dh.elements,m_dh.nodes,m_dh.loads,m_dh.materials]=UpdateFromInputs(m_dh.elements,m_dh.nodes,m_dh.loads,m_dh.materials,inputs);
	end

	displaystring(debug,'\n%s',['      computing gradJ...']);
	if isstokes,
		[u_g c(n).grad_g]=GradJCompute(m_ds,inputs,'diagnostic','stokes');
		inputs=add(inputs,'velocity',u_g,'doublevec',4,m_ds.parameters.numberofnodes);
	else
		[u_g c(n).grad_g]=GradJCompute(m_dh,inputs,'diagnostic','horiz');
		inputs=add(inputs,'velocity',u_g,'doublevec',2,m_dh.parameters.numberofnodes);
	end

	displaystring(debug,'\n%s',['      normalizing directions...']);
	if n>=2,
		c(n).grad_g=Orth(c(n).grad_g,c(n-1).grad_g);
	else
		c(n).grad_g=Orth(c(n).grad_g,{});
	end

	%visualize direction.
	if m_dh.parameters.plot
		plot_direction;
	end

	displaystring(debug,'\n%s',['      optimizing along gradient direction...']);
	if isstokes,
		[search_scalar c(n).J]=ControlOptimization('objectivefunctionC',0,1,options,m_dh,inputs,param_g,c(n).grad_g,n,'diagnostic','horiz');
	else
		[search_scalar c(n).J]=ControlOptimization('objectivefunctionC',0,1,options,m_ds,inputs,param_g,c(n).grad_g,n,'diagnostic','horiz');
	end

	displaystring(debug,'\n%s',['      updating parameter using optimized search scalar...']);
	param_g=param_g+search_scalar*m_dh.parameters.optscal(n)*c(n).grad_g;

	displaystring(debug,'\n%s',['      constraining the new distribution...']);
	param_g=ControlConstrain(param_g,m_dh.parameters);

	%visualize direction.
	if m_dh.parameters.plot,
		plot_newdistribution;
	end

	disp(['      value of misfit J after optimization #' num2str(n) ':' num2str(c(n).J)]);
end

%generate output
displaystring(debug,'\n%s',['      preparing final velocity solution...']);

%compute final velocity from diagnostic_core (horiz+vertical)
inputs=add(inputs,m_dh.parameters.control_type,param_g,'doublevec',1,m_dh.parameters.numberofnodes);
results_diag=diagnostic_core(models,inputs);

%Recover misfit at each iteration of the control method 
J=zeros(length(c),1);
for i=1:length(c),
	J(i)=c(i).J;
end

%build results
results.time=0;
results.step=1;
results.J=J;
results.param_g=param_g;
results.u_g=results_diag.u_g;
