/*\file FetchData.cpp:
 * \brief: general I/O interface to fetch data in matlab
 */

#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <mex.h>
#include "../../shared/shared.h"
#include "../../include/include.h"
#include "./matlabio.h"

/*FUNCTION FetchData(double** pmatrix,int* pM,int *pN,const mxArray* dataref){{{1*/
void FetchData(double** pmatrix,int* pM,int *pN,const mxArray* dataref){

	double*  outmatrix=NULL;
	int      outmatrix_rows,outmatrix_cols;

	if(mxIsEmpty(dataref) ){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outmatrix_rows=0;
		outmatrix_cols=0;
		outmatrix=NULL;
	}
	else if(mxIsClass(dataref,"double") || mxIsClass(dataref,"single")){
		/*Check dataref is not pointing to NaN: */
		if ( mxIsNaN(*(mxGetPr(dataref))) && (mxGetM(dataref)==1) && (mxGetN(dataref)==1) ){
			outmatrix_rows=0;
			outmatrix_cols=0;
			outmatrix=NULL;
		}
		else{
			/*Convert matlab matrix to double* matrix: */
			MatlabMatrixToDoubleMatrix(&outmatrix,&outmatrix_rows,&outmatrix_cols,dataref);
		}
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}
			
	/*Assign output pointers:*/
	*pmatrix=outmatrix;
	if (pM)*pM=outmatrix_rows;
	if (pN)*pN=outmatrix_cols;

}
/*}}}*/
/*FUNCTION FetchData(double** pmatrix,int* pnumel,int* pndims,int** psize,const mxArray* dataref){{{1*/
void FetchData(double** pmatrix,int* pnumel,int* pndims,int** psize,const mxArray* dataref){

	double*  outmatrix=NULL;
	int      outmatrix_numel,outmatrix_ndims;
	int*     outmatrix_size=NULL;

	if(mxIsEmpty(dataref) ){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outmatrix_numel=0;
		outmatrix_ndims=0;
		outmatrix_size =NULL;
		outmatrix=NULL;
	}
	else if (mxIsClass(dataref,"double") ){

		/*Check dataref is not pointing to NaN: */
		if ( mxIsNaN(*(mxGetPr(dataref))) && (mxGetNumberOfElements(dataref)==1) ){
			outmatrix_numel=0;
			outmatrix_ndims=0;
			outmatrix_size =NULL;
			outmatrix=NULL;
		}
		else{

			/*Convert matlab n-dim array to double* matrix: */
			MatlabNArrayToNArray(&outmatrix,&outmatrix_numel,&outmatrix_ndims,&outmatrix_size,dataref);
		}
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}
			
	/*Assign output pointers:*/
	*pmatrix=outmatrix;
	if (pnumel)*pnumel=outmatrix_numel;
	if (pndims)*pndims=outmatrix_ndims;
	if (psize )*psize =outmatrix_size;
	else xfree((void**)&outmatrix_size);

}
/*}}}*/
/*FUNCTION FetchData(int** pmatrix,int* pM,int *pN,const mxArray* dataref){{{1*/
void FetchData(int** pmatrix,int* pM,int *pN,const mxArray* dataref){

	int     i,outmatrix_rows,outmatrix_cols;
	double *doublematrix=NULL;
	int    *outmatrix=NULL;

	if(mxIsEmpty(dataref) ){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outmatrix_rows=0;
		outmatrix_cols=0;
		outmatrix=NULL;
	}
	else if (mxIsClass(dataref,"double") ){

		/*Check dataref is not pointing to NaN: */
		if ( mxIsNaN(*(mxGetPr(dataref))) && (mxGetM(dataref)==1) && (mxGetN(dataref)==1) ){
			outmatrix_rows=0;
			outmatrix_cols=0;
			outmatrix=NULL;
		}
		else{

			/*Convert matlab matrix to double* matrix: */
			MatlabMatrixToDoubleMatrix(&doublematrix,&outmatrix_rows,&outmatrix_cols,dataref);

			/*Convert double matrix into integer matrix: */
			outmatrix=(int*)xmalloc(outmatrix_rows*outmatrix_cols*sizeof(int));
			for(i=0;i<outmatrix_rows*outmatrix_cols;i++)outmatrix[i]=(int)doublematrix[i];
		}
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pmatrix=outmatrix;
	if (pM)*pM=outmatrix_rows;
	if (pN)*pN=outmatrix_cols;
}
/*}}}*/
/*FUNCTION FetchData(bool** pmatrix,int* pM,int *pN,const mxArray* dataref){{{1*/
void FetchData(bool** pmatrix,int* pM,int *pN,const mxArray* dataref){

	int     i,outmatrix_rows,outmatrix_cols;
	double *doublematrix=NULL;
	bool   *outmatrix=NULL;

	if(mxIsEmpty(dataref) ){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outmatrix_rows=0;
		outmatrix_cols=0;
		outmatrix=NULL;
	}
	else if (mxIsClass(dataref,"double") ){

		/*Check dataref is not pointing to NaN: */
		if ( mxIsNaN(*(mxGetPr(dataref))) && (mxGetM(dataref)==1) && (mxGetN(dataref)==1) ){
			outmatrix_rows=0;
			outmatrix_cols=0;
			outmatrix=NULL;
		}
		else{

			/*Convert matlab matrix to double* matrix: */
			MatlabMatrixToDoubleMatrix(&doublematrix,&outmatrix_rows,&outmatrix_cols,dataref);

			/*Convert double matrix into integer matrix: */
			outmatrix=(bool*)xmalloc(outmatrix_rows*outmatrix_cols*sizeof(bool));
			for(i=0;i<outmatrix_rows;i++)outmatrix[i]=(bool)doublematrix[i];
		}
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pmatrix=outmatrix;
	if (pM)*pM=outmatrix_rows;
	if (pN)*pN=outmatrix_cols;
}
/*}}}*/
/*FUNCTION FetchData(bool** pmatrix,int* pnumel,int* pndims,int** psize,const mxArray* dataref){{{1*/
void FetchData(bool** pmatrix,int* pnumel,int* pndims,int** psize,const mxArray* dataref){

	int      i;
	int      outmatrix_numel,outmatrix_ndims;
	int*     outmatrix_size=NULL;
	double*  doublematrix=NULL;
	bool*    outmatrix=NULL;

	if(mxIsEmpty(dataref) ){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outmatrix_numel=0;
		outmatrix_ndims=0;
		outmatrix_size =NULL;
		outmatrix=NULL;
	}
	else if (mxIsClass(dataref,"logical") ){

		/*Check dataref is not pointing to NaN: */
		if ( mxIsNaN(*((bool*)mxGetData(dataref))) && (mxGetNumberOfElements(dataref)==1) ){
			outmatrix_numel=0;
			outmatrix_ndims=0;
			outmatrix_size =NULL;
			outmatrix=NULL;
		}
		else{

			/*Convert matlab n-dim array to bool* matrix: */
			MatlabNArrayToNArray(&outmatrix,&outmatrix_numel,&outmatrix_ndims,&outmatrix_size,dataref);
		}
	}
	else if (mxIsClass(dataref,"double") ){

		/*Check dataref is not pointing to NaN: */
		if ( mxIsNaN(*(mxGetPr(dataref))) && (mxGetNumberOfElements(dataref)==1) ){
			outmatrix_numel=0;
			outmatrix_ndims=0;
			outmatrix_size =NULL;
			outmatrix=NULL;
		}
		else{

			/*Convert matlab n-dim array to double* matrix: */
			MatlabNArrayToNArray(&doublematrix,&outmatrix_numel,&outmatrix_ndims,&outmatrix_size,dataref);

			/*Convert double matrix into bool matrix: */
			outmatrix=(bool*)xmalloc(outmatrix_numel*sizeof(bool));
			for(i=0;i<outmatrix_numel;i++)outmatrix[i]=(bool)doublematrix[i];
			xfree((void**)&doublematrix);
		}
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}
			
	/*Assign output pointers:*/
	*pmatrix=outmatrix;
	if (pnumel)*pnumel=outmatrix_numel;
	if (pndims)*pndims=outmatrix_ndims;
	if (psize )*psize =outmatrix_size;
	else xfree((void**)&outmatrix_size);

}
/*}}}*/
/*FUNCTION FetchData(Matrix** pmatrix,const mxArray* dataref){{{1*/
void FetchData(Matrix** pmatrix,const mxArray* dataref){
	
	Matrix* outmatrix=NULL;
	int dummy=0;

	if (mxIsClass(dataref,"double") ){
			
		/*Convert matlab matrix to matrix: */
		outmatrix=MatlabMatrixToMatrix(dataref);

	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pmatrix=outmatrix;
}
/*}}}*/
/*FUNCTION FetchData(double** pvector,int* pM,const mxArray* dataref){{{1*/
void FetchData(double** pvector,int* pM,const mxArray* dataref){

	double* outvector=NULL;
	int outvector_rows;

	if(mxIsEmpty(dataref)){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outvector_rows=0;
		outvector=NULL;
	}
	else if (mxIsClass(dataref,"double") ){

		/*Convert matlab vector to double*  vector: */
		MatlabVectorToDoubleVector(&outvector,&outvector_rows,dataref);

	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pvector=outvector;
	if (pM)*pM=outvector_rows;
}
/*}}}*/
/*FUNCTION FetchData(int** pvector,int* pM,const mxArray* dataref){{{1*/
void FetchData(int** pvector,int* pM,const mxArray* dataref){

	int    i;
	double *doublevector   = NULL;
	int    *outvector      = NULL;
	int     outvector_rows;

	if(mxIsEmpty(dataref)){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outvector_rows=0;
		outvector=NULL;
	}
	else if (mxIsClass(dataref,"double") ){

		/*Convert matlab vector to double*  vector: */
		MatlabVectorToDoubleVector(&doublevector,&outvector_rows,dataref);

		/*Convert double vector into integer vector: */
		outvector=(int*)xmalloc(outvector_rows*sizeof(int));
		for(i=0;i<outvector_rows;i++)outvector[i]=(int)doublevector[i];
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pvector=outvector;
	if (pM)*pM=outvector_rows;
}
/*}}}*/
/*FUNCTION FetchData(bool** pvector,int* pM,const mxArray* dataref){{{1*/
void FetchData(bool** pvector,int* pM,const mxArray* dataref){

	int    i;
	double *doublevector   = NULL;
	bool   *outvector      = NULL;
	int     outvector_rows;

	if(mxIsEmpty(dataref)){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outvector_rows=0;
		outvector=NULL;
	}
	else if (mxIsClass(dataref,"double") ){

		/*Convert matlab vector to double*  vector: */
		MatlabVectorToDoubleVector(&doublevector,&outvector_rows,dataref);

		/*Convert double vector into integer vector: */
		outvector=(bool*)xmalloc(outvector_rows*sizeof(bool));
		for(i=0;i<outvector_rows;i++)outvector[i]=(bool)doublevector[i];
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pvector=outvector;
	if (pM)*pM=outvector_rows;
}
/*}}}*/
/*FUNCTION FetchData(float** pvector,int* pM,const mxArray* dataref){{{1*/
void FetchData(float** pvector,int* pM,const mxArray* dataref){

	int    i;
	double *doublevector   = NULL;
	float  *outvector      = NULL;
	int     outvector_rows;

	if(mxIsEmpty(dataref)){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outvector_rows=0;
		outvector=NULL;
	}
	else if (mxIsClass(dataref,"double") ){

		/*Convert matlab vector to double*  vector: */
		MatlabVectorToDoubleVector(&doublevector,&outvector_rows,dataref);

		/*Convert double vector into float vector: */
		outvector=(float*)xmalloc(outvector_rows*sizeof(float));
		for(i=0;i<outvector_rows;i++)outvector[i]=(float)doublevector[i];
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pvector=outvector;
	if (pM)*pM=outvector_rows;
}
/*}}}*/
/*FUNCTION FetchData(Vector** pvector,const mxArray* dataref){{{1*/
void FetchData(Vector** pvector,const mxArray* dataref){
	
	Vector* vector=NULL;
	int dummy;

	if(mxIsEmpty(dataref)){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		vector=new Vector(0);
	}
	else if (mxIsClass(dataref,"double") ){

		/*Convert matlab vector to petsc vector: */
		vector=MatlabVectorToVector(dataref);
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pvector=vector;
}
/*}}}*/
/*FUNCTION FetchData(char** pstring,const mxArray* dataref){{{1*/
void FetchData(char** pstring,const mxArray* dataref){

	char* outstring=NULL;


	/*Ok, the string should be coming directly from the matlab workspace: */
	if (!mxIsClass(dataref,"char")){
		_error_("input data_type is not a string!");
	}
	else{
		/*Recover the string:*/
		int stringlen;
		
		stringlen = mxGetM(dataref)*mxGetN(dataref)+1;
		outstring = (char*)xmalloc(sizeof(mxChar)*stringlen);
		mxGetString(dataref,outstring,stringlen);
	}

	/*Assign output pointers:*/
	*pstring=outstring;
}
/*FUNCTION FetchData(char** pmatrix,int* pnumel,int* pndims,int** psize,const mxArray* dataref){{{1*/
void FetchData(char** pmatrix,int* pnumel,int* pndims,int** psize,const mxArray* dataref){

	int      outmatrix_numel,outmatrix_ndims;
	int*     outmatrix_size=NULL;
	char*    outmatrix=NULL;

	if(mxIsEmpty(dataref) ){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outmatrix_numel=0;
		outmatrix_ndims=0;
		outmatrix_size =NULL;
		outmatrix=NULL;
	}
	else if (mxIsClass(dataref,"char") ){

		/*Check dataref is not pointing to NaN: */
		if ( mxIsNaN(*(mxGetPr(dataref))) && (mxGetNumberOfElements(dataref)==1) ){
			outmatrix_numel=0;
			outmatrix_ndims=0;
			outmatrix_size =NULL;
			outmatrix=NULL;
		}
		else{

			/*Convert matlab n-dim array to char* matrix: */
			MatlabNArrayToNArray(&outmatrix,&outmatrix_numel,&outmatrix_ndims,&outmatrix_size,dataref);
		}
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}
			
	/*Assign output pointers:*/
	*pmatrix=outmatrix;
	if (pnumel)*pnumel=outmatrix_numel;
	if (pndims)*pndims=outmatrix_ndims;
	if (psize )*psize =outmatrix_size;
	else xfree((void**)&outmatrix_size);

}
/*}}}*/
/*FUNCTION FetchData(double* pscalar,const mxArray* dataref){{{1*/
void FetchData(double* pscalar,const mxArray* dataref){

	double scalar;

	if (!mxIsClass(dataref,"double")){
		_error_("input data_type is not a double!");
	}
	else{
		/*Recover the double: */
		scalar=mxGetScalar(dataref);
	}

	/*Assign output pointers:*/
	*pscalar=scalar;
}
/*}}}*/
/*FUNCTION FetchData(int* pinteger,const mxArray* dataref){{{1*/
void FetchData(int* pinteger,const mxArray* dataref){

	int integer;

	if (!mxIsClass(dataref,"double")){
		_error_("input data_type is not a scalar!");
	}
	else{
		/*Recover the double: */
		integer=(int)mxGetScalar(dataref);
	}

	/*Assign output pointers:*/
	*pinteger=integer;
}
/*}}}*/
/*FUNCTION FetchData(bool* pboolean,const mxArray* dataref){{{1*/
void FetchData(bool* pboolean,const mxArray* dataref){

	bool* mxbool_ptr=NULL;

	if (mxIsClass(dataref,"logical")){
		if(mxGetM(dataref)!=1) _error_("input data is not of size 1x1");
		if(mxGetN(dataref)!=1) _error_("input data is not of size 1x1");
		mxbool_ptr=mxGetLogicals(dataref);
	}
	else{
		_error_("input data_type is not a bool!");
	}

	*pboolean=*mxbool_ptr;
}
/*}}}*/
