/*\file SystemMatrices.c
 *\brief: build system matrices (stiffness matrix, loads vector)
 */

#include "./SystemMatrices.h"

void mexFunction( int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[]){

	/*input datasets: */
	Elements   *elements   = NULL;
	Nodes      *nodes      = NULL;
	Vertices   *vertices   = NULL;
	Loads      *loads      = NULL;
	Materials  *materials  = NULL;
	Parameters *parameters = NULL;
	bool        kflag,pflag,penalty_kflag,penalty_pflag;
	
	/* output datasets: */
	Matrix*    Kff  = NULL;
	Matrix*    Kfs  = NULL;
	Vector*    pf   = NULL;
	Vector*    df   = NULL;

	double kmax;

	/*Boot module: */
	MODULEBOOT();

	/*checks on arguments on the matlab side: */
	if((nlhs!=NLHS) || (nrhs!=6 && nrhs!=10)){
		SystemMatricesUsage();
		_error_(" usage. See above");
	}

	/*Input datasets: */
	FetchData((DataSet**)&elements,ELEMENTS);
	FetchData((DataSet**)&nodes,NODES);
	FetchData((DataSet**)&vertices,VERTICES);
	FetchData((DataSet**)&loads,LOADS);
	FetchData((DataSet**)&materials,MATERIALS);
	FetchData(&parameters,PARAMETERS);

	/*configure: */
	elements->  Configure(elements,loads, nodes,vertices, materials,parameters);
	nodes->     Configure(elements,loads, nodes,vertices, materials,parameters);
	loads->     Configure(elements, loads, nodes,vertices, materials,parameters);
	materials-> Configure(elements, loads, nodes,vertices, materials,parameters);

	/*!Generate internal degree of freedom numbers: */
	if(nrhs==10){
		FetchData(&kflag,KFLAG);
		FetchData(&pflag,PFLAG);
		FetchData(&penalty_kflag,PENALTYKFLAG);
		FetchData(&penalty_pflag,PENALTYPFLAG);
		SystemMatricesx(&Kff,&Kfs,&pf,&df,&kmax,elements,nodes,vertices,loads,materials,parameters,kflag,pflag,penalty_kflag,penalty_pflag);
	}
	else
	 SystemMatricesx(&Kff,&Kfs,&pf,&df,&kmax,elements,nodes,vertices,loads,materials,parameters);

	/*write output datasets: */
	WriteData(KFF,Kff);
	WriteData(KFS,Kfs);
	WriteData(PF,pf);
	WriteData(DF,df);
	WriteData(KMAX,kmax);
	
	/*Free ressources: */
	delete elements;
	delete nodes;
	delete vertices;
	delete loads;
	delete materials;
	delete parameters;
	xdelete(&Kff);
	xdelete(&Kfs);
	xdelete(&pf);
	xdelete(&df);

	/*end module: */
	MODULEEND();
}

void SystemMatricesUsage(void)
{
	_printf_(true,"\n");
	_printf_(true,"   usage: [Kff,Kfs,pf,df,kmax] = %s(elements,nodes,vertices,loads,materials,parameters);\n",__FUNCT__);
	_printf_(true,"   usage: [Kff,Kfs,pf,df,kmax] = %s(elements,nodes,vertices,loads,materials,parameters,kflag,pflag,penalty_kflag,penalty_pflag);\n",__FUNCT__);
	_printf_(true,"\n");
}
