/*\file KMLFileRead.c
 *\brief: KML file reader module.
 */
#include "./KMLFileRead.h"

void mexFunction( int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[]){

	int i,j,nnodes=0,verbose=1;
	KML_Object* kobj;

	/*input: */
	char*   name=NULL;

	char*   notes=NULL;
	const mxArray* notesi;
	mwIndex        nindex;

	int*    elem=NULL;
	int     melem=0,nelem=0;

	int*    nodecon=NULL;
	int     mncon=0,nncon=0;

	double* lat=NULL;
	int     mlat=0,nlat=0,llat=0;

	double* lng=NULL;
	int     mlng=0,nlng=0,llng=0;

	int     nparts=0;

	int*    part=NULL;
	int     mprt=0,nprt=0,lprt=0;

	double* data=NULL;
	int     mdata=0,ndata=0;

	double* cmap=NULL;
	int     mcmap=0,ncmap=0;

	char*   filnam=NULL;

	FILE*   fidi=NULL;
	FILE*   fido=NULL;

	Options* options=NULL;
	char*    echo    =NULL;
	char*    deepecho=NULL;
	char*    write   =NULL;

	/* output: */
	int     ierror=0;

	/*Boot module: */
	MODULEBOOT();

	/*checks on arguments on the matlab side: */
	if (nlhs > NLHS) {
		KMLFileReadUsage();
		_error_("KMLFileRead usage error");
	}
	if (nrhs < NRHS) {
		KMLFileReadUsage();
		_error_("KMLFileRead usage error");
	}

	/*Input datasets: */
	if (verbose) printf("Fetching inputs:\n");
	FetchData(&filnam,FILENAME);
	if (verbose) printf("  filnam =\"%s\"\n",filnam);

	if (verbose) printf("Parsing options:\n");
	options=new Options(NRHS,nrhs,prhs);
	if (options->Size()) for(i=0;i<options->Size();i++) ((Option*)options->GetObjectByOffset(i))->DeepEcho();
	options->Get(&echo    ,"echo"    ,"off");
	options->Get(&deepecho,"deepecho","off");
	options->Get(&write   ,"write"   ,"off");

	/*some checks*/
	if (verbose) printf("Checking inputs:\n");

	if (!strlen(filnam))
		strcpy(filnam,"stdout");

	if (verbose) printf("Opening file \"%s\".\n",filnam);
	fidi=fopen(filnam,"r");

	/* Run core computations: */
	if (verbose) printf("Calling core:\n");
	kobj=KMLFileReadx(fidi);

	if (verbose) printf("Closing file \"%s\".\n",filnam);
	fclose(fidi);

	if (kobj) {
		if (!strncmp(echo    ,"on",2) || !strncmp(echo    ,"y",1))
			kobj->Echo();
		if (!strncmp(deepecho,"on",2) || !strncmp(deepecho,"y",1))
			kobj->DeepEcho();
		if (strncmp(write,"off",3) && strncmp(write,"no",2)) {
			if (!strncmp(write,"on",2) || !strncmp(write,"yes",3) || !strncmp(write,"stdout",6)) {
				kobj->Write(stdout,"");
			}
			else {
				if (verbose) printf("Opening file \"%s\".\n",write);
				fido=fopen(write,"w");
				kobj->Write(fido,"");
				if (verbose) printf("Closing file \"%s\".\n",write);
				ierror=fclose(fido);
			}
		}

		delete kobj;
	}

	/*Write data: */
	WriteData(ERRORFLAG,ierror);

	/*Clean-up*/
	xfree((void**)&write);
	xfree((void**)&deepecho);
	xfree((void**)&echo);
	delete options;

	/*end module: */
	MODULEEND();
}

void KMLFileReadUsage(void)
{
	_printf_(true,"KMLFileRead - KML file reader module:\n");
	_printf_(true,"\n");
	_printf_(true,"   This module reads a KML file.\n");
	_printf_(true,"\n");
	_printf_(true,"   Usage:\n");
	_printf_(true,"      [ierror]=KMLFileRead(kmlfile,'param name',param,...);\n");
	_printf_(true,"\n");
	_printf_(true,"      kmlfile      file name of kml file to be read (char)\n");
	_printf_(true,"\n");
	_printf_(true,"      echo         echo command (char, optional, 'off'/'on')\n");
	_printf_(true,"      deepecho     deep echo command (char, optional, 'off'/'on')\n");
	_printf_(true,"      write        write command (char, optional, 'off'/'stdout'/kmlfile)\n");
	_printf_(true,"\n");
	_printf_(true,"      ierror       return code (non-zero for error)\n");
	_printf_(true,"\n");
	_printf_(true,"   Examples:\n");
	_printf_(true,"      [ierror]=KMLFileRead('file.kml','deepecho','on');\n");
	_printf_(true,"      [ierror]=KMLFileRead('filin.kml','echo','on','write','filout.kml');\n");
	_printf_(true,"\n");
}

