#include "rundeck_opts.h"
      MODULE TES_SIMULATOR
!@sum  Routines for the Tropospheric Emission Spectrometer (TES) 
!@sum  'water simulator'
!@auth Robert Field
!@ver  1.1
C********************************************************************** 
C********************************************************************** 
C
C                   TES Water Simulator
C  
C The TES Water Simulator produces H2O and HDO vapor diagnostics 
C suitable for comparison with retrievals from the TES instrument.
C This is done by excluding model profiles for which TES would have low
C sensitivity, and for those remaining, applying a TES operator to 
C account for TES's limited vertical resolution and the contributions
C from a prior profile.
C 
C It can operate in two ways:
C  
C 1) Retrieval-based: 
C    Averaging kernels are extracted directly from individual high
C    quality TES retrievals along the satellite path, and applied
C    to the model profiles. This should only be used for nudged
C    simulations. 
C
C 2) Category-based:
C    Averaging kernels are selected from 'climatogolgies', stratified
C    across cloud characteristics, surface temperature and
C    moisture. This can be used for either nudged or free-running
C    simulations. There is an option to use categorical selection
C    but with collocation, sampling the model only along the satellite
C    path.
C
C                   Configuration
C
C To turn it on in ModelE:
C
C  - turn the water and isotope tracers on
C  - turn the ISCCP diagnostics on
C  - include #define TES_SIM in pre-processor options 
C  - include TES_SIMULATOR in 'Object modules'
C
C  - in 'Data input files:'
C      Set the mnemonic 'TES_AK', a netcdf file with categorical 
C      averaging kernel info , located in 
C      /discover/nobackup/projects/giss/prod_input_files/
C       
C      Different files with the suffix *.AvgKrnls.nc have been placed
C      there. The names of the files indicate the type of 
C      categorization. The categorical variable 'bins' for a file can
C      viewed with the command 'ncdump <fileName> -v <varName> where 
C      varName is one of:
C          srfTmpCats: surface temperature
C          optDepCats: cloud optical depth
C          cldTopPresCats: cloud top pressure
C          pwLowCats: pcp water near the surface
C          pwHighCats: pcp water in the free atmosphere
    
C      These are generated by the MATLAB program 
C      'processRawTESData_***.m'. This can be found in
C      /home/rfield1/matlab/TES/ 

C  - In '&&PARAMETERS', set the following switches:
C
C      TESObsDiagnosticsOn: Set to 1 to use TES diagnostics with avg
C      kernels and quality extracted directly from the measurements. 
C      This should only be used for nudged runs, but there's no
C      internal check.
C
C      TESObsMinDegFr: When checking quality of TES retrievals,
C      require this minimum degrees of freedom to be included.
C      A standard value is 0.5.
C
C      TESObsDataDir: Directory containing the daily TES files. On
C      discover, use:
C      '/discover/nobackup/projects/giss/OBS/TES/v4L2/out/v14/netcdf/'
C      These were also produced using 'processRawTESData_***.m'
C
C      TESCatDiagnosticsOn: Set to 1 to use TES diagnostics with 
C      avg krnls selected from categories. The simulator will use the 
C      file specified by TES_AK above.
C
C      TESCatMinQuality: Minimum % good quality data for profiles in a
C      category to be included in categorical averaging kernel 
C      selection. The categorical quality is stored in the TES_AK file
C      specified above. Those numbers are compared against this
C      threshold. 70 seems reasonable.
C
C      TESCatUseColloc: Set to 1 to use categorical avg krnl selection,
C      but only along the satellite path. This requires that 
C      all the TESObs switches above be set. Note too that the 
C      collocation here includes poor quality observations, unlike for
C      the 'TESObs' case. The quality filtering is done by comparing 
C      the overall quality for a profile's corresponding category to 
C      TESCatMinQuality.
C
C                   Output
C
C The module will produce the following 'aijl' diagnostics:
C
C For categorical avg kernel selection (_CAT)
C  - H2O_DUP_CAT, a duplicate H2O tracer for categorical mode
C  - HDO_DUP_CAT, a duplicate HDO tracer for categorical mode
C  - H2O_TES_CAT, TES-style H2O on a truncated vertical range for 
C    categorical mode
C  - HDO_TES_CAT, TES-style HDO on a truncated vertical range for 
C    categorical mode
C  - H2O_TES_OLD_CAT, H2O after vertical interpolation only, on
C   truncated vertical levels
C  - HDO_TES_OLD_CAT, HDO after vertical interpolation only, on 
C    truncated vertical levels
C
C For avg krnl selection directly from retrievals (_RETR)
C  - H2O_DUP_RETR, a duplicate H2O tracer for retrieval mode
C  - HDO_DUP_RETR, a duplicate HDO tracer for retrieval mode
C  - H2O_TES_RETR, TES-style H2O on a truncated vertical range for 
C    retrieval mode
C  - HDO_TES_RETR, TES-style HDO on a truncated vertical range for 
C    retrieval mode
C  - H2O_TES_OLD_RETR, H2O after vertical interpolation only, on
C    truncated vertical levels, for retrieval mode
C  - HDO_TES_OLD_RETR, HDO after vertical interpolation only, on
C    truncated vertical levels,for retrieval mode
C
C It will also produce the following 'aij' diagnostics:
C  - ISCCP_subgrid_ttl, total number of subgrid columns in ISCCP
C  - ISCCP_subgrid_good, number of subgrid columns with 'good' 
C    conditions for TES (i.e. those whose categorical quality in the 
C    observations is > TESCatMinQuality)
C  - TES_meas_ttl, total number of TES measurements
C  - TES_meas_good, number of good TES measurements
C  - TES_PWHIGH, precipitable water in the free atmosphere
C  - TES_PWLOW, precipitable water near the surface
C
C Of these, H2O_TES_CAT,HDO_TES_CAT,H2O_TES_RETR and HDO_TES_RETR are
C of primary interest. The others are for examining the effects of the 
C data quality filtering, and for partitioning of the effects of the
C TES operator between vertical interpolation and averaging kernel 
C convolution.
C
C********************************************************************** 
C**********************************************************************


C**********************************************************************
C**** Variable declarations specific to categorical AK selection.
C**********************************************************************

!@var TESCatDiagnosticsOn to enable categorical averaging kernel selection
!@var TESCatUseColloc to sample only along TES path
      INTEGER, public :: TESCatDiagnosticsOn,TESCatUseColloc
!@var TESCatMinQuality minimum data quality for a TES category
      REAL*8, public :: TESCatMinQuality

C****
C**** These are read in from the TESAvgKrnlFileName file in 
C**** the subroutine ReadTESAvgKrnlFile.
C****
!@var nTESLevs number of TES pressure levels
!@var nDataLevs number of usable TES levels for HDO
!@var nAvgKrnlLevs number of avg krnl levels required for TES operator
!@var nPriorLevs number of levels in prior
!@var nsrfTmpCats number of surface temperature categories
!@var noptDepCats number of cloud optical depth categories
!@var ncldTopPresCats number of cloud top pressure categories
!@var npwLowCats number of surface precipitable water categories
!@var npwHighCats number of free atmophsere pw categories
      INTEGER, public :: nTESLevs,nDataLevs,nAvgKrnlLevs,nPriorLevs
      INTEGER, public :: nsrfTmpCats,noptDepCats,ncldTopPresCats
      INTEGER, public :: npwLowCats,npwHighCats

!@var TESPres 1D vector of all (67) pressure levels
!@var TESDataLevs levels in TESPres over which avg krnls are applied
!@var TESAvgKrnlLevs levels in TESPres over which avg krnls are needed
      REAL*8, ALLOCATABLE, DIMENSION(:), public :: TESPres
      INTEGER, ALLOCATABLE, DIMENSION(:), public :: TESDataLevs
      INTEGER, ALLOCATABLE, DIMENSION(:), public :: TESAvgKrnlLevs
	
!@var srfTmpCats surface temperature categories 
!@var optDepCats optical depth categories 
!@var cldTopPresCats cloud top pressure categories 
!@var pwLowCats surface precipitable watercategories 
!@var pwHighCats free atmosphere pw categories 
      REAL*8, ALLOCATABLE, DIMENSION(:), public :: srfTmpCats,optDepCats
      REAL*8, ALLOCATABLE, DIMENSION(:), public :: cldTopPresCats
      REAL*8, ALLOCATABLE, DIMENSION(:), public :: pwLowCats,pwHighCats
!@var rPrior 1D vector of TES prior isotopic ratio
      REAL*8, ALLOCATABLE, DIMENSION(:), public :: rPrior

!@var avgKrnlH2O mean categorical H2O averaging kernels 
!@var avgKrnlHDO mean categorical HDO averaging kernels 
!@var avgKrnlHD mean categorical HD averaging kernels 
!@var avgKrnlDH mean categorical DH averaging kernels
!@var qPriors mean categorical q priors
!@var percGoodObs percent of high quality categorical TES measurements
C These arrays have one dimension for each categorical variable (5),
C one dimension for the TESDataLevs, and one for the TESAvgKrnlLevs
C The qprior array is similar, but is has one dimension on TESPres.
C percGoodObs is similar, but with no vertical dimensions
      REAL*8,ALLOCATABLE,DIMENSION(:,:,:,:,:,:,:),public :: avgKrnlsH2O
      REAL*8,ALLOCATABLE,DIMENSION(:,:,:,:,:,:,:),public :: avgKrnlsHDO
      REAL*8,ALLOCATABLE,DIMENSION(:,:,:,:,:,:,:),public :: avgKrnlsHD
      REAL*8,ALLOCATABLE,DIMENSION(:,:,:,:,:,:,:),public :: avgKrnlsDH
      REAL*8,ALLOCATABLE,DIMENSION(:,:,:,:,:,:),public :: qPriors
      REAL*8,ALLOCATABLE,DIMENSION(:,:,:,:,:),public::percGoodObs
!@var latBdy latitude boundary of TES domain
!@var bdyLyrDep boundary layer depth separating pwLow and pwHigh
      REAL*8, public:: latBdy,bdyLyrDep

C**********************************************************************
C**** Variable declarations specific to  AK selection from retrievals
C**********************************************************************

      INTEGER, public :: TESObsDiagnosticsOn
!@var TESObsDataDir location of daily TES data files
      CHARACTER(LEN=256),public ::  TESObsDataDir

!@var maxObsPerCell twice the number of TES obs in a cell ever seen
      INTEGER, PARAMETER :: maxObsPerCell =10
      INTEGER,public :: timeStrLen
      INTEGER, public :: nTESLevsPt,nDataLevsPt,nAvgKrnlLevsPt
      
!@var TESObsMinDegFr minimum deg. of freedom for a retrieval 
      REAL*8, public :: TESObsMinDegFr

!@var TESPresPt 1D vector of all (67) pressure levels
!@var TESDataLevsPt levels in TESPres over which Avg Kernels are applied
!@var TESAvgKrnlLevsPT levels in TESPres over which Avg Kernels are used
      REAL*8, ALLOCATABLE, DIMENSION(:), public :: TESPresPt
      INTEGER, ALLOCATABLE, DIMENSION(:), public :: TESDataLevsPt
      INTEGER, ALLOCATABLE, DIMENSION(:), public :: TESAvgKrnlLevsPt

C****
C**** Pointwise variables, first dimension is always # of daily obs
C****
!@var avgKrnlsH2OPt H2O averaging kernels
!@var avgKrnlsHDOPt HDO averaging kernels
!@var avgKrnlsHDPt HD averaging kernels
!@var avgKrnlsDHPt DH averaging kernels
      REAL*8,ALLOCATABLE,DIMENSION(:,:,:),public :: avgKrnlsH2OPt
      REAL*8,ALLOCATABLE,DIMENSION(:,:,:),public :: avgKrnlsHDOPt
      REAL*8,ALLOCATABLE,DIMENSION(:,:,:),public :: avgKrnlsHDPt
      REAL*8,ALLOCATABLE,DIMENSION(:,:,:),public :: avgKrnlsDHPt
!@var qPriorPt q priors
!@var dDPriorPt dD priors (even though they're invariant)
      REAL*8,ALLOCATABLE,DIMENSION(:,:),public :: qPriorPt,dDPriorPt
!@var secondsPt number of seconds since midnight
      INTEGER,ALLOCATABLE,DIMENSION(:),public :: secondsPt
!@var dayTimeFlagPt day/night flag
      INTEGER,ALLOCATABLE,DIMENSION(:),public :: dayTimeFlagPt
!@var latVecPt latitude of TES measurement
!@var lonVecPt longitude of TES measurement
      REAL*8,ALLOCATABLE,DIMENSION(:),public :: latVecPt,lonVecPt
!@var latIndPt J index on model grid of measurements
!@var lonIndPt I index on model grid of measurements
      INTEGER,ALLOCATABLE,DIMENSION(:),public :: latIndPt,lonIndPt
!@var H2ODegFrPt retrieval degOfFr for H2O
!@var HDODegFrPt retrieval degOfFr for HDO
      REAL*8,ALLOCATABLE,DIMENSION(:),public :: H2ODegFrPt,HDODegFrPt
!@var H2OQualPt H2O quality flag
!@var HDOQualPt HDO quality flag
      INTEGER,ALLOCATABLE,DIMENSION(:),public :: H2OQualPt,HDOQualPt

C****
C**** Gridded variables: the point data above get mapped onto these
C****
!@var nObsGrid number of daily TES measurements on grid
      INTEGER,ALLOCATABLE,DIMENSION(:,:),public :: nObsGrid
!@var goodObFlag 1 if good quality, 0 if not
      INTEGER,ALLOCATABLE,DIMENSION(:,:,:),public:: goodObFlag
!@var secOfDayGrid seconds since midnight
      INTEGER,ALLOCATABLE,DIMENSION(:,:,:),public::
     & secOfDayGrid
!@var avgKrnlH2OGrid H2O averaging kernels mapped onto grid
!@var avgKrnlHDOGrid HDO averaging kernels mapped onto grid 
!@var avgKrnlHDGrid HD averaging kernels mapped onto grid 
!@var avgKrnlDHGrid DH averaging kernels mapped onto grid
C The dimensions here are I,J,nObs,nDataLev,nAvgKrnlLev
      REAL*8,ALLOCATABLE,DIMENSION(:,:,:,:,:),public::
     & avgKrnlsH2OGrid
      REAL*8,ALLOCATABLE,DIMENSION(:,:,:,:,:),public::
     & avgKrnlsHDOGrid
      REAL*8,ALLOCATABLE,DIMENSION(:,:,:,:,:),public::
     & avgKrnlsHDGrid
      REAL*8,ALLOCATABLE,DIMENSION(:,:,:,:,:),public::
     & avgKrnlsDHGrid
!@var qPriorsGrid q priors mapped onto grid
!@var dDPriorsGrid dD priors mapped onto grid (invariant)
      REAL*8,ALLOCATABLE,DIMENSION(:,:,:,:),public::
     & qPriorsGrid
      REAL*8,ALLOCATABLE,DIMENSION(:,:,:,:),public::
     & dDPriorsGrid

      END MODULE TES_SIM

C********************************************************************** 
C********************************************************************** 
C
C                   GISS ModelE-specific routines
C
C********************************************************************** 
C**********************************************************************

      SUBROUTINE ApplyTESOpCategoricalAK(J_0,J_1,I_0,I_1)	  
C********************************************************************** 
C Description: This applies the categorical TES operator to H2O and 
C HDO model profiles. It is currently called in TRACEA, but in theory
C could appear anywhere after the model physics is completed.
C
C This is done only within a latitude boundary defined by latBdy 
C and during daytime, for compatibility with the ISCCP simulator.
C The steps are to:
C    - convert profiles to vmr from mass-mixing ratio
C    - calculate the surface radiative temperature with fractional
C      contributions from the different surface types
C    - calculate preciptable water near surface and in free atmosphere
C    - for each ISCCP subgrid column, 
C       - retrieve the COD and CTP
C  	- along with tsurf and low/high precip water, see if this
C         category is of good enough retrieval quality
C   	- if it is, get the appropriate averaging kernels and
C         q prior, apply TES operator and accumulate new profiles
C
C Note that TES diagnostics have been defined in the aijl arrays, 
C rather than the taijl arrays. The taijl arrays would be a more 
C logical place, but don't have 'denom' arrays and so don't allow
C for conditional accumulation.
C********************************************************************** 
      USE CONSTANT, only : undef	  
      USE DYNAMICS, only: pmid,am
      USE GEOM, only: axyp,LAT_DG
      USE MODEL_COM, only: im,jm,lm ,q,focean,flice,
     *                     JDAY,JMON,JDATE,JYEAR,JHOUR,
     *                     itime,nday,ItimeI,IYEAR1,JDPERY,
     *                     NDAY,DTSrc
      USE SEAICE_COM, only : rsi
      USE LAKES_COM, only : flake
      USE GHY_COM, only : fearth
      USE FLUXES, only : GTEMP,GTEMPR
      USE CLOUDS_COM, only: ncol
 
      USE TRACER_COM, only : trm,n_HDO,n_water
      USE DIAG_COM, only: aijl=>aijl_loc, aij=>aij_loc
     *     ,IJL_DP_DUP,IJL_H2O_DUP,IJL_HDO_DUP
     *     ,IJL_DP_TES,IJL_H2O_TES,IJL_HDO_TES
     *     ,IJL_H2O_TES_OLD,IJL_HDO_TES_OLD
     *     ,IJ_TES_TTL,IJ_TES_GOOD
     *     ,IJ_TES_PWL,IJ_TES_PWH
     *     ,IJ_TES_SUNL
     *     ,saveSCLDI
     *     ,saveBOXTAU, saveBOXCTP
     *     ,scale_ijl
      USE TES_SIMULATOR
      
      IMPLICIT NONE      
C***** INPUT VARIABLES
      INTEGER,INTENT(IN) :: I_0,I_1,J_0,J_1

C**** INTERNAL VARIABLES
      REAL*8, DIMENSION(nTESLevs) ::currQPrior
      REAL*8, DIMENSION(nDataLevs,nAvgKrnlLevs) :: currAKH2O
      REAL*8, DIMENSION(nDataLevs,nAvgKrnlLevs) :: currAKHDO
      REAL*8, DIMENSION(nDataLevs,nAvgKrnlLevs) :: currAKHD
      REAL*8, DIMENSION(nDataLevs,nAvgKrnlLevs) :: currAKDH
      REAL*8, DIMENSION(lm) :: h2oModelSH,hdoModelSH
      REAL*8, DIMENSION(lm) :: h2oModelNewSH,hdoModelNewSH
      REAL*8, DIMENSION(lm) :: h2oModelIntpSH,hdoModelIntpSH
      INTEGER :: code,currCol,I,J,L
      REAL*8,DIMENSION(lm) :: dDOld
      REAL*8,DIMENSION(lm) :: dDNew
      REAL*8 POICE,PEARTH,PLANDI,POCEAN,currTgrnd
      REAL*8 :: pcpWLow,pcpWHigh
      REAL*8 :: currCOD, currCTP
      INTEGER :: currSec,currSecITU,minSecITU,maxSecITU
      INTEGER :: currQualFlag,collocOK
      INTEGER :: tgrndI,pwLowI,pwHighI,codI,ctpI
      REAL*8 :: currPercGood,currNObs,currOb

C**** Loop over local domain
      do j=J_0,J_1
          do i=I_0,I_1
C****         Option to continue only if there was a good 
C****         TES measurement during the current time. We don't care
C****         about the actual AKs, just whether the satellite was 
C****         there.
              collocOK = 1
              IF (TESCatUseColloc.eq.1) THEN
              collocOK = 0
              if (ALLOCATED(nObsGrid)) then
                  currNObs = nObsGrid(I,J)
                  if (currNObs .gt. 0) then

C****             See if any of the measurements in the cell occurred
C****             during the current time step. Unlike
C****             ApplyTESOpRetrievalAK, there's no check on the 
C****             measurement quality - that's part of what we're
C****             tring to model with the categories.
                  do currOb=1,currNObs

C****                 Convert seconds-since-00z of measurement to
C****                 model's internal time units
                      currSec= secOfDayGrid(i,j,currOb)
                      currSecITU = (JYEAR-IYEAR1)*JDPERY + JDAY-1
                      currSecITU = currSecITU*NDay*DTSrc
                      currSecITU = currSecITU + currSec
                      
C****                 The current time range of interest
                      minSecITU=ITIME*DTSrc
                      maxSecITU=(ITIME+1)*DTSrc

                      if (currSecITU >= minSecITU .and.
     &                    currSecITU < maxSecITU) then
                          collocOK = 1
                      end if
                  enddo
                  end if
              end if
              END IF

C****         Analysis is restricted to region within latBdy 
C****         (defined in AK files, for now tropical) during daytime
              if (ABS(LAT_DG(j,1)).le.latBdy .and.
     &            saveSCLDI(i,j).eq.1 .and. 
     &             collocOK.eq.1) then

                  aij(i,j,IJ_TES_SUNL) = aij(i,j,IJ_TES_SUNL) + 1.               

C****             Retrieve model profiles of H2O and HDO moist mmr 
C****             (i.e. specific humidity)
                  do L=1,LM
C****                 Use of prognostic q or passive trm water produce
C****                 identical results. So does including water
C****                 in denominator of h2o and hdo mmr.
                      h2oModelSH(l)=q(i,j,l)
                      hdoModelSH(l)=trm(i,j,l,n_HDO)/
     *                         (am(l,i,j)*axyp(i,j))
C     *                         (trm(i,j,l,n_Water)
C     *                         + am(l,i,j)*axyp(i,j))

C****                 Store un-modified profiles to accumulation 
C****                 diagnostics, for comparison
                      aijl(i,j,L,IJL_H2O_DUP)=
     *                     aijl(i,j,L,IJL_H2O_DUP)+
     *                     h2oModelSH(l)
                      aijl(i,j,L,IJL_HDO_DUP)=
     *                     aijl(i,j,L,IJL_HDO_DUP)+
     *                     hdoModelSH(l)
                      aijl(i,j,L,IJL_DP_DUP)=
     *                     aijl(i,j,L,IJL_DP_DUP) + 1.
                  enddo

C****             Compute ground radiative temp as in DIAG.f around
C****             L1863 - should just save retrieve that value, 
C****             rather than recalculate.
                  POCEAN=(1.-RSI(I,J))*(FOCEAN(I,J)+FLAKE(I,J))
                  POICE=RSI(I,J)*(FOCEAN(I,J)+FLAKE(I,J))
                  PEARTH=FEARTH(I,J)
                  PLANDI=FLICE(I,J)
                  currTgrnd=
     *                POCEAN*GTEMPR(1,I,J)+
     *                POICE*GTEMPR(2,I,J)+
     *                PLANDI*GTEMPR(3,I,J)+
     *                PEARTH*GTEMPR(4,I,J)

C****             Calculate precipitable water near surface and in 
C****             free atm.
                  CALL CalcPcpWat(lm,pmid(:,i,j),h2oModelSH,
     &                bdyLyrDep,pcpWLow,pcpWHigh)
                  aij(i,j,IJ_TES_PWL) = aij(i,j,IJ_TES_PWL) + pcpWLow
                  aij(i,j,IJ_TES_PWH) = aij(i,j,IJ_TES_PWH) + pcpWHigh

C****             For each ISCCP subgrid column, retrieve and apply 
C****             appropriate averaging kernel, accumulate results
C****             over 'good' profiles.
                  do currCol=1,ncol
 	              aij(i,j,IJ_TES_TTL)=aij(i,j,IJ_TES_TTL)+1.

C****                 Get COD and CTP for current subgrid column
                      currCOD = saveBOXTAU(i,j,currCol)
                      currCTP = saveBOXCTP(i,j,currCol)

C****                 Retrieve the categorical indices corresponding 
C****                 to current cod,ctp,tsurf, and pw values.
                      CALL RetrieveAKIndices(currCOD,currCTP,currTgrnd
     *                     ,pcpWLow,pcpWHigh
     *                     ,noptDepCats,ncldTopPresCats,nsrfTmpCats
     *                     ,npwLowCats,npwHighCats
     *                     ,optDepCats,cldTopPresCats,srfTmpCats
     *                     ,pwLowCats,pwHighCats	 
     *                     ,codI,ctpI,tgrndI
     *                     ,pwLowI,pwHighI)

C****                 We're only interested in categories for which 
C****                 TES measurements are of some sufficient quality.
C****                 This attempts to model the sensitivity thresholds
C****                 and quality flags used to filter the TES data.
                      currPercGood = percGoodObs(tgrndI,codI,ctpI
     *                     ,pwLowI,pwHighI)

                      if (currPercGood .ge. TESCatMinQuality) then
                          aij(i,j,IJ_TES_GOOD)=aij(i,j,IJ_TES_GOOD)+1.
C****                     Retrieve AKs and Q prior for current set of 
C****                     indices
	                      currAKH2O = avgKrnlsH2O(tgrndI,codI,ctpI,
     &                                pwLowI,pwHighI,:,:)
	                      currAKHDO = avgKrnlsHDO(tgrndI,codI,ctpI,
     &                                pwLowI,pwHighI,:,:)
	                      currAKHD = avgKrnlsHD(tgrndI,codI,ctpI,
     &                                pwLowI,pwHighI,:,:)
	                      currAKDH = avgKrnlsDH(tgrndI,codI,ctpI,
     &                                pwLowI,pwHighI,:,:)
                              currQPrior = qPriors(tgrndI,codI,ctpI,
     &                                pwLowI,pwHighI,:)

C****                 Apply the TES operator to profiles with 
C****                 current AKs and prior
                      CALL ApplyTESOperator(nTESLevs,TESPres
     &                        ,h2oModelSH,hdoModelSH,pmid(:,i,j)
     &                        ,currQPrior,rPrior
     &                        ,nDataLevs,nAvgKrnlLevs
     &                        ,TESDataLevs,TESAvgKrnlLevs
     &                        ,currAKH2O,currAKHDO
     &                        ,currAKHD,currAKDH
     &                        ,h2oModelNewSH,hdoModelNewSH
     &                        ,h2oModelIntpSH,hdoModelIntpSH)

C****                     Accumulate new convolved profiles, and also 
C****                     interpolation-only profiles. The range of
C****                     valid values is limited by nDataLevs,so need
C****                     to check for undefs in the new profiles.
                          do L=1,LM 
                          if (h2oModelNewSH(L).ne.undef
     &                      .and. hdoModelNewSH(L).ne.undef) then
                            aijl(i,j,L,IJL_H2O_TES)=
     *                        aijl(i,j,L,IJL_H2O_TES)+h2oModelNewSH(L)
                            aijl(i,j,L,IJL_HDO_TES)=
     *                        aijl(i,j,L,IJL_HDO_TES)+hdoModelNewSH(L)
                            aijl(i,j,L,IJL_H2O_TES_OLD)=
     *                        aijl(i,j,L,IJL_H2O_TES_OLD)
     *                        + h2oModelIntpSH(L)
                            aijl(i,j,L,IJL_HDO_TES_OLD)=
     *                        aijl(i,j,L,IJL_HDO_TES_OLD)
     *                        + hdoModelIntpSH(L)
    	                    aijl(i,j,L,IJL_DP_TES)=
     *                        aijl(i,j,L,IJL_DP_TES) + 1.
                          endif 
                          enddo
                      endif
                  enddo 
              endif				   
          enddo
      enddo

      END SUBROUTINE ApplyTESOpCategoricalAK

      SUBROUTINE ApplyTESOpRetrievalAK(J_0,J_1,I_0,I_1)
C**********************************************************************
C Description: For each cell populated with TES observations,
C apply the AKs and priors from the actual, individual retrievals to
C the modeled profile, store in diagnostic arrays.
C
C Specifically:
C    For each (i,j) on the local grid
C        - If the averaging kernel count at (i,j) >=1 
C            - Extract the model's H2O and HDO profiles from (i,j)
C            - Convert from model levels to TES pressure levels
C            - Do any necessary unit conversions
C            - For each good TES observation in that cell
C                - apply the avg krnls to the modeled
C                  H2O and HDO profiles
C                - convert back to mmr, model levels
C                - accumulate the new profiles
C**********************************************************************

      USE CONSTANT, only : undef	  
      USE DYNAMICS, only: pmid,am
      USE GEOM, only: axyp,LAT_DG
      USE MODEL_COM, only: im,jm,lm,q,JDAY,JMON,JDATE,JYEAR,JHOUR,
     *                     itime,nday,ItimeI,IYEAR1,JDPERY,
     *                     NDAY,DTSrc
      USE DOMAIN_DECOMP_1D, only: am_i_root
      USE TRACER_COM, only : trm,n_HDO,n_water
      USE DIAG_COM, only: aijl=>aijl_loc, aij=>aij_loc
     *     ,IJL_DP_DUP_R,IJL_H2O_DUP_R,IJL_HDO_DUP_R
     *     ,IJL_DP_TES_R,IJL_H2O_TES_R,IJL_HDO_TES_R
     *     ,IJL_H2O_TES_OLD_R,IJL_HDO_TES_OLD_R
     *     ,IJ_TES_TTL_R,IJ_TES_GOOD_R
     *     ,scale_ijl
     *     ,saveSCLDI
      USE TES_SIMULATOR
      
      IMPLICIT NONE      
C***** INPUT VARIABLES
!@var I_0,I_1,J_0,J_1 current array bounds
      INTEGER,INTENT(IN) :: I_0,I_1,J_0,J_1

C**** INTERNAL VARIABLES
      REAL*8, DIMENSION(nTESLevs) ::currQPrior
      REAL*8, DIMENSION(nDataLevs,nAvgKrnlLevs) :: currAKH2O
      REAL*8, DIMENSION(nDataLevs,nAvgKrnlLevs) :: currAKHDO
      REAL*8, DIMENSION(nDataLevs,nAvgKrnlLevs) :: currAKHD
      REAL*8, DIMENSION(nDataLevs,nAvgKrnlLevs) :: currAKDH
      REAL*8, DIMENSION(lm) :: h2oModelSH,hdoModelSH
      REAL*8, DIMENSION(lm) :: h2oModelNewSH,hdoModelNewSH
      REAL*8, DIMENSION(lm) :: h2oModelIntpSH,hdoModelIntpSH
      INTEGER :: code,currNObs,currOb,I,J,L,currQualFlag
      INTEGER :: currSec,currSecITU,minSecITU,maxSecITU
      REAL*8,DIMENSION(lm) :: dDOld
      REAL*8,DIMENSION(lm) :: dDNew
      REAL*8 :: currHourFrac

      IF (ALLOCATED(nObsGrid)) THEN
      do j=J_0,J_1
          do i=I_0,I_1

C****         For consistency with categorical AK selection, works only
C****         during the day.
              if (ABS(LAT_DG(j,1)).le.latBdy
     &        .and.saveSCLDI(i,j).eq.1) then

C****         Look for grid cells with measurements during
C****         current day
              currNObs = nObsGrid(I,J)
              
              if (currNObs.ge.1) then
C****             Retrieve profiles of H2O and HDO moist
C****             mmr (i.e. specific humidity)
                  do L=1,LM
C****                 Use of prognostic q or passive trm water produce
C****                 identical results. So does including water in
C****                 denominator of h2o and hdo mmr.
                      h2oModelSH(l)=q(i,j,l)
                      hdoModelSH(l)=trm(i,j,l,n_HDO)/
     *                         (am(l,i,j)*axyp(i,j))
C     *                         (trm(i,j,l,n_Water)
C     *                         + am(l,i,j)*axyp(i,j))

C****                 Store un-modified profiles to accumulation 
C****                 diagnostics, for comparison
                      aijl(i,j,L,IJL_H2O_DUP_R)=
     *                     aijl(i,j,L,IJL_H2O_DUP_R)+
     *                     h2oModelSH(l)
                      aijl(i,j,L,IJL_HDO_DUP_R)=
     *                     aijl(i,j,L,IJL_HDO_DUP_R)+
     *                     hdoModelSH(l)
                      aijl(i,j,L,IJL_DP_DUP_R)=
     *                     aijl(i,j,L,IJL_DP_DUP_R) + 1.
                  enddo

C****             For each good TES measurement during the current
C****             time step, apply its AK and prior in the
C****             convolution, accumulating the results.
                  do currOb=1,currNObs

C****                 Convert seconds-since-00z of measurement to
C****                 model's internal time units
                      currSec= secOfDayGrid(i,j,currOb)
                      currSecITU = (JYEAR-IYEAR1)*JDPERY + JDAY-1
                      currSecITU = currSecITU*NDay*DTSrc
                      currSecITU = currSecITU + currSec
                      
C****                 The current time range of interest
                      minSecITU=ITIME*DTSrc
                      maxSecITU=(ITIME+1)*DTSrc

                      currQualFlag = goodObFlag(i,j,currOb)

C****                 This should really nested in the next if 
C****                 statement
                      if (currSecITU >= minSecITU .and.
     &                    currSecITU < maxSecITU) then
                      aij(i,j,IJ_TES_TTL_R)=
     &                    aij(i,j,IJ_TES_TTL_R)+1.
                      end if

                      if (currSecITU >= minSecITU .and.
     &                    currSecITU < maxSecITU .and.
     &                    currQualFlag .eq. 1) then
                      aij(i,j,IJ_TES_GOOD_R)=
     &                    aij(i,j,IJ_TES_GOOD_R)+1.
  
C****                 Retrieve AKs and Q prior for current observation
	              currAKH2O = avgKrnlsH2OGrid(i,j,currOb,:,:)
	              currAKHDO = avgKrnlsHDOGrid(i,j,currOb,:,:)
	              currAKHD = avgKrnlsHDGrid(i,j,currOb,:,:)
	              currAKDH = avgKrnlsDHGrid(i,j,currOb,:,:)
	              currQPrior = qPriorsGrid(i,j,currOb,:)

C                     Should really get rPrior from individual measurements, rather
C                     than relying on categorical averaging kernel file. They're
C                     identical, but the categorical avg krnl file might not 
C                     always be used. 

C****                 Apply the TES operator to profiles with current
C****                 AKs and prior
                      CALL ApplyTESOperator(nTESLevs,TESPres
     &                        ,h2oModelSH,hdoModelSH,pmid(:,i,j)
     &                        ,currQPrior,rPrior
     &                        ,nDataLevs,nAvgKrnlLevs
     &                        ,TESDataLevs,TESAvgKrnlLevs
     &                        ,currAKH2O,currAKHDO
     &                        ,currAKHD,currAKDH
     &                        ,h2oModelNewSH,hdoModelNewSH
     &                        ,h2oModelIntpSH,hdoModelIntpSH)

C                          dDNew=1000*(hdoModelNewSH/
C     &                          (h2oModelNewSH*0.000329) - 1)

C****                 Accumulate new convolved profiles, and also 
C****                 interpolation-only profiles. The range of valid
C****                 values is limited by nDataLevs, so need to check
C****                 for undefs in the new profiles.
                      do L=1,LM 
                      if (h2oModelNewSH(L).ne.undef
     &                  .and. hdoModelNewSH(L).ne.undef) then
                        aijl(i,j,L,IJL_H2O_TES_R)=
     *                    aijl(i,j,L,IJL_H2O_TES_R)
     *                    +h2oModelNewSH(L)

                        aijl(i,j,L,IJL_HDO_TES_R)=
     *                    aijl(i,j,L,IJL_HDO_TES_R)
     *                    +hdoModelNewSH(L)

                        aijl(i,j,L,IJL_H2O_TES_OLD_R)=
     *                    aijl(i,j,L,IJL_H2O_TES_OLD_R)
     *                    + h2oModelIntpSH(L)

                        aijl(i,j,L,IJL_HDO_TES_OLD_R)=
     *                    aijl(i,j,L,IJL_HDO_TES_OLD_R)
     *                    + hdoModelIntpSH(L)

    	                aijl(i,j,L,IJL_DP_TES_R)=
     *                    aijl(i,j,L,IJL_DP_TES_R) + 1.
                       endif 
                       enddo
                  endif
                  enddo 
                  endif
              endif 				   
          enddo 
      enddo
      END IF

      END SUBROUTINE ApplyTESOpRetrievalAK
      
      SUBROUTINE ReadDailyTESFile(currYear,currMonth,currDay,localNObs)
C********************************************************************** 
C Description: Read in point-wise data from daily TES netcdf file.
C********************************************************************** 
      USE TES_SIMULATOR

      IMPLICIT NONE
      include 'netcdf.inc'
C 
C                               INPUT:
C
      INTEGER, INTENT(IN) :: currYear,currMonth,currDay
C 
C                               OUTPUT:
C 
      INTEGER, INTENT(OUT) :: localNObs
C 
C                             LOCAL VARIABLES
C
      INTEGER :: netcdfID,status,code
      character(len=4) :: yearStr
      character(len=4) :: monthStr
      character(len=4) :: dayStr
      character(len=16) :: dateStr
      character(len=256) :: TESFilePath

!@var netcdf variable for dimension IDs
      INTEGER :: nLocationsID,nTESLevsID,nDataLevsID,nAvgKrnlLevsID
      INTEGER :: timeStrLenID

!@var netcdf variable for variables IDs
      INTEGER :: TESPresLevID,TESDataLevID,TESAvgKrnlLevID      
      INTEGER :: avgKrnlH2OID,avgKrnlHDOID,avgKrnlHDID,avgKrnlDHID
      INTEGER :: qPriorID,dDPriorID
      INTEGER :: timeStrID,secondsID,dayTimeFlagID
      INTEGER :: latVectorID,lonVectorID
      INTEGER :: modelLatID,modelLonID,modLatIndID,modLonIndID
      INTEGER :: H2ODegFrID,HDODegFrID,H2OQualID,HDOQualID


C**** Deallocate the point or grid arrays if they already exist
      IF (ALLOCATED(TESPresPt)) THEN
       DEALLOCATE(TESPresPt,TESDataLevsPt,TESAvgKrnlLevsPt, 
     & avgKrnlsH2OPt,avgKrnlsHDOPt, 
     & avgKrnlsHDPt,avgKrnlsDHPt,qPriorPt,dDPriorPt,
     & secondsPt,dayTimeFlagPt,latVecPt,lonVecPt,
     & latIndPt,lonIndPt,H2ODegFrPt,HDODegFrPt,
     & H2OQualPt,HDOQualPt)
      END IF

C**********************************************************************
C**** Read in daily point data from netcdf file
C****

C**** Try opening daily netcdf file, which has YYYY.MM.DD.nc format
      write(yearStr,'(I0)') currYear
      write(monthStr,'(I0)') currMonth
      write(dayStr,'(I0)') currDay

C     should be better way for this
      if (currMonth .lt. 10) then
          monthStr = '0'//monthStr
      end if
      if (currDay .lt. 10) then
          dayStr = '0'//dayStr
      end if
    
      dateStr = yearStr//'.'//trim(monthStr)//'.'//trim(dayStr)//'.nc'
      TESFilePath = trim(TESObsDataDir)//trim(dateStr)

      status=NF_OPEN(TESFilePath,NCNOWRIT,netcdfID)

      IF ( status .ne. NF_NOERR ) THEN
          print *,'No TES file found: ',TESFilePath
      ELSE
C**** Daily TES file exists, proceed
      
C**** Retrieve dimension IDs
      status=NF_INQ_DIMID(netcdfID,'nLocations',nLocationsID)
      status=NF_INQ_DIMID(netcdfID,'tesLevelDim',nTESLevsID)
      status=NF_INQ_DIMID(netcdfID,'dataLevelDim',nDataLevsID)
      status=NF_INQ_DIMID(netcdfID,'avgKrnlLevelDim',
     &                        nAvgKrnlLevsID)
      status=NF_INQ_DIMID(netcdfID,'timeStrLenDim',timeStrLenID)

C**** Retrieve variable IDs
      status=NF_INQ_VARID(netcdfID,'tesPressureLevels',
     & TESPresLevID)
      status=NF_INQ_VARID(netcdfID,'dataLevels',TESDataLevID)
      status=NF_INQ_VARID(netcdfID,'avgKernelLevels',
     & TESAvgKrnlLevID)
      status=NF_INQ_VARID(netcdfID,'H2OAvgKernels',avgKrnlH2OID)
      status=NF_INQ_VARID(netcdfID,'HDOAvgKernels',avgKrnlHDOID)
      status=NF_INQ_VARID(netcdfID,'HDCrossKernels',avgKrnlHDID)
      status=NF_INQ_VARID(netcdfID,'DHCrossKernels',avgKrnlDHID)
      status=NF_INQ_VARID(netcdfID,'qPrior',qPriorID)
      status=NF_INQ_VARID(netcdfID,'dDPrior',dDPriorID)
      status=NF_INQ_VARID(netcdfID,'seconds',secondsID)
      status=NF_INQ_VARID(netcdfID,'dayTimeFlag',dayTimeFlagID)
      status=NF_INQ_VARID(netcdfID,'latVector',latVectorID)
      status=NF_INQ_VARID(netcdfID,'lonVector',lonVectorID)
      status=NF_INQ_VARID(netcdfID,'latIndices',modLatIndID)
      status=NF_INQ_VARID(netcdfID,'lonIndices',modLonIndID)
      status=NF_INQ_VARID(netcdfID,'H2ODegFr',H2ODegFrID)
      status=NF_INQ_VARID(netcdfID,'HDODegFr',HDODegFrID)
      status=NF_INQ_VARID(netcdfID,'HDOSpeciesQuality',H2OQualID)
      status=NF_INQ_VARID(netcdfID,'H2OSpeciesQuality',HDOQualID)

C**** Retrieve dimensions
      status=NF_INQ_DIMLEN(netcdfID,nLocationsID,localNObs)
      status=NF_INQ_DIMLEN(netcdfID,timeStrLenID,timeStrLen)
      status=NF_INQ_DIMLEN(netcdfID,nTESLevsID,nTESLevsPt)
      status=NF_INQ_DIMLEN(netcdfID,nDataLevsID,nDataLevsPt)
      status=NF_INQ_DIMLEN(netcdfID,nAvgKrnlLevsID,nAvgKrnlLevsPt)

C**** Allocate apace
      ALLOCATE(TESPresPt(nTESLevsPt),STAT=code)
      ALLOCATE(TESDataLevsPt(nDataLevs),STAT=code)
      ALLOCATE(TESAvgKrnlLevsPt(nAvgKrnlLevs),STAT=code)
      ALLOCATE(avgKrnlsH2OPt(nDataLevsPt,nAvgKrnlLevsPt,localNObs),
     & STAT=code)
      ALLOCATE(avgKrnlsHDOPt(nDataLevsPt,nAvgKrnlLevsPt,localNObs),
     & STAT=code)
      ALLOCATE(avgKrnlsHDPt(nDataLevsPt,nAvgKrnlLevsPt,localNObs),
     & STAT=code)
      ALLOCATE(avgKrnlsDHPt(nDataLevsPt,nAvgKrnlLevsPt,localNObs),
     & STAT=code)
      ALLOCATE(qPriorPt(nTESLevsPt,localNObs),
     & STAT=code)
      ALLOCATE(dDPriorPt(nTESLevsPt,localNObs),
     & STAT=code)
      ALLOCATE(secondsPt(localNObs),STAT=code)
      ALLOCATE(dayTimeFlagPt(localNObs),STAT=code)
      ALLOCATE(latVecPt(localNObs),STAT=code)
      ALLOCATE(lonVecPt(localNObs),STAT=code)
      ALLOCATE(latIndPt(localNObs),STAT=code)
      ALLOCATE(lonIndPt(localNObs),STAT=code)
      ALLOCATE(H2ODegFrPt(localNObs),STAT=code)
      ALLOCATE(HDODegFrPt(localNObs),STAT=code)
      ALLOCATE(H2OQualPt(localNObs),STAT=code)
      ALLOCATE(HDOQualPt(localNObs),STAT=code)

C**** Retrieve data
      status = NF_GET_VAR_DOUBLE(netcdfID,TESPresLevID,TESPresPt)
      status = NF_GET_VAR_INT(netcdfID,TESDataLevID,TESDataLevsPt)
      status = NF_GET_VAR_INT(netcdfID,TESAvgKrnlLevID,
     & TESAvgKrnlLevsPt)
      status = NF_GET_VAR_DOUBLE(netcdfID,avgKrnlH2OID,avgKrnlsH2OPt)
      status = NF_GET_VAR_DOUBLE(netcdfID,avgKrnlHDOID,avgKrnlsHDOPt)
      status = NF_GET_VAR_DOUBLE(netcdfID,avgKrnlHDID,avgKrnlsHDPt)
      status = NF_GET_VAR_DOUBLE(netcdfID,avgKrnlDHID,avgKrnlsDHPt)
      status = NF_GET_VAR_DOUBLE(netcdfID,qPriorID,qPriorPt)
      status = NF_GET_VAR_DOUBLE(netcdfID,dDPriorID,dDPriorPt)
      status = NF_GET_VAR_INT(netcdfID,secondsID,secondsPt)
      status = NF_GET_VAR_INT(netcdfID,dayTimeFlagID,dayTimeFlagPt)
      status = NF_GET_VAR_DOUBLE(netcdfID,latVectorID,latVecPt)
      status = NF_GET_VAR_DOUBLE(netcdfID,lonVectorID,lonVecPt)
      status = NF_GET_VAR_INT(netcdfID,modLatIndID,latIndPt)
      status = NF_GET_VAR_INT(netcdfID,modLonIndID,lonIndPt)
      status = NF_GET_VAR_DOUBLE(netcdfID,H2ODegFrID,H2ODegFrPt)
      status = NF_GET_VAR_DOUBLE(netcdfID,HDODegFrID,HDODegFrPt)
      status = NF_GET_VAR_INT(netcdfID,H2OQualID,H2OQualPt)
      status = NF_GET_VAR_INT(netcdfID,HDOQualID,HDOQualPt)

      status=NF_CLOSE(netcdfID)
      END IF
      END SUBROUTINE ReadDailyTESFile

C********************************************************************** 
C Description: Map daily point-wise data to the ModelE grid. This is very 
C ModelE specfic, and has been separated from ReadDailyTESFile.
C********************************************************************** 
      SUBROUTINE InitializeTESOperatorCollocation(I_0,I_1,J_0,J_1,
     &           localNObs)

      USE TES_SIMULATOR
      USE CONSTANT, only : undef
      USE DOMAIN_DECOMP_1D, only : grid, unpack_data, am_i_root 
      USE GEOM, ONLY: lat2d_dg,lon2d_dg
      USE MODEL_COM, ONLY : ITIME

      IMPLICIT NONE

C 
C                               INPUT
C
      INTEGER, INTENT(IN) :: I_0,I_1,J_0,J_1
      INTEGER, INTENT(IN) :: localNObs
C 
C                               OUTPUT
C

C 
C                               LOCAL VARIABLES
C
      INTEGER :: currOb,currSecond,currGoodObs
      INTEGER :: currModLatInd,currModLonInd
      INTEGER :: I,J,code

    
C**** Allocate the relevant grid arrays
      IF (ALLOCATED(nObsGrid)) THEN
        DEALLOCATE(avgKrnlsH2OGrid,
     & avgKrnlsHDOGrid,
     & avgKrnlsHDGrid,
     & avgKrnlsDHGrid,
     & qPriorsGrid,
     & dDPriorsGrid,
     & nObsGrid,
     & goodObFlag,
     & secOfDayGrid)
      END IF

      ALLOCATE(nObsGrid(I_0:I_1,J_0:J_1),STAT=code)
      ALLOCATE(goodObFlag(I_0:I_1,J_0:J_1,maxObsPerCell),STAT=code)
      ALLOCATE(secOfDayGrid(I_0:I_1,J_0:J_1,maxObsPerCell),STAT=code)
      ALLOCATE(avgKrnlsH2OGrid(I_0:I_1,J_0:J_1,maxObsPerCell,
     & nDataLevsPt,nAvgKrnlLevsPt),
     & STAT=code)
      ALLOCATE(avgKrnlsHDOGrid(I_0:I_1,J_0:J_1,maxObsPerCell,
     & nDataLevsPt,nAvgKrnlLevsPt),
     & STAT=code)
      ALLOCATE(avgKrnlsHDGrid(I_0:I_1,J_0:J_1,maxObsPerCell,
     & nDataLevsPt,nAvgKrnlLevsPt),
     & STAT=code)
      ALLOCATE(avgKrnlsDHGrid(I_0:I_1,J_0:J_1,maxObsPerCell,
     & nDataLevsPt,nAvgKrnlLevsPt),
     & STAT=code)
      ALLOCATE(qPriorsGrid(I_0:I_1,J_0:J_1,maxObsPerCell,nTESLevsPt),
     & STAT=code)
      ALLOCATE(dDPriorsGrid(I_0:I_1,J_0:J_1,maxObsPerCell,nTESLevsPt),
     & STAT=code)

C**** Set counter array to 0
      DO I=I_0,I_1
          DO J=J_0,J_1
              nObsGrid(I,J) = 0
              DO currOb = 1,maxObsPerCell
                  secOfDayGrid(I,J,currOb) = undef
                  goodObFlag(I,J,currOb) = 0
              ENDDO
          ENDDO
      ENDDO

C**** Map each observation onto local grid, taking into account cases
C**** when there are more than 1 observation per grid cell
      DO currOb=1,localNObs
          currSecond  = secondsPt(currOb)
          currModLatInd = latIndPt(currOb)
          currModLonInd = lonIndPt(currOb)

C****     Only concerned with observations on local domain          
          IF ((currModLonInd >= I_0) .and. (currModLonInd <= I_1) .and.
     &        (currModLatInd >= J_0) .and.(currModLatInd <= J_1))THEN

C****         Increment the number of observations in the cell.
C****         The cell value of nObsGrid also becomes an index
C****         for the point info mapped onto the grid.    
              nObsGrid(currModLonInd,currModLatInd) = 
     &        nObsGrid(currModLonInd,currModLatInd) + 1

C****         Store the observation time, in seconds from midnight
              secOfDayGrid(currModLonInd,currModLatInd,
     &            nObsGrid(currModLonInd,currModLatInd)) 
     &            = currSecond

C****         Store the four averaging kernels and priors
              avgKrnlsH2OGrid(currModLonInd,currModLatInd,
     &            nObsGrid(currModLonInd,currModLatInd),:,:) 
     &            = avgKrnlsH2OPt(:,:,currOb)

              avgKrnlsHDOGrid(currModLonInd,currModLatInd,
     &            nObsGrid(currModLonInd,currModLatInd),:,:) 
     &            = avgKrnlsHDOPt(:,:,currOb)

              avgKrnlsHDGrid(currModLonInd,currModLatInd,
     &            nObsGrid(currModLonInd,currModLatInd),:,:) 
     &            = avgKrnlsHDPt(:,:,currOb)

              avgKrnlsDHGrid(currModLonInd,currModLatInd,
     &            nObsGrid(currModLonInd,currModLatInd),:,:) 
     &            = avgKrnlsDHPt(:,:,currOb)

              qPriorsGrid(currModLonInd,currModLatInd,
     &            nObsGrid(currModLonInd,currModLatInd),:) 
     &            = qPriorPt(:,currOb)

              dDPriorsGrid(currModLonInd,currModLatInd,
     &            nObsGrid(currModLonInd,currModLatInd),:) 
     &            = dDPriorPt(:,currOb)

C****         Note which observations have good quality and 
C****         high sensitivity.
              IF (H2ODegFrPt(currOb) >= TESObsMinDegFr .and. 
     &           HDODegFrPt(currOb) >= TESObsMinDegFr .and.
     &           (H2OQualPt(currOb) .eq. 1) .and.
     &           (HDOQualPt(currOb) .eq. 1)) THEN

                  goodObFlag(currModLonInd,currModLatInd,
     &                nObsGrid(currModLonInd,currModLatInd)) = 1
              ELSE
C****             Flag this as a bad observation
                  goodObFlag(currModLonInd,currModLatInd,
     &                nObsGrid(currModLonInd,currModLatInd)) = 0
              END IF
          END IF
      ENDDO

      END SUBROUTINE InitializeTESOperatorCollocation

C********************************************************************** 
C********************************************************************** 
C
C               Routines that should be model-independent
C
C********************************************************************** 
C**********************************************************************

      SUBROUTINE ReadTESAvgKrnlFile()
C********************************************************************** 
C Description: Reads in averaging kernel information from the file 
C              TESAvgKrnlFileName
C********************************************************************** 
      USE TES_SIMULATOR
	  
      IMPLICIT NONE
      include 'netcdf.inc'
C 
C                               INPUT:
C
!@var theDirectory the directory where the files reside
C     CHARACTER, INTENT(IN) :: fileName 
C 
C                                 OUTPUT:
C
 
C 
C                             LOCAL VARIABLES
C
!@var netcdf variable for file ID
      INTEGER :: netcdfID,status
!@var netcdf variable for dimension IDs
      INTEGER :: nTESLevsID,nDataLevsID,nAvgKrnlLevsID
      INTEGER :: timeStrLenID
      INTEGER :: nsrfTmpCatsID,noptDepCatsID,ncldTopPresCatsID
      INTEGER :: npwLowCatsID,npwHighCatsID
      INTEGER :: code

!@var netcdf variable for variables IDs
      INTEGER :: TESPresLevID,TESDataLevID,TESAvgKrnlLevID      
      INTEGER :: srfTmpCatsID,optDepCatsID,cldTopPresCatsID
      INTEGER :: pwLowCatsID,pwHighCatsID
      INTEGER :: avgKrnlH2OID,avgKrnlHDOID,avgKrnlHDID,avgKrnlDHID
      INTEGER :: percGoodObsID
      INTEGER :: latBdyID,bdyLyrDepID
      INTEGER :: qPriorID,rPriorID

C**** Open netcdf file
      status=NF_OPEN('TES_AK',NCNOWRIT,netcdfID)
      if ( status .ne. NF_NOERR ) then
	     call stop_model('Error opening: TESAvgKrnlFileName',255)
      endif
    
C
C   SHOULD ADD MORE ERROR CHECKING
C
C**** Retrieve dimension IDs 
      status=NF_INQ_DIMID(netcdfID,'tesLevelDim',nTESLevsID)
      status=NF_INQ_DIMID(netcdfID,'dataLevelDim',nDataLevsID)
      status=NF_INQ_DIMID(netcdfID,'avgKrnlLevelDim',nAvgKrnlLevsID)

      status=NF_INQ_DIMID(netcdfID,'srfTmpCatsDim',nsrfTmpCatsID)
      status=NF_INQ_DIMID(netcdfID,'optDepCatsDim',noptDepCatsID)
      status=NF_INQ_DIMID(netcdfID,
     &                    'cldTopPresCatsDim',ncldTopPresCatsID)
      status=NF_INQ_DIMID(netcdfID,'pwLowCatsDim',npwLowCatsID)
      status=NF_INQ_DIMID(netcdfID,'pwHighCatsDim',npwHighCatsID)

C**** Retrieve variable IDs 
      status=NF_INQ_VARID(netcdfID,'tesPressureLevels',TESPresLevID)
      status=NF_INQ_VARID(netcdfID,'dataLevels',TESDataLevID)
      status=NF_INQ_VARID(netcdfID,'avgKernelLevels',TESAvgKrnlLevID)
      status=NF_INQ_VARID(netcdfID,'H2OAvgKernels',avgKrnlH2OID)
      status=NF_INQ_VARID(netcdfID,'HDOAvgKernels',avgKrnlHDOID)
      status=NF_INQ_VARID(netcdfID,'HDCrossKernels',avgKrnlHDID)
      status=NF_INQ_VARID(netcdfID,'DHCrossKernels',avgKrnlDHID)
      status=NF_INQ_VARID(netcdfID,'percGoodObs',percGoodObsID)
      status=NF_INQ_VARID(netcdfID,'srfTmpCats',srfTmpCatsID)
      status=NF_INQ_VARID(netcdfID,'optDepCats',optDepCatsID)
      status=NF_INQ_VARID(netcdfID,'cldTopPresCats',cldTopPresCatsID)
      status=NF_INQ_VARID(netcdfID,'pwLowCats',pwLowCatsID)
      status=NF_INQ_VARID(netcdfID,'pwHighCats',pwHighCatsID)

      status=NF_INQ_VARID(netcdfID,'qPriors',qPriorID)
      status=NF_INQ_VARID(netcdfID,'RPrior',rPriorID)

      status=NF_INQ_VARID(netcdfID,'latBoundary',latBdyID)
      status=NF_INQ_VARID(netcdfID,'bdyLayerDepth',bdyLyrDepID)

C**** Retrieve dimensions
      status=NF_INQ_DIMLEN(netcdfID,nTESLevsID,nTESLevs)
      status=NF_INQ_DIMLEN(netcdfID,nDataLevsID,nDataLevs)
      status=NF_INQ_DIMLEN(netcdfID,nAvgKrnlLevsID,nAvgKrnlLevs)

      status=NF_INQ_DIMLEN(netcdfID,nsrfTmpCatsID,nsrfTmpCats)
      status=NF_INQ_DIMLEN(netcdfID,noptDepCatsID,noptDepCats)
      status=NF_INQ_DIMLEN(netcdfID,ncldTopPresCatsID,ncldTopPresCats)
      status=NF_INQ_DIMLEN(netcdfID,npwLowCatsID,npwLowCats)
      status=NF_INQ_DIMLEN(netcdfID,npwHighCatsID,npwHighCats)

C**** Allocate space
      ALLOCATE(TESPres(nTESLevs),STAT=code)
      ALLOCATE(TESDataLevs(nDataLevs),STAT=code)
      ALLOCATE(TESAvgKrnlLevs(nAvgKrnlLevs),STAT=code)

      ALLOCATE(srfTmpCats(nsrfTmpCats),STAT=code)
      ALLOCATE(optDepCats(noptDepCats),STAT=code)
      ALLOCATE(cldTopPresCats(ncldTopPresCats),STAT=code)
      ALLOCATE(pwLowCats(npwLowCats),STAT=code)
      ALLOCATE(pwHighCats(npwHighCats),STAT=code)

      ALLOCATE(avgKrnlsH2O(nsrfTmpCats-1,noptDepCats-1,ncldTopPresCats-1
     &                    ,npwLowCats-1,npwHighCats-1
     &                    ,nDataLevs,nAvgKrnlLevs),STAT=code)
      ALLOCATE(avgKrnlsHDO(nsrfTmpCats-1,noptDepCats-1,ncldTopPresCats-1
     &                    ,npwLowCats-1,npwHighCats-1
     &                    ,nDataLevs,nAvgKrnlLevs),STAT=code)
      ALLOCATE(avgKrnlsHD(nsrfTmpCats-1,noptDepCats-1,ncldTopPresCats-1
     &                   ,npwLowCats-1,npwHighCats-1
     &                   ,nDataLevs,nAvgKrnlLevs),STAT=code)
      ALLOCATE(avgKrnlsDH(nsrfTmpCats-1,noptDepCats-1,ncldTopPresCats-1
     &                   ,npwLowCats-1,npwHighCats-1
     &                   ,nDataLevs,nAvgKrnlLevs),STAT=code)
      ALLOCATE(qPriors(nsrfTmpCats-1,noptDepCats-1,ncldTopPresCats-1
     &                ,npwLowCats-1,npwHighCats-1
     &                ,nTESLevs),STAT=code)
      ALLOCATE(percGoodObs(nsrfTmpCats-1,noptDepCats-1,ncldTopPresCats-1
     &                ,npwLowCats-1,npwHighCats-1),STAT=code)
      ALLOCATE(rPrior(nTESLevs),STAT=code)	  

C**** Where de-allocate?

C**** Retrieve data into variables
      status = NF_GET_VAR_DOUBLE(netcdfID,TESPresLevID,TESPres)
      status = NF_GET_VAR_INT(netcdfID,TESDataLevID,TESDataLevs)
      status = NF_GET_VAR_INT(netcdfID,TESAvgKrnlLevID,TESAvgKrnlLevs)
      status = NF_GET_VAR_DOUBLE(netcdfID,srfTmpCatsID,srfTmpCats)
      status = NF_GET_VAR_DOUBLE(netcdfID,optDepCatsID,optDepCats)
      status = NF_GET_VAR_DOUBLE(netcdfID,cldTopPresCatsID,
     &                           cldTopPresCats)
      status = NF_GET_VAR_DOUBLE(netcdfID,pwLowCatsID,pwLowCats)
      status = NF_GET_VAR_DOUBLE(netcdfID,pwHighCatsID,pwHighCats)	
      status = NF_GET_VAR_DOUBLE(netcdfID,avgKrnlH2OID,avgKrnlsH2O)
      status = NF_GET_VAR_DOUBLE(netcdfID,avgKrnlHDOID,avgKrnlsHDO)
      status = NF_GET_VAR_DOUBLE(netcdfID,avgKrnlHDID,avgKrnlsHD)
      status = NF_GET_VAR_DOUBLE(netcdfID,avgKrnlDHID,avgKrnlsDH)
      status = NF_GET_VAR_DOUBLE(netcdfID,percGoodObsID,percGoodObs)
      status = NF_GET_VAR_DOUBLE(netcdfID,qPriorID,qPriors)
      status = NF_GET_VAR_DOUBLE(netcdfID,rPriorID,rPrior)

      status = NF_GET_VAR_DOUBLE(netcdfID,latBdyID,latBdy)
      status = NF_GET_VAR_DOUBLE(netcdfID,bdyLyrDepID,bdyLyrDep)

      status=NF_CLOSE(netcdfID)

      END SUBROUTINE ReadTESAvgKrnlFile

      SUBROUTINE RetrieveAKIndices(cldOD,cldTP,tSurf,pwLow,pwHigh
     * ,nCldODCats,nCldTPCats,nTSurfCats,nPwLowCats,nPwHighCats
     * ,cldODCats,cldTPCats,tSurfCats,pwLowCats,pwHighCats	 
     * ,cldODIndex,cldTPIndex,tSurfIndex,pwLowIndex,pwHighIndex)
C********************************************************************** 
C Description: Given model cldOD, cldTP,tSurf,pwLow,pwHigh and the 
C              corresponding categorical 'bins', return the indices
C              for the corresponding averaging kernel and prior. 
C              Any returned values of '0' indicate a problem.
C********************************************************************** 
C 
      USE CONSTANT, only : undef
      IMPLICIT NONE
C 
C                               INPUT:
C
!@var cldOD cloud optical depth
!@var tsurf surface (ground) temperature (deg C)
!@var cldTP cloud top pressure (hPa)
!@var pwLow precipitable water in 'boundary layer'
!@var pwHigh precipitable water in free atmosphere
      REAL*8, INTENT(IN) :: cldOD, cldTP,tSurf,pwLow,pwHigh
!@var nCldODCats number of cloud optical depth categories
!@var nCldTPCats number of cloud top pressure categories
!@var nTSurfCats number of surface temperature categories
!@var nPwLowCats number of bdy layer precipitable water categories
!@var nPwHighCats number of free atmosphere pw categories
      INTEGER, INTENT(IN) :: nCldODCats,nCldTPCats,nTSurfCats
      INTEGER, INTENT(IN) :: nPwLowCats,nPwHighCats
!@var cldODCats cloud optical depth categories
!@var cldTPCats cloud top pressure categories
!@var tSurfCats surface temperature categories
!@var pwLowCats bdy layer precipitable water categories
!@var pwHighCats free atmosphere precip water categories
      REAL*8, INTENT(IN), DIMENSION(nCldODCats) :: cldODCats 
      REAL*8, INTENT(IN), DIMENSION(nCldTPCats) :: cldTPCats 
      REAL*8, INTENT(IN), DIMENSION(nTSurfCats) :: tSurfCats 
      REAL*8, INTENT(IN), DIMENSION(nTSurfCats) :: pwLowCats 
      REAL*8, INTENT(IN), DIMENSION(nTSurfCats) :: pwHighCats 
C 
C                               OUTPUT:
C
!@var cldODIndex index of cldODCats corresponding to cldOD
!@var cldTPIndex index of cldTPCats corresponding to cldTP
!@var tSurfIndex index of tSurfCats corresponding to tSurf
!@var pwLowIndex index of pwLowCats corresponding to pwLow
!@var pwHighIndex index of pwHighCats corresponding to pwHigh
      INTEGER, INTENT(OUT) :: cldODIndex,cldTPIndex,tSurfIndex
      INTEGER, INTENT(OUT) :: pwLowIndex,pwHighIndex

C 
C                               INTERNAL:
C
      INTEGER currCat

      tSurfIndex = 0
      DO currCat=1,nTSurfCats-1
          if (tSurf.ge.tsurfCats(currCat) 
     &  .and. tsurf.lt.tsurfCats(currCat+1)) then
              tSurfIndex = currCat
          endif
      ENDDO
       
      cldODIndex = 0
      DO currCat=1,nCldODCats-1
          if (cldOD.ge.cldODCats(currCat) 
     &  .and. cldOD.lt.cldODCats(currCat+1)) then
              cldODIndex = currCat
          endif
      ENDDO
     
      cldTPIndex = 0
      DO currCat=1,nCldTPCats-1
          if (cldTP.ge.cldTPCats(currCat) 
     &  .and. cldTP.lt.cldTPCats(currCat+1)) then
              cldTPIndex = currCat
          endif
      ENDDO

      pwLowIndex = 0
      DO currCat=1,nPwLowCats-1
          if (pwLow.ge.pwLowCats(currCat) 
     &  .and. pwLow.lt.pwLowCats(currCat+1)) then
              pwLowIndex = currCat
          endif
      ENDDO

      pwHIghIndex = 0
      DO currCat=1,nPwHighCats-1
          if (pwHigh.ge.pwHighCats(currCat) 
     &  .and. pwHigh.lt.pwHighCats(currCat+1)) then
              pwHighIndex = currCat
          endif
      ENDDO

      END SUBROUTINE RetrieveAKIndices


      SUBROUTINE ApplyTESOperator(nPresTES,presTES
     *     ,h2oModelSH,hdoModelSH,presModel
     *     ,qPrior,rPrior
     *     ,nDataLev,nAKLev,dataLevIndTES,tesAKIndTES
     *     ,H2OAK,HDOAK,HDAK,DHAK
     *     ,h2oModelNewSH,hdoModelNewSH
     *     ,h2oModelIntpSH,hdoModelIntpSH)
C********************************************************************** 
C Description: Given model and prior profiles of q and hdo, and 
C              a set of 4 averaging kernels, this:
C              - interpolates model profiles onto the TES levels
C              - converts from mmr (spec hum) to vmr
C              - applies the TES avg krnl convolution 
C              - converts back to spec hum
C              - interpolates back to model levels
C********************************************************************** 
C 
C                               INPUT:
C

      USE MODEL_COM, only : lm
      USE CONSTANT, only : undef
      IMPLICIT NONE
!@var nPresTES number of TES pressure levels
      INTEGER, INTENT(IN) :: nPresTES
      REAL*8, INTENT(IN), DIMENSION(nPresTES) :: presTES
!@var h2oModelSH single model H2O profile on hybrid levels in kg/kg
!@var hdoModelSH single model HDO profile on hybrid levels in kg/kg
!@var presModel pressure at midpoint of box (i.e. from PMID)
      REAL*8, INTENT(IN), DIMENSION(LM)::h2oModelSH,hdoModelSH,presModel
!@var qPrior prior H2O profile on pressure levels
!@var rPrior prior HDO/H2O (volume) profile on pressure levels (invariant)
      REAL*8, INTENT(IN), DIMENSION(nPresTES) :: qPrior,rPrior
!@var dataLevIndTES indices in presTES over which avg krnls are applied
!@var tesAKIndTES indices presTES over which avg krnls are used
      INTEGER, INTENT(IN) :: nDataLev, nAKLev
      INTEGER, INTENT(IN), DIMENSION(nDataLev):: dataLevIndTES
      INTEGER, INTENT(IN), DIMENSION(nAKLev):: tesAKIndTES
!@var H2OAK single H2O averaging kernel
!@var HDOAK single HDO averaging kernel
!@var HDAK single HD averaging kernel
!@var DHAK single DH averaging kernel
      REAL*8, INTENT(IN), DIMENSION(nDataLev,nAKLev) :: H2OAK,HDOAK
      REAL*8, INTENT(IN), DIMENSION(nDataLev,nAKLev) :: HDAK,DHAK

C 
C                               OUTPUT:
C
!@var h2oModelNewSH convolved H2O profile
!@var hdoModelNewSH convolved H2O profile
!@var h2oModelIntpSH H2O profile, inteprolation only
!@var hdoModelIntpSH H2O profile, inteprolation only
      REAL*8,INTENT(OUT),DIMENSION(lm) :: h2oModelNewSH
      REAL*8,INTENT(OUT),DIMENSION(lm) :: hdoModelNewSH
      REAL*8,INTENT(OUT),DIMENSION(lm) :: h2oModelIntpSH
      REAL*8,INTENT(OUT),DIMENSION(lm) :: hdoModelIntpSH

C 
C                               INTERNAL VARIABLES
C
      REAL*8, DIMENSION(nPresTES) :: h2oModelTESsh,hdoModelTESsh
      REAL*8, DIMENSION(nPresTES) :: h2oModelTESvmr,hdoModelTESvmr
      REAL*8, DIMENSION(nPresTES) :: h2oPriorTESvmr,hdoPriorTESvmr	  
      REAL*8, DIMENSION(nDataLev) :: h2oNewShortvmr
      REAL*8, DIMENSION(nDataLev) :: hdoNewShortvmr
      REAL*8, DIMENSION(lm) :: dDNew,h2oNewModelvmr,hdoNewModelvmr
      REAL*8, DIMENSION(lm) :: h2oModelIntpvmr,hdoModelIntpvmr

C molecular masses of dry air, h2o and hdo
C REALLY NEED TO ORGANIZE THESE CONSTANTS BETTER - THEY'RE EVERYWHERE
      REAL*8 :: Mdry,Mh2o,Mhdo
      REAL*8 :: SMOWMASS,SMOWVOL
      REAL*8, DIMENSION(nDataLev) :: dDVol,dDMass

      h2oModelNewSH(1:lm) = undef
      hdoModelNewSH(1:lm) = undef

C oxygen - 15.9994
C hydrogen - 1.00794
C deuterium - 	2.01410178
      Mdry=28.966               
      Mh2o=18.016
      Mhdo=19.11144178
      SMOWMASS=3.29d-4
      SMOWVOL =(Mh2o/Mhdo)*SMOWMASS

C**** Interpolate model profiles onto TES pressure levels
      CALL LINTERP(0,lm,presModel,h2oModelSH,
     &            nPresTES,presTES,h2oModelTESsh)
      CALL LINTERP(0,lm,presModel,hdoModelSH,
     &            nPresTES,presTES,hdoModelTESsh)

C**** convert from spec hum to vmr
      CALL sh2vmr(nPresTES,h2oModelTESsh,hdoModelTESsh,
     &            h2oModelTESvmr,hdoModelTESvmr)

C**** the hdoPriorTESvmr provided to the call is a dummy variable      
      CALL sh2vmr(nPresTES,qPrior,undef*qPrior,
     &            h2oPriorTESvmr,hdoPriorTESvmr)
      hdoPriorTESvmr = rPrior*h2oPriorTESvmr

C**** Apply averaging kernel convolution
C****
C**** This is what it's all about.
C****
      CALL ApplyTESAvgKrnls(nPresTES,presTES
     *     ,h2oModelTESvmr,hdoModelTESvmr
     *     ,H2OAK,HDOAK,HDAK,DHAK
     *     ,h2oPriorTESvmr,hdoPriorTESvmr
     *     ,nDataLev,nAKLev,dataLevIndTES,tesAKIndTES
     *     ,h2oNewShortvmr,hdoNewShortvmr)

C**** Interpolate back onto model levels, convert back to moist mmr
      CALL LINTERP(0,nDataLev,presTES(dataLevIndTES),h2oNewShortvmr,
     &                    lm,presModel,h2oNewModelvmr)
      CALL LINTERP(0,nDataLev,presTES(dataLevIndTES),hdoNewShortvmr,
     &                    lm,presModel,hdoNewModelvmr)
      CALL vmr2sh(lm,h2oNewModelvmr,hdoNewModelvmr,
     &            h2oModelNewSH,hdoModelNewSH)

C**** Same for profiles subject to interpolation only, no convolution	 
      CALL LINTERP(0,nPresTES,presTES,h2oModelTESvmr,
     &                    lm,presModel,h2oModelIntpvmr)
      CALL LINTERP(0,nPresTES,presTES,hdoModelTESvmr,
     &                    lm,presModel,hdoModelIntpvmr)
      CALL vmr2sh(lm,h2oModelIntpvmr,hdoModelIntpvmr,
     &            h2oModelIntpSH,hdoModelIntpSH)

      END SUBROUTINE ApplyTESOperator
	  


      SUBROUTINE ApplyTESAvgKrnls(nPresTES,presTES,H2OModelv,HDOModelv
     *     ,H2OAK,HDOAK,HDAK,DHAK
     *     ,H2OPriorv,HDOPriorv
     *     ,nDataLev, nAKLev,dataLevIndTES,AKIndTES
     *     ,H2OModelNewv,HDOModelNewv)
C********************************************************************** 
C Description: Computes averaging kernel convolution, given raw model
C profiles of H2O and HDO, TES prior profiles of H2O and HDO, and a 
C set of four averaging kernels. All input profiles are on the TES
C pressure levels, and in volume mixing ratio. 
C********************************************************************** 
      USE CONSTANT, only : undef
      IMPLICIT NONE
C 
C                               INPUT:
C
!@var presTES vector of full TES pressureLevels
      INTEGER, INTENT(IN) :: nPresTES
      REAL*8, INTENT(IN),DIMENSION(nPresTES) :: presTES
!@var H2OModelv original H2O profile (vmr) from model on TES levels
!@var HDOModelv original HDO profile (vmr) from model on TES levels
      REAL*8, INTENT(IN),DIMENSION(nPresTES) :: H2OModelv, HDOModelv
!@var H2OPriorv H2O prior (vmr) from TES file 
!@var HDOPriorv HDO prior(vmr) from TES file
      REAL*8, INTENT(IN),DIMENSION(nPresTES) :: H2OPriorv,HDOPriorv
!@var H2OAK averaging kernel 
!@var HDOAK averaging kernel 
!@var HDAK averaging kernel 
!@var DHAK averaging kernel 
      INTEGER, INTENT(IN) :: nDataLev, nAKLev
      REAL*8, INTENT(IN),DIMENSION(nDataLev,nAKLev) :: H2OAK,HDOAK
      REAL*8, INTENT(IN),DIMENSION(nDataLev,nAKLev) :: HDAK,DHAK
!@var dataLevIndTES TES data level indices
!@var AKIndTES TES avg krnl level indices
      INTEGER, INTENT(IN),DIMENSION(nDataLev) :: dataLevIndTES
      INTEGER, INTENT(IN),DIMENSION(nAKLev) :: AKIndTES
C 
C                               OUTPUT:
C
!@var newH2OProfile new model H2O profile (vmr) after avg krnl.
      REAL*8,INTENT(OUT),DIMENSION(nDataLev):: H2OModelNewv,HDOModelNewv
C 
C                         LOCAL VARIABLES
C

      REAL*8 RPrior,lnH2ONew,lnRNew,isoRatioNew
C      REAL*8, DIMENSION(nDataLev) :: H2OModelNewv, HDOModelNewv
      INTEGER dataLev,akLev,dataLevInd,akLevInd


      H2OModelNewv = undef
      HDOModelNewv = undef

C**** See L344-416 from Camille's 'program_apply_kernels_monthly'
      DO dataLev=1,nDataLev
          dataLevInd = dataLevIndTES(dataLev)
          lnH2ONew = log(H2OPriorv(dataLevInd))
          DO akLev = 1,nAKLev
              akLevInd = AKIndTES(akLev)

	      IF ( (.NOT. ISNAN(H2OAK(dataLev,akLev))) .AND.
     &             (.NOT. ISNAN(H2OPriorv(akLevInd))) .AND.
     &             (H2OModelv(akLevInd).NE.undef)) THEN
	          lnH2ONew = lnH2ONew
     *            + H2OAK(dataLev,akLev)
     *            *(log(H2OModelv(akLevInd)) - log(H2OPriorv(akLevInd)))
              ENDIF  
          ENDDO
          H2OModelNewv(dataLev) = exp(lnH2ONew)
      ENDDO

      DO dataLev=1,nDataLev
            dataLevInd = dataLevIndTES(dataLev)
            lnRNew = log(HDOPriorv(dataLevInd)/H2OPriorv(dataLevInd))
            DO akLev = 1,nAKLev
                akLevInd = AKIndTES(akLev)
		IF ( (.NOT. ISNAN(H2OAK(dataLev,akLev))) .AND.
     &	             (.NOT. ISNAN(HDOAK(dataLev,akLev))) .AND.
     &	             (.NOT. ISNAN(HDAK(dataLev,akLev))) .AND.
     &	             (.NOT. ISNAN(DHAK(dataLev,akLev))) .AND.				 
     &	             (.NOT. ISNAN(H2OPriorv(akLevInd))) .AND.				 
     &	             (.NOT. ISNAN(HDOPriorv(akLevInd))) .AND.				 
     &	             (H2OModelv(akLevInd).NE.undef) .AND.
     &	             (HDOModelv(akLevInd).NE.undef)) THEN

		     lnRNew = lnRNew
     *          + (HDOAK(dataLev,akLev)-HDAK(dataLev,akLev))
     *          * (log(HDOModelv(akLevInd)) - log(HDOPriorv(akLevInd)))
     *          - (H2OAK(dataLev,akLev)-DHAK(dataLev,akLev))
     *          * (log(H2OModelv(akLevInd)) - log(H2OPriorv(akLevInd)))
		ENDIF  
            ENDDO
            isoRatioNew = exp(lnRNew)
            HDOModelNewv(dataLev) = isoRatioNew*H2OModelNewv(dataLev)
      ENDDO

      END SUBROUTINE ApplyTESAvgKrnls

      SUBROUTINE LINTERP(ASCENDING,N,XI,FI,NNEW,XNEW,FNEW)
C
C     Linear interpolation. This could be more efficient...
C
C
      USE CONSTANT, only : undef
      IMPLICIT NONE
      INTEGER, INTENT(IN) :: ASCENDING,N,NNEW
      REAL*8, INTENT(IN),DIMENSION(N) :: XI,FI
      REAL*8, INTENT(IN),DIMENSION(NNEW) :: XNEW
      REAL*8, INTENT(OUT),DIMENSION(NNEW) :: FNEW
      REAL*8 ratioPart
      INTEGER I,J

C****      print *,"XI",XI
C****      print *,"FI",FI
C****      print *,"XNEW",XNEW
	  
      DO I=1,NNEW-1
          FNEW(I) = undef
          DO J=1,N-1
              IF (XNEW(I).le.XI(J) .and. XNEW(I).ge.XI(J+1)) THEN
                  IF (FI(J).ne.undef .and. FI(J+1).ne.undef) THEN
                      ratioPart = (FI(J)-FI(J+1))/(XI(J) - XI(J+1))
                      FNEW(I)=FI(J+1) + (XNEW(I)-XI(J+1))*ratioPart
                  ENDIF
              ENDIF
          ENDDO
C****          print *,"FNEW(I)",FNEW(I)
      ENDDO

      END SUBROUTINE LINTERP

      SUBROUTINE sh2vmr(nLev,h2oSH,hdoSH,h2oVMR,hdoVMR)
C********************************************************************** 
C Description: Converts h2o and hdo from 'specific humidity' 
C (moist air mass mixing ratio) to volume mixing ratio.
C********************************************************************** 
      USE CONSTANT, only : undef
C 
C                               INPUT:
C
      INTEGER, INTENT(IN) :: nLev
!@var h2oSH h2o moist air mmr (spec hum)
!@var hdoSH hdo moist air mmr
      REAL*8, INTENT(IN),DIMENSION(nLev) :: h2oSH,hdoSH
C 
C                               OUTPUT:
C
!@var h2oVMR h2o volume mixing ratio
!@var hdoVMR h2o volume mixing ratio
      REAL*8, INTENT(OUT),DIMENSION(nLev) :: h2oVMR,hdoVMR
C 
C                               INTERNAL
C
c molecular masses of dry air, h2o and hdo
      REAL*8 :: Mdry,Mh2o,Mhdo
      INTEGER :: l

C oxygen - 15.9994
C hydrogen - 1.00794
C deuterium - 	2.01410178
      Mdry=28.966               
      Mh2o=18.016
      Mhdo=19.11144178

C These were derived starting from p32 of Jacobson
C "Fundamentals of atmospheric modeling"

      do  l=1,nLev
          if ((h2oSH(l).ne.undef).and.(hdoSH(l).ne.undef)) then
              h2oVMR(l) = (h2oSH(l)/(1.-h2oSH(l)))*(Mdry/Mh2o)
              hdoVMR(l) = (hdoSH(l)/Mhdo)*(Mh2o*h2oVMR(l) + Mdry)
          else
              h2oVMR(l) = undef
              hdoVMR(l) = undef
          endif
      enddo

      END SUBROUTINE sh2vmr

      SUBROUTINE vmr2sh(nLev,h2oVMR,hdoVMR,h2oSH,hdoSH)

      USE CONSTANT, only : undef
      INTEGER, INTENT(IN) :: nLev
      REAL*8, INTENT(IN),DIMENSION(nLev) :: h2oVMR,hdoVMR
      REAL*8, INTENT(OUT),DIMENSION(nLev) :: h2oSH,hdoSH
c molecular masses of dry air, h2o and hdo
      REAL*8 :: Mdry,Mh2o,Mhdo,mmr
C oxygen - 15.9994
C hydrogen - 1.00794
C deuterium - 	2.01410178
      Mdry=28.966
      Mh2o=18.016
      Mhdo=19.11144178

      do  l=1,nLev
          if ((h2oVMR(l).ne.undef) .and.
     &        (hdoVMR(l).ne.undef)) then
              mmr = h2oVMR(l)*(Mh2o/Mdry)
              h2oSH(l) = mmr/(1.+mmr)
              hdoSH(l) = (hdoVMR(l)*Mhdo)/
     *            (Mh2o*h2oVMR(l) + Mdry)
          else
              h2oSH(l) = undef
              hdoSH(l) = undef
          endif
      enddo
      END SUBROUTINE vmr2sh
	  
	  
	  
	  
      SUBROUTINE CalcPcpWat(nLevels,presLevs,q,srfDist,
     &                           pcpWLow,pcpWHigh)
C********************************************************************** 
C Description: Computes precipitable water given q on pressure levels, 
C              within srfDist hPa of surface, and above that.
C********************************************************************** 
      USE CONSTANT, only : undef,grav
      IMPLICIT NONE
C 
C                               INPUT:
C
!@var presLevs vector of pressure levels
!@var q specific humidity (kg/kg)
      INTEGER, INTENT(IN) :: nLevels
      REAL*8, INTENT(IN),DIMENSION(nLevels) :: presLevs,q
      REAL*8, INTENT(IN) :: srfDist

C
C                               OUTPUT:
C
!@var pcpWLow precipitable water over presLevs near surface
!@var pcpWHigh precipitable water in free atmosphere
      REAL*8,INTENT(OUT) :: pcpWLow,pcpWHigh
C 
C                         LOCAL VARIABLES
C
      REAL*8 :: minPres,maxPres,srfPres,bdyPres
      REAL*8, DIMENSION(nLevels) :: dstFrmGrnd
      INTEGER :: i,srfIndex,bdyIndex
      REAL*8, DIMENSION(nLevels) :: qFlip,presFlip

      REAL SplnCoeff(3,nLevels)
      REAL IntParts(nLevels)
      REAL Work(nLevels)

      pcpWLow = undef
      pcpWHigh = undef
      dstFrmGrnd = undef

C    Find index of surface, defined as first defined value 
C    of presLevs
      srfPres = undef
      srfIndex = 1
      do while (srfPres .eq. undef)
         if (presLevs(srfIndex) .ne. undef) then
             srfPres = presLevs(srfIndex)
         else
             srfIndex = srfIndex + 1
         endif
      enddo

C    make array containing the distance of each level to ground
      DO i=1,nLevels
          if (presLevs(i) .ne. undef) then
              dstFrmGrnd(i) = srfPres - presLevs(i)
          endif
      ENDDO

C    Find index of 'boundary layer' top
      bdyPres = 0.
      i = 1
      do while (bdyPres .eq. 0. .and. i.le.nLevels)
          if (dstFrmGrnd(i) .ge. srfDist) then
              bdyPres = presLevs(i)
              bdyIndex = i
          endif
          i = i + 1
      enddo
      
C      This could be more elegant using difference vectors
      pcpWLow = 0.
      DO i=1,bdyIndex-1
          pcpWLow = pcpWLow + (presLevs(i)-presLevs(i+1))*
     *        (q(i)+q(i+1))/2   
      ENDDO
      pcpWLow = 100*pcpWLow/grav

      pcpWHigh = 0.
      DO i=bdyIndex,nLevels-1
          pcpWHigh = pcpWHigh + (presLevs(i)-presLevs(i+1))*
     *        (q(i)+q(i+1))/2
      ENDDO
      pcpWHigh = 100*pcpWHigh/grav
      if (pcpWLow < 0 .or. pcpWHigh < 0) then
          print *,'pcpWLow',pcpWLow
          print *,'pcpWHigh',pcpWHigh
      endif


C    None of 3 canned numerical integration routines could handle profiles robustly.
C    Pre-Interpolating to a finer spacing might help, but for now, 
C    just use crude 'midpoint-rule type'appx integral above.
C
C	http://old.math.iastate.edu/burkardt/f_src/intlib/intlib.html

 
      END SUBROUTINE CalcPcpWat

