/* \file WriteData.c:
 * \brief: general interface for writing data
 */

#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../../include/include.h"
#include "../../shared/shared.h"

#if defined(_HAVE_MATLAB_) && defined(_SERIAL_)
#include <mex.h>

/*FUNCTION WriteData(mxArray** pdataref,DataSet* dataset){{{1*/
void WriteData(mxArray** pdataref,DataSet* dataset){

	mxArray* dataref=NULL;
	char* marshalled_dataset=NULL;
	int   marshalled_dataset_size;

	/*Write a dataset: */
	if(dataset){
			/* marshall the dataset: */
			marshalled_dataset=dataset->Marshall();
			marshalled_dataset_size=dataset->MarshallSize();
			
			dataref = mxCreateDoubleMatrix(0,0,mxREAL);
			mxSetM(dataref,(mwSize)(marshalled_dataset_size/sizeof(double)));
			mxSetN(dataref,(mwSize)1);
			mxSetPr(dataref,(double*)marshalled_dataset);	
	}
	else{
		/* return empty matrix: */
		dataref=mxCreateDoubleMatrix(0,0,mxREAL);
	}
	*pdataref=dataref;
	
}
/*}}}*/
/*FUNCTION WriteData(mxArray** pdataref,Matrix* matrix){{{1*/
void WriteData(mxArray** pdataref,Matrix* matrix){
		
	mxArray* dataref=NULL;
	
	if(matrix){
		
		/*call toolkit routine: */
		dataref=matrix->ToMatlabMatrix();
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}

	*pdataref=dataref;
}
/*}}}*/
/*FUNCTION WriteData(mxArray** pdataref,double* matrix, int M,int N){{{1*/
void WriteData(mxArray** pdataref,double* matrix, int M,int N){
	
	mxArray* dataref=NULL;
	mxArray* tdataref=NULL;
		
	if(matrix){
		
		/*data is a double* pointer. Copy into a matrix: */
		tdataref = mxCreateDoubleMatrix(0,0,mxREAL);
		mxSetM(tdataref,(mwSize)N);
		mxSetN(tdataref,(mwSize)M);
		mxSetPr(tdataref,(double*)matrix);

		//transpose
		mexCallMATLAB(1,&dataref,1,&tdataref, "transpose");
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}
	*pdataref=dataref;
}
/*}}}*/
/*FUNCTION WriteData(mxArray** pdataref,int* matrix, int M,int N){{{1*/
void WriteData(mxArray** pdataref,int* matrix, int M,int N){

	mxArray* dataref=NULL;
	mxArray* tdataref=NULL;

	if(matrix){

		/*convert to double matrix*/
		double* doublematrix=(double*)xmalloc(M*N*sizeof(double));
		for(int i=0;i<M*N;i++) doublematrix[i]=(double)matrix[i];

		/*data is a double* pointer. Copy into a matrix: */
		tdataref = mxCreateDoubleMatrix(0,0,mxREAL);
		mxSetM(tdataref,(mwSize)N);
		mxSetN(tdataref,(mwSize)M);
		mxSetPr(tdataref,(double*)doublematrix);

		//transpose
		mexCallMATLAB(1,&dataref,1,&tdataref, "transpose");
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}
	*pdataref=dataref;
}
/*}}}*/
/*FUNCTION WriteData(mxArray** pdataref,Vector* vector){{{1*/
void WriteData(mxArray** pdataref,Vector* vector){
	
	mxArray* dataref=NULL;
	
	if(vector){
		/*call toolkit routine: */
		dataref=vector->ToMatlabVector();
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}
	*pdataref=dataref;

}
/*}}}*/
/*FUNCTION WriteData(mxArray** pdataref,double* vector, int M){{{1*/
void WriteData(mxArray** pdataref,double* vector, int M){
	
	mxArray* dataref=NULL;

	if(vector){

		/*data is a double* pointer. Copy into a vector: */
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
		mxSetM(dataref,(mwSize)M);
		mxSetN(dataref,(mwSize)1);
		mxSetPr(dataref,vector);
	}
	else{
		dataref = mxCreateDoubleMatrix(0,0,mxREAL);
	}

	*pdataref=dataref;
}
/*}}}*/
/*FUNCTION WriteData(mxArray** pdataref,double scalar){{{1*/
void WriteData(mxArray** pdataref,double scalar){

	*pdataref=mxCreateDoubleScalar(scalar);
}
/*}}}*/
/*FUNCTION WriteData(mxArray** pdataref,int integer){{{1*/
void WriteData(mxArray** pdataref,int integer){

		*pdataref=mxCreateDoubleScalar((double)integer);

}
/*}}}*/
/*FUNCTION WriteData(mxArray** pdataref,int boolean){{{1*/
void WriteData(mxArray** pdataref,bool boolean){

	*pdataref=mxCreateDoubleScalar((double)boolean);

}
/*}}}*/
/*FUNCTION WriteData(mxArray** pdataref,char* string){{{1*/
void WriteData(mxArray** pdataref,char* string){

		*pdataref=mxCreateString(string);
}
/*}}}*/
/*FUNCTION WriteData(mxArray** pdataref,Parameters* parameters){{{1*/
void WriteData(mxArray** pdataref,Parameters* parameters){

	int i;

	/*output: */
	mxArray  *dataref     = NULL;
	mwSize    nfields;
	char    **fnames      = NULL;
	mwSize    onebyone[2] = {1,1};
	mwSize    ndim        = 2;

	/*intermediary: */
	Param*      param=NULL;

	/*Recover data from the parameters dataset: */
	nfields=(mwSize)parameters->Size();
	fnames=(char**)xmalloc(nfields*sizeof(char*));
	
	/*Build structure in matlab workspace with all the parameter fields: */
	for(i=0;i<nfields;i++){
		param=(Param*)parameters->GetObjectByOffset(i);
		param->GetParameterName(&fnames[i]);
	}
	/*Initialize structure: */
	dataref=mxCreateStructArray( ndim,onebyone,nfields,(const char**)fnames);

	/*Fill each field: */
	for(i=0;i<nfields;i++){

		param=(Param*)parameters->GetObjectByOffset(i);
		param->SetMatlabField(dataref);
	}
		
	/*Assign output pointers:*/
	*pdataref=dataref;

}
/*}}}*/
#endif
