/* \file macros.h
 * \brief: global macros used in the whole code
 */

/*Header {{{1*/
#ifndef _MACROS_H_
#define _MACROS_H_

#include "./typedefs.h"

#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif
/*}}}*/

/* _printf_ {{{1*/
/*Printing macro: only cpu number 0 */
#define _printf_(flag,...) do { if(flag) PrintfFunction(__VA_ARGS__); }while (0)
/*}}}*/
/* _error_ {{{1*/
/*Error exception macro*/
#ifdef _INTEL_WIN_
#define _error_(...)\
  throw ErrorException(exprintf(__VA_ARGS__))
#else
#define _error_(...)\
  throw ErrorException(__FILE__,__func__,__LINE__,exprintf(__VA_ARGS__))
#endif
/*}}}*/
/* _assert_ {{{1*/
/*Assertion macro: do nothing if macro _ISSM_DEBUG_ undefined*/
#ifdef _ISSM_DEBUG_ 
#define _assert_(statement)\
  if (!(statement)) _error_("Assertion \"%s\" failed, please report bug to %s",#statement,PACKAGE_BUGREPORT)
#else
#define _assert_(ignore)\
  ((void) 0)
#endif
/*}}}*/

/* MODULEBOOT/MODULEEND {{{1*/

/*The following macros hide the error exception handling in a matlab module. Just put 
 * MODULEBOOT(); and MODULEEND(); at the beginning and end of a module, and c++ exceptions 
 * will be trapped. Really nifty!*/

#ifdef _SERIAL_
#ifdef _HAVE_MATLAB_ //{{{2
#define MODULEBOOT(); ModuleBoot(); \
	try{

#define MODULEEND(); ModuleEnd(); }\
	catch(ErrorException &exception){\
		exception.Report(); \
		mexErrMsgTxt(""); \
	}\
	catch (exception& e) {\
		_printf_(true,"Standard exception: %s\n",e.what());\
		mexErrMsgTxt(" ");\
	}
#endif //}}}
#ifdef _HAVE_PYTHON_ //{{{2
#define MODULEBOOT(); ModuleBoot();  \
	PyObject* output = PyTuple_New(NLHS); if (!output) return NULL;

#define MODULEEND();  ModuleEnd(); \
						 return output;
#endif //}}}
#else 
//{{{2
#define MODULEBOOT(); \
	try{

#define MODULEEND(); }\
	catch(ErrorException &exception){\
		exception.Report(); \
		return 1;\
	}\
	catch (exception& e) {\
		_printf_(true,"Standard exception: %s\n",e.what());\
		return 1;\
	}
//}}}
#endif
/*}}}*/
/* WRAPPER {{{1*/
#ifdef _HAVE_MATLAB_
#define WRAPPER(modulename,...) void mexFunction(int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[]) 
#endif
#ifdef _HAVE_PYTHON_
#define WRAPPER(modulename,...)  \
\
static PyObject* modulename(PyObject* self,PyObject* args);\
static PyMethodDef modulename##_funcs[] = {\
	{#modulename, (PyCFunction)modulename, METH_VARARGS, ""},\
	{NULL,NULL,0,NULL}\
};\
\
static struct PyModuleDef modulename##module= {\
	PyModuleDef_HEAD_INIT,\
	#modulename,   /* name of module */\
	NULL, /* module documentation, may be NULL */\
	-1,       /* size of per-interpreter state of the module,\
				 or -1 if the module keeps state in global variables. */\
	modulename##_funcs\
};\
\
PyMODINIT_FUNC PyInit_##modulename(void){\
\
	import_array();\
	return PyModule_Create(&modulename##module);\
}\
\
static PyObject* modulename(PyObject* self,PyObject* args)

#endif

/*}}}*/
/* CHECKARGUMENTS {{{1*/
#ifdef _HAVE_MATLAB_
#define CHECKARGUMENTS(NLHS,NRHS,functionpointer) CheckNumMatlabArguments(nlhs,NLHS,nrhs,NRHS,__FUNCT__,functionpointer)
#endif
#ifdef _HAVE_PYTHON_
#define CHECKARGUMENTS(NLHS,NRHS,functionpointer) CheckNumPythonArguments(args, NRHS,functionpointer)
#endif
/*}}}*/


#endif
