/*
 * TriMesh: mesh a domain using an .exp file
 */

#include "./TriMesh.h"

WRAPPER(TriMesh, char* DOMAINOUTLINE, double AREA, bool ORDER){ 
	
	/* input: */
	char*  domainname=NULL;
	double area;
	bool   order;

	/*intermediary: */
	DataSet* domain=NULL;

	/* output: */
	Matrix* index=NULL;
	Vector* x=NULL;
	Vector* y=NULL;
	Matrix* segments=NULL;
	Vector* segmentmarkerlist=NULL;

	/*Boot module: */
	MODULEBOOT();

	/*checks on arguments on the matlab side: */
	CHECKARGUMENTS(NLHS,NRHS,&TriMeshUsage);
	
	/*Fetch data needed for meshing: */
	FetchData(&domainname,DOMAINOUTLINE);
	FetchData(&area,AREA);
	FetchData(&order,ORDER);

	/*Read domain outline: */
	domain=DomainOutlineRead(domainname,false);

	/*call x core: */
	TriMeshx(&index,&x,&y,&segments,&segmentmarkerlist,domain,area,order);

	/*write outputs: */
	WriteData(INDEX,index);
	WriteData(X,x);
	WriteData(Y,y);
	WriteData(SEGMENTS,segments);
	WriteData(SEGMENTMARKERLIST,segmentmarkerlist);

	/*free ressources: */
	delete domain;
	xdelete(&index);
	xdelete(&x);
	xdelete(&y);
	xdelete(&segments);
	xdelete(&segmentmarkerlist);

	/*end module: */
	MODULEEND();
}

void TriMeshUsage(void)
{
	printf("\n");
	printf("   usage: [index,x,y,segments,segmentmarkers]=TriMesh(domainoutlinefilename,area,ordered) \n");
	printf("      where: index,x,y defines a triangulation, segments is an array made \n");
	printf("      of exterior segments to the mesh domain outline, segmentmarkers is an array flagging each segment, \n");
	printf("      outlinefilename an Argus domain outline file, \n");
	printf("      area is the maximum area desired for any element of the resulting mesh, \n");
	printf("      and ordered is a bool that determines whether segments are output in the \n");
	printf("      order they are made by Triangle (ie none), or ordered counter clockwise around the domain outline.\n");
	printf("\n");
}


#ifdef _HAVE_PYTHON_
BOOST_PYTHON_MODULE(TriMesh){
	boost::python::numeric::array::set_module_and_type( "numpy", "ndarray");
	def("TriMesh",TriMesh);
}
#endif
