/*!\file InterpFromMesh3d.c
 * \brief: data interpolation from a list of (x,y,values) into mesh grids
 
	InterpFromMesh3d.c

	usage:
	data_mesh=InterpFromMesh3d(index,x,y,z,data,x_mesh,y_mesh,z_mesh);
	
	where:

		input:
		x,y,z: coordinates of matrix data
		data - matrix holding the data to be interpolated onto the mesh.
		x_mesh,y_mesh,z_mesh: coordinates of the mesh grids onto which we interpolate.
		
		output: 
		data_mesh:  vector of mesh interpolated data.

*/
	
#include "./InterpFromMesh3d.h"

void mexFunction( int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[]) {

	/*input: */
	double* index_data=NULL;
	int     index_data_rows;

	double* x_data=NULL;
	double* y_data=NULL;
	double* z_data=NULL;

	int     x_data_rows;
	int     y_data_rows;
	int     z_data_rows;

	double* data=NULL; 
	int     data_rows;
	int     data_cols;

	double* x_prime=NULL;
	double* y_prime=NULL;
	double* z_prime=NULL;
	
	int     x_prime_rows;
	int     y_prime_rows;
	int     z_prime_rows;

	double  default_value;

	/*Intermediary*/
	int nods_data;
	int nels_data;
	int nods_prime;

	/* output: */
	Vec  data_prime=NULL;

	/*Boot module: */
	MODULEBOOT();

	/*checks on arguments on the matlab side: */
	CheckNumMatlabArguments(nlhs,NLHS,nrhs,NRHS,__FUNCT__,&InterpFromMesh3dUsage);

	/*Input datasets: */
	FetchData((void**)&index_data,&index_data_rows,NULL,INDEXHANDLE,"Matrix","Mat");
	FetchData((void**)&x_data,&x_data_rows,NULL,XHANDLE,"Matrix","Mat");
	FetchData((void**)&y_data,&y_data_rows,NULL,YHANDLE,"Matrix","Mat");
	FetchData((void**)&z_data,&z_data_rows,NULL,ZHANDLE,"Matrix","Mat");
	FetchData((void**)&data,&data_rows,&data_cols,DATAHANDLE,"Matrix","Mat");
	FetchData((void**)&x_prime,&x_prime_rows,NULL,XPRIMEHANDLE,"Matrix","Mat");
	FetchData((void**)&y_prime,&y_prime_rows,NULL,YPRIMEHANDLE,"Matrix","Mat");
	FetchData((void**)&z_prime,&z_prime_rows,NULL,ZPRIMEHANDLE,"Matrix","Mat");
	FetchData((void**)&default_value,NULL,NULL,DEFAULTHANDLE,"Scalar",NULL);

	/*some checks*/
	if (x_data_rows!=y_data_rows || x_data_rows!=z_data_rows){
		throw ErrorException(__FUNCT__,"vectors x, y and z should have the same length!");
	}
	if (x_prime_rows!=y_prime_rows || x_prime_rows!=z_prime_rows){
		throw ErrorException(__FUNCT__,"vectors x_prime, y_prime and z_prime should have the same length!");
	}
	/*get number of elements and number of nodes in the data*/
	nels_data=index_data_rows;
	nods_data=x_data_rows;
	nods_prime=x_prime_rows;

	/* Run core computations: */
	InterpFromMesh3dx(&data_prime,index_data,x_data,y_data,z_data,nods_data,nels_data,data,data_rows,x_prime,y_prime,z_prime,nods_prime,default_value);

	/*Write data: */
	WriteData(DATAPRIME,data_prime,0,0,"Vector",NULL);

	/*end module: */
	MODULEEND();
}

void InterpFromMesh3dUsage(void)
{
	_printf_("   usage:\n");
	_printf_("      data_prime=InterpFromMesh3d(index,x,y,z,data,x_prime,y_prime,z_prime,default_value);\n\n");
	_printf_("   where:\n");
	_printf_("      x,y,z: coordinates of the nodes where data is defined\n");
	_printf_("      index: index of the mesh where data is defined\n");
	_printf_("      data - vector holding the data to be interpolated onto the points.\n");
	_printf_("      x_prime,y_prime,z_prime: coordinates of the mesh grids onto which we interpolate.\n");
	_printf_("      default_value - default value if no interpolation is found.\n");
	_printf_("      data_prime:  vector of prime interpolated data.\n");
	_printf_("\n");
}
