function varargout=antzoom(region)
%ANTZOOM - zoom on a region of Antarctica
%
%   This function zooms on an existing figure describing Antarctica. 
%   The zooming depends on the region name provided as input. 
%
%   Usage:
%      varargout=antzoom(region)


%Initialize regions
available_regions=InitializeRegionInfo();

%Check arguments
if nargin==0,
	help antzoom;
	disp('   Available regions include:');
	for i=1:length(available_regions),
		disp(['     ' available_regions(i).name]);
	end
	return;
end


found=0;
for i=1:length(available_regions),
	if strcmp(available_regions(i).name,region),
		%Make sure the region is a square, enlarge it otherwise, except for Antarctica. 
		x0=available_regions(i).x0;
		x1=available_regions(i).x1;
		y0=available_regions(i).y0;
		y1=available_regions(i).y1;
		if ~strcmp(region,'antarctica'),
			dx=x1-x0;
			dy=y1-y0;
			if dy>dx,
				delta=dy-dx;
				x0=x0-delta/2;
				x1=x1+delta/2;
			elseif dy<dx,
				delta=dx-dy;
				y0=y0-delta/2;
				y1=y1+delta/2;
			else
				%do nothing;
			end
		end

		%if output arguments are present, return the limits, 
		%otherwise, set them on the current graphic. 
		if nargout==2,
			found=1;
			varargout{1}=[x0 x1];
			varargout{2}=[y0 y1];
		else
			xlim([x0 x1]);
			ylim([y0 y1]);
			found=1;
			daspect([1;1;1]);
		end
	else
		continue;
	end
end

if found==0,
	disp(['Unknow region: ',region]);
	disp('Available regions include:');
	for i=1:length(available_regions),
		disp(['   ' available_regions(i).name]);
	end
end

end

function incrementedregions=AddAvailableRegion(regions,name,x0,x1,y0,y1);

	if isempty(regions),
		incrementedregions(1).name=name;
	else
		incrementedregions=regions;
		incrementedregions(end+1).name=name;
	end
	incrementedregions(end).x0=x0;
	incrementedregions(end).x1=x1;
	incrementedregions(end).y0=y0;
	incrementedregions(end).y1=y1;
end

function regions=InitializeRegionInfo()
	regions=AddAvailableRegion([],'antarctica',-2507187,2743299,-2143434,2241152);
	regions=AddAvailableRegion(regions,'adelieland',2.699*10^4,2.4636*10^6,-2.1848*10^6,-5.2865*10^5);
	regions=AddAvailableRegion(regions,'ameryiceshelf',1.5945*10^6,2.3807*10^6,5.1372*10^5,9.0186*10^5);
	regions=AddAvailableRegion(regions,'amundsencoast',-2.8*10^5,-1*10^5,-5.8*10^5,-4.6*10^5);
	regions=AddAvailableRegion(regions,'amundsensea',-2.1*10^6,-1.45*10^6,-7.5*10^5,-2.5*10^5);
	regions=AddAvailableRegion(regions,'banzarecoast',1.6*10^6,2.3*10^6,-1.9*10^6,-1.2*10^6);
	regions=AddAvailableRegion(regions,'borchgrevinkcoast',3*10^5,7*10^5,-2.05*10^6,-1.6*10^6);
	regions=AddAvailableRegion(regions,'buddcoast',2.1*10^6,2.7*10^6,-1.3*10^6,-.7*10^6);
	regions=AddAvailableRegion(regions,'byrdglacier',2.5*10^5,4.5*10^5,-10.8*10^5,-9.2*10^5);
	regions=AddAvailableRegion(regions,'dufekcoast',-1.6*10^5,0*10^5,-6.2*10^5,-5.5*10^5);
	regions=AddAvailableRegion(regions,'edouardpeninsula',-7.5*10^5,-4*10^5,-1.4*10^6,-1.05*10^6);
	regions=AddAvailableRegion(regions,'filchnericeshelf',-9.3781*10^5,-3.6671*10^5,.7389*10^6,1.0360*10^6);
	regions=AddAvailableRegion(regions,'fordranges',-8.5*10^5,-5.7*10^5,-1.33*10^6,-1.12*10^6);
	regions=AddAvailableRegion(regions,'georgevcoast',.9*10^6,1.7*10^6,-2.4*10^6,-1.7*10^6);
	regions=AddAvailableRegion(regions,'getziceshelf',-1.7*10^6,-1.15*10^6,-1.23*10^6,-.5*10^6);
	regions=AddAvailableRegion(regions,'hillarycoast',2.6*10^5,4.4*10^5,-1.24*10^6,-1.06*10^6);
	regions=AddAvailableRegion(regions,'larseniceshelf',-2.3855*10^6,-1.9649*10^6,0.9498*10^6,1.2996*10^6);
	regions=AddAvailableRegion(regions,'nimrodglacier',1*10^5,3.8*10^5,-9*10^5,-7*10^5);
	regions=AddAvailableRegion(regions,'oatescoast',5.3*10^5,8.5*10^5,-2.15*10^6,-1.9*10^6);
	regions=AddAvailableRegion(regions,'pennelcoast',3*10^5,6*10^5,-2.1*10^6,-1.85*10^6); 
	regions=AddAvailableRegion(regions,'pineislandglacier',-1.72*10^6,-1.45*10^6,-4.3*10^5,-1.4*10^5);
	regions=AddAvailableRegion(regions,'ronneiceshelf',-1.5*10^6,-.7*10^6,1*10^5,9.3*10^5);
	regions=AddAvailableRegion(regions,'rossiceshelf',-8*10^5,4.5*10^5,-13.8*10^5,-5*10^5);
	regions=AddAvailableRegion(regions,'ruppertcoast',-11*10^5,-8.5*10^5,-1.34*10^6,-1.14*10^6);
	regions=AddAvailableRegion(regions,'scottcoast',2.5*10^5,6*10^5,-1.6*10^6,-1.2*10^6);
	regions=AddAvailableRegion(regions,'shackletoncoast',2*10^5,3.8*10^5,-9.8*10^5,-8.4*10^5);
	regions=AddAvailableRegion(regions,'shackletoniceshelf',2.4*10^6,2.8*10^6,-7*10^5,-1*10^5);
	regions=AddAvailableRegion(regions,'shirasecoast',-6*10^5,-4.4*10^5,-10.9*10^5,-8.2*10^5);
	regions=AddAvailableRegion(regions,'siplecoast',-4.3*10^5,-1.5*10^5,-9*10^5,-6.5*10^5);
	regions=AddAvailableRegion(regions,'westiceshelf',2.2*10^6,2.7*10^6,0,5*10^5);
	regions=AddAvailableRegion(regions,'icestreamE',2.2*10^6,2.7*10^6,0,5*10^5);
	regions=AddAvailableRegion(regions,'peninsula',-2.7617*10^6,-1.2831*10^6,.4492*10^6,1.6940*10^6);
	regions=AddAvailableRegion(regions,'roosevelt',-6.2797*10^5,-1.5414*10^5,-1.2734*10^6,-.9609*10^6);
	regions=AddAvailableRegion(regions,'mcmurdo',1.3750*10^5,4.8255*10^5,-1.4569*10^6,-1.1119*10^6);
	regions=AddAvailableRegion(regions,'abbot',-2.1473*10^6,-1.6815*10^6,-3.5119*10^5,.3785*10^5);

end
