/*!\file Vector.cpp
 * \brief: implementation of the Vector object
 */

/*Headers:*/
/*{{{1*/
#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <stdio.h>
#include <string.h>
#include "../objects.h"
#include "../../shared/shared.h"
#include "../../Container/Container.h"
#include "../../include/include.h"
#include "../../toolkits/toolkits.h"
#include "./Vector.h"
/*}}}*/

/*Vector constructors and destructor*/
/*FUNCTION Vector::Vector(){{{1*/
Vector::Vector(){

	#ifdef _HAVE_PETSC_
	this->vector=NULL;
	#else
	this->vector=NULL;
	#endif
	#ifdef _HAVE_ADOLC_
	this->avector=NULL;
	#endif
}
/*}}}*/
/*FUNCTION Vector::Vector(int M,bool fromlocalsize){{{1*/
Vector::Vector(int pM,bool fromlocalsize){

	#ifdef _HAVE_PETSC_
	this->vector=NewVec(pM,fromlocalsize);
	#else
	this->vector=new SeqVec(pM,fromlocalsize);
	#endif
	#ifdef _HAVE_ADOLC_
	this->avector=(adouble*)xmalloc(pM*sizeof(adouble));
	#endif
}
/*}}}*/
/*FUNCTION Vector::Vector(double* serial_vec,int M){{{1*/
Vector::Vector(double* serial_vec,int M){

	#ifdef _HAVE_PETSC_
		int* idxm=(int*)xmalloc(M*sizeof(int));
		for(int i=0;i<M;i++) idxm[i]=i;

		this->vector=NewVec(M);
		VecSetValues(this->vector,M,idxm,serial_vec,INSERT_VALUES);
		VecAssemblyBegin(this->vector);
		VecAssemblyEnd(this->vector);

		xfree((void**)&idxm);
	#else
		this->vector=new SeqVec(serial_vec,M);
	#endif
	#ifdef _HAVE_ADOLC_
		this->avector=(adouble*)xmalloc(M*sizeof(adouble));
	#endif
}
/*}}}*/
#ifdef _HAVE_PETSC_
/*FUNCTION Vector::Vector(Vec petsc_vec){{{1*/
Vector::Vector(Vec petsc_vec){

	if(petsc_vec==NULL){
		this->vector=NewVec(0);
	}
	else{
		/*copy vector*/
		VecDuplicate(petsc_vec,&this->vector);
		VecCopy(petsc_vec,this->vector);
	}

}
/*}}}*/
#endif
#ifdef _HAVE_GSL_
/*FUNCTION Vector::Vector(SeqVec* seq_vec){{{1*/
Vector::Vector(SeqVec*  seq_vec){

	if(seq_vec==NULL){
		this->vector=NULL;
	}
	else{
		/*copy vector*/
		this->vector=seq_vec->Duplicate();
	}
}
/*}}}*/
#endif

		/*FUNCTION Vector::~Vector(){{{1*/
Vector::~Vector(){

 	#ifdef _HAVE_PETSC_
	VecFree(&this->vector);
	#else
	delete this->vector;
	#endif
	#ifdef _HAVE_ADOLC_
	xfree((void**)&this->avector);
	#endif
}
/*}}}*/

/*Vector specific routines: */
/*FUNCTION Vector::Echo{{{1*/
void Vector::Echo(void){

	int i;

	#ifdef _HAVE_PETSC_
	_assert_(this->vector);
	VecView(this->vector,PETSC_VIEWER_STDOUT_WORLD);
	#else
	this->vector->Echo();
	#endif

	#ifdef _HAVE_ADOLC_
	/*do nothing for now: */
	#endif
}
/*}}}*/

#ifdef _HAVE_MATLAB_
/*FUNCTION Vector::ToMatlabVector{{{1*/
mxArray* Vector::ToMatlabVector(void){

	mxArray* dataref=NULL;
	#ifdef _HAVE_PETSC_
	PetscVectorToMatlabVector(&dataref,this->vector);
	#else
	dataref=this->vector->ToMatlabVector();
	#endif
	return dataref;

}
/*}}}*/
/*FUNCTION MatlabVectorToVector{{{1*/
Vector* MatlabVectorToVector(const mxArray* mxvector){

	int dummy;
	Vector* vector=NULL;

	/*allocate vector object: */
	vector=new Vector();

	#ifdef _HAVE_PETSC_
	MatlabVectorToPetscVector(&vector->vector,&dummy,mxvector);
	#else
	vector->vector=MatlabVectorToSeqVec(mxvector);
	#endif
	
	return vector;
}
/*}}}*/
#endif
/*FUNCTION Vector::Assemble{{{1*/
void Vector::Assemble(void){
		
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecAssemblyBegin(this->vector); 
		VecAssemblyEnd(this->vector);
	#else
		this->vector->Assemble();
	#endif

}
/*}}}*/
/*FUNCTION Vector::SetValues{{{1*/
void Vector::SetValues(int ssize, int* list, double* values, InsMode mode){
		
		
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecSetValues(this->vector,ssize,list,values,ISSMToPetscInsertMode(mode));
	#else
		this->vector->SetValues(ssize,list,values,mode);
	#endif

}
/*}}}*/
/*FUNCTION Vector::SetValue{{{1*/
void Vector::SetValue(int dof, double value, InsMode mode){
		
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecSetValues(this->vector,1,&dof,&value,ISSMToPetscInsertMode(mode));
	#else
		this->vector->SetValue(dof,value,mode);
	#endif

}
/*}}}*/
/*FUNCTION Vector::GetValue{{{1*/
void Vector::GetValue(double* pvalue,int dof){
		
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecGetValues(this->vector,1,&dof,pvalue);
	#else
	this->vector->GetValue(pvalue,dof);
	#endif
}
/*}}}*/
/*FUNCTION Vector::GetSize{{{1*/
void Vector::GetSize(int* pM){
		
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecGetSize(this->vector,pM);
	#else
		this->vector->GetSize(pM);
	#endif

}
/*}}}*/
/*FUNCTION Vector::IsEmpty{{{1*/
bool Vector::IsEmpty(void){

	int M;

	_assert_(this->vector);
	this->GetSize(&M);

	if(M==0) 
	 return true;
	else
	 return false;
}
/*}}}*/
/*FUNCTION Vector::GetLocalSize{{{1*/
void Vector::GetLocalSize(int* pM){
		
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecGetLocalSize(this->vector,pM);
	#else
		this->vector->GetLocalSize(pM);
	#endif

}
/*}}}*/
/*FUNCTION Vector::Duplicate{{{1*/
Vector* Vector::Duplicate(void){
	
	Vector* output=NULL;
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		Vec vec_output=NULL;
		VecDuplicate(this->vector,&vec_output);
		output=new Vector(vec_output);
	#else
		output=new Vector();
		output->vector=this->vector->Duplicate();
	#endif
	return output;

}
/*}}}*/
/*FUNCTION Vector::Set{{{1*/
void Vector::Set(double value){
	
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecSet(this->vector,value);
	#else
		this->vector->Set(value);
	#endif

}
/*}}}*/
/*FUNCTION Vector::AXPY{{{1*/
void Vector::AXPY(Vector* X, double a){
	
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecAXPY(this->vector,a,X->vector);
	#else
		this->vector->AXPY(X->vector,a);
	#endif
}
/*}}}*/
/*FUNCTION Vector::AYPX{{{1*/
void Vector::AYPX(Vector* X, double a){
	
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecAYPX(this->vector,a,X->vector);
	#else
		this->vector->AYPX(X->vector,a);
	#endif

}
/*}}}*/
/*FUNCTION Vector::ToMPISerial{{{1*/
double* Vector::ToMPISerial(void){

	double* vec_serial=NULL;

	#ifdef _HAVE_PETSC_
		VecToMPISerial(&vec_serial, this->vector);
	#else
		vec_serial=this->vector->ToMPISerial();
	#endif

	return vec_serial;

}
/*}}}*/
/*FUNCTION Vector::Copy{{{1*/
void Vector::Copy(Vector* to){

	#ifdef _HAVE_PETSC_
		if(this->vector) VecCopy(this->vector,to->vector);
	#else
		this->vector->Copy(to->vector);
	#endif

}
/*}}}*/
/*FUNCTION Vector::Norm{{{1*/
double Vector::Norm(NormMode norm_type){
	
	double norm=0;
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecNorm(this->vector,ISSMToPetscNormMode(norm_type),&norm);
	#else
		norm=this->vector->Norm(norm_type);
	#endif
	return norm;
}
/*}}}*/
/*FUNCTION Vector::Scale{{{1*/
void Vector::Scale(double scale_factor){
	
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecScale(this->vector,scale_factor); 
	#else
		this->vector->Scale(scale_factor);
	#endif
}
/*}}}*/
/*FUNCTION Vector::Dot{{{1*/
double Vector::Dot(Vector* vector){

	double dot;
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecDot(this->vector,vector->vector,&dot);
	#else
		dot=this->vector->Dot(vector->vector);
	#endif
	return dot;
}
/*}}}*/
/*FUNCTION Vector::PointwiseDivide{{{1*/
void Vector::PointwiseDivide(Vector* x,Vector* y){

	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecPointwiseDivide(this->vector,x->vector,y->vector);
	#else
		this->vector->PointwiseDivide(x->vector,y->vector);
	#endif
}
/*}}}*/
