/*\file FetchMatlabData.cpp:
 * \brief: general I/O interface to fetch data in matlab
 */

#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../../shared/shared.h"
#include "../../include/include.h"

#ifdef _HAVE_MATLAB_
#include <mex.h>
/*FUNCTION FetchMatlabData(DataSet** pdataset,const mxArray* dataref){{{1*/
void FetchMatlabData(DataSet** pdataset,const mxArray* dataref){

	/*output*/
	DataSet* outdataset=NULL;
	char*    outdataset_buffer=NULL;
	int      outdataset_size;

	/*First, check that our reference is a double, otherwise, error out: */
	if (mxIsClass(dataref,"double")){
			
		/*We need to copy the data pointed by dataref, so that our dataset is not actually a pointer!:*/
		if (!mxIsEmpty(dataref)){
			outdataset_buffer=(char*)mxGetPr(dataref);
			outdataset_size=mxGetM(dataref)*mxGetN(dataref);
			if(outdataset_size)outdataset=DataSetDemarshall(outdataset_buffer);
		}
	}
	else{
		if (mxIsEmpty(dataref)){
			/*Nothing to pick up. Just initialize pointer to NULL, and warn the server we are not uploading anything: */
			outdataset_size=0;
			outdataset=NULL;
		}
		else{
			/*This is an error: we don't have the correct input!: */
			_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
		}
	}

	/*Assign output pointers:*/
	*pdataset=outdataset;
}
/*}}}*/
/*FUNCTION FetchMatlabData(double** pmatrix,int* pM,int *pN,const mxArray* dataref){{{1*/
void FetchMatlabData(double** pmatrix,int* pM,int *pN,const mxArray* dataref){

	double*  outmatrix=NULL;
	int      outmatrix_rows,outmatrix_cols;

	if(mxIsEmpty(dataref) ){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outmatrix_rows=0;
		outmatrix_cols=0;
		outmatrix=NULL;
	}
	else if(mxIsClass(dataref,"double") || mxIsClass(dataref,"single")){
		/*Check dataref is not pointing to NaN: */
		if ( mxIsNaN(*(mxGetPr(dataref))) && (mxGetM(dataref)==1) && (mxGetN(dataref)==1) ){
			outmatrix_rows=0;
			outmatrix_cols=0;
			outmatrix=NULL;
		}
		else{
			/*Convert matlab matrix to double* matrix: */
			MatlabMatrixToDoubleMatrix(&outmatrix,&outmatrix_rows,&outmatrix_cols,dataref);
		}
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}
			
	/*Assign output pointers:*/
	*pmatrix=outmatrix;
	if (pM)*pM=outmatrix_rows;
	if (pN)*pN=outmatrix_cols;

}
/*}}}*/
/*FUNCTION FetchMatlabData(double** pmatrix,int* pnumel,int* pndims,int** psize,const mxArray* dataref){{{1*/
void FetchMatlabData(double** pmatrix,int* pnumel,int* pndims,int** psize,const mxArray* dataref){

	double*  outmatrix=NULL;
	int      outmatrix_numel,outmatrix_ndims;
	int*     outmatrix_size=NULL;

	if(mxIsEmpty(dataref) ){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outmatrix_numel=0;
		outmatrix_ndims=0;
		outmatrix_size =NULL;
		outmatrix=NULL;
	}
	else if (mxIsClass(dataref,"double") ){

		/*Check dataref is not pointing to NaN: */
		if ( mxIsNaN(*(mxGetPr(dataref))) && (mxGetNumberOfElements(dataref)==1) ){
			outmatrix_numel=0;
			outmatrix_ndims=0;
			outmatrix_size =NULL;
			outmatrix=NULL;
		}
		else{

			/*Convert matlab n-dim array to double* matrix: */
			MatlabNArrayToNArray(&outmatrix,&outmatrix_numel,&outmatrix_ndims,&outmatrix_size,dataref);
		}
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}
			
	/*Assign output pointers:*/
	*pmatrix=outmatrix;
	if (pnumel)*pnumel=outmatrix_numel;
	if (pndims)*pndims=outmatrix_ndims;
	if (psize )*psize =outmatrix_size;
	else xfree((void**)&outmatrix_size);

}
/*}}}*/
/*FUNCTION FetchMatlabData(int** pmatrix,int* pM,int *pN,const mxArray* dataref){{{1*/
void FetchMatlabData(int** pmatrix,int* pM,int *pN,const mxArray* dataref){

	int     i,outmatrix_rows,outmatrix_cols;
	double *doublematrix=NULL;
	int    *outmatrix=NULL;

	if(mxIsEmpty(dataref) ){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outmatrix_rows=0;
		outmatrix_cols=0;
		outmatrix=NULL;
	}
	else if (mxIsClass(dataref,"double") ){

		/*Check dataref is not pointing to NaN: */
		if ( mxIsNaN(*(mxGetPr(dataref))) && (mxGetM(dataref)==1) && (mxGetN(dataref)==1) ){
			outmatrix_rows=0;
			outmatrix_cols=0;
			outmatrix=NULL;
		}
		else{

			/*Convert matlab matrix to double* matrix: */
			MatlabMatrixToDoubleMatrix(&doublematrix,&outmatrix_rows,&outmatrix_cols,dataref);

			/*Convert double matrix into integer matrix: */
			outmatrix=(int*)xmalloc(outmatrix_rows*outmatrix_cols*sizeof(int));
			for(i=0;i<outmatrix_rows*outmatrix_cols;i++)outmatrix[i]=(int)doublematrix[i];
		}
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pmatrix=outmatrix;
	if (pM)*pM=outmatrix_rows;
	if (pN)*pN=outmatrix_cols;
}
/*}}}*/
/*FUNCTION FetchMatlabData(bool** pmatrix,int* pM,int *pN,const mxArray* dataref){{{1*/
void FetchMatlabData(bool** pmatrix,int* pM,int *pN,const mxArray* dataref){

	int     i,outmatrix_rows,outmatrix_cols;
	double *doublematrix=NULL;
	bool   *outmatrix=NULL;

	if(mxIsEmpty(dataref) ){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outmatrix_rows=0;
		outmatrix_cols=0;
		outmatrix=NULL;
	}
	else if (mxIsClass(dataref,"double") ){

		/*Check dataref is not pointing to NaN: */
		if ( mxIsNaN(*(mxGetPr(dataref))) && (mxGetM(dataref)==1) && (mxGetN(dataref)==1) ){
			outmatrix_rows=0;
			outmatrix_cols=0;
			outmatrix=NULL;
		}
		else{

			/*Convert matlab matrix to double* matrix: */
			MatlabMatrixToDoubleMatrix(&doublematrix,&outmatrix_rows,&outmatrix_cols,dataref);

			/*Convert double matrix into integer matrix: */
			outmatrix=(bool*)xmalloc(outmatrix_rows*outmatrix_cols*sizeof(bool));
			for(i=0;i<outmatrix_rows;i++)outmatrix[i]=(bool)doublematrix[i];
		}
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pmatrix=outmatrix;
	if (pM)*pM=outmatrix_rows;
	if (pN)*pN=outmatrix_cols;
}
/*}}}*/
/*FUNCTION FetchMatlabData(bool** pmatrix,int* pnumel,int* pndims,int** psize,const mxArray* dataref){{{1*/
void FetchMatlabData(bool** pmatrix,int* pnumel,int* pndims,int** psize,const mxArray* dataref){

	int      i;
	int      outmatrix_numel,outmatrix_ndims;
	int*     outmatrix_size=NULL;
	double*  doublematrix=NULL;
	bool*    outmatrix=NULL;

	if(mxIsEmpty(dataref) ){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outmatrix_numel=0;
		outmatrix_ndims=0;
		outmatrix_size =NULL;
		outmatrix=NULL;
	}
	else if (mxIsClass(dataref,"logical") ){

		/*Check dataref is not pointing to NaN: */
		if ( mxIsNaN(*((bool*)mxGetData(dataref))) && (mxGetNumberOfElements(dataref)==1) ){
			outmatrix_numel=0;
			outmatrix_ndims=0;
			outmatrix_size =NULL;
			outmatrix=NULL;
		}
		else{

			/*Convert matlab n-dim array to bool* matrix: */
			MatlabNArrayToNArray(&outmatrix,&outmatrix_numel,&outmatrix_ndims,&outmatrix_size,dataref);
		}
	}
	else if (mxIsClass(dataref,"double") ){

		/*Check dataref is not pointing to NaN: */
		if ( mxIsNaN(*(mxGetPr(dataref))) && (mxGetNumberOfElements(dataref)==1) ){
			outmatrix_numel=0;
			outmatrix_ndims=0;
			outmatrix_size =NULL;
			outmatrix=NULL;
		}
		else{

			/*Convert matlab n-dim array to double* matrix: */
			MatlabNArrayToNArray(&doublematrix,&outmatrix_numel,&outmatrix_ndims,&outmatrix_size,dataref);

			/*Convert double matrix into bool matrix: */
			outmatrix=(bool*)xmalloc(outmatrix_numel*sizeof(bool));
			for(i=0;i<outmatrix_numel;i++)outmatrix[i]=(bool)doublematrix[i];
			xfree((void**)&doublematrix);
		}
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}
			
	/*Assign output pointers:*/
	*pmatrix=outmatrix;
	if (pnumel)*pnumel=outmatrix_numel;
	if (pndims)*pndims=outmatrix_ndims;
	if (psize )*psize =outmatrix_size;
	else xfree((void**)&outmatrix_size);

}
/*}}}*/
/*FUNCTION FetchMatlabData(Matrix** pmatrix,const mxArray* dataref){{{1*/
void FetchMatlabData(Matrix** pmatrix,const mxArray* dataref){
	
	Matrix* outmatrix=NULL;
	int dummy=0;

	if (mxIsClass(dataref,"double") ){
			
		/*Convert matlab matrix to matrix: */
		outmatrix=MatlabMatrixToMatrix(dataref);

	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pmatrix=outmatrix;
}
/*}}}*/
/*FUNCTION FetchMatlabData(double** pvector,int* pM,const mxArray* dataref){{{1*/
void FetchMatlabData(double** pvector,int* pM,const mxArray* dataref){

	double* outvector=NULL;
	int outvector_rows;

	if(mxIsEmpty(dataref)){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outvector_rows=0;
		outvector=NULL;
	}
	else if (mxIsClass(dataref,"double") ){

		/*Convert matlab vector to double*  vector: */
		MatlabVectorToDoubleVector(&outvector,&outvector_rows,dataref);

	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pvector=outvector;
	if (pM)*pM=outvector_rows;
}
/*}}}*/
/*FUNCTION FetchMatlabData(int** pvector,int* pM,const mxArray* dataref){{{1*/
void FetchMatlabData(int** pvector,int* pM,const mxArray* dataref){

	int    i;
	double *doublevector   = NULL;
	int    *outvector      = NULL;
	int     outvector_rows;

	if(mxIsEmpty(dataref)){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outvector_rows=0;
		outvector=NULL;
	}
	else if (mxIsClass(dataref,"double") ){

		/*Convert matlab vector to double*  vector: */
		MatlabVectorToDoubleVector(&doublevector,&outvector_rows,dataref);

		/*Convert double vector into integer vector: */
		outvector=(int*)xmalloc(outvector_rows*sizeof(int));
		for(i=0;i<outvector_rows;i++)outvector[i]=(int)doublevector[i];
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pvector=outvector;
	if (pM)*pM=outvector_rows;
}
/*}}}*/
/*FUNCTION FetchMatlabData(bool** pvector,int* pM,const mxArray* dataref){{{1*/
void FetchMatlabData(bool** pvector,int* pM,const mxArray* dataref){

	int    i;
	double *doublevector   = NULL;
	bool   *outvector      = NULL;
	int     outvector_rows;

	if(mxIsEmpty(dataref)){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outvector_rows=0;
		outvector=NULL;
	}
	else if (mxIsClass(dataref,"double") ){

		/*Convert matlab vector to double*  vector: */
		MatlabVectorToDoubleVector(&doublevector,&outvector_rows,dataref);

		/*Convert double vector into integer vector: */
		outvector=(bool*)xmalloc(outvector_rows*sizeof(bool));
		for(i=0;i<outvector_rows;i++)outvector[i]=(bool)doublevector[i];
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pvector=outvector;
	if (pM)*pM=outvector_rows;
}
/*}}}*/
/*FUNCTION FetchMatlabData(float** pvector,int* pM,const mxArray* dataref){{{1*/
void FetchMatlabData(float** pvector,int* pM,const mxArray* dataref){

	int    i;
	double *doublevector   = NULL;
	float  *outvector      = NULL;
	int     outvector_rows;

	if(mxIsEmpty(dataref)){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outvector_rows=0;
		outvector=NULL;
	}
	else if (mxIsClass(dataref,"double") ){

		/*Convert matlab vector to double*  vector: */
		MatlabVectorToDoubleVector(&doublevector,&outvector_rows,dataref);

		/*Convert double vector into float vector: */
		outvector=(float*)xmalloc(outvector_rows*sizeof(float));
		for(i=0;i<outvector_rows;i++)outvector[i]=(float)doublevector[i];
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pvector=outvector;
	if (pM)*pM=outvector_rows;
}
/*}}}*/
/*FUNCTION FetchMatlabData(Vector** pvector,const mxArray* dataref){{{1*/
void FetchMatlabData(Vector** pvector,const mxArray* dataref){
	
	Vector* vector=NULL;
	int dummy;

	if(mxIsEmpty(dataref)){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		vector=new Vector(0);
	}
	else if (mxIsClass(dataref,"double") ){

		/*Convert matlab vector to petsc vector: */
		vector=MatlabVectorToVector(dataref);
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}

	/*Assign output pointers:*/
	*pvector=vector;
}
/*}}}*/
/*FUNCTION FetchMatlabData(char** pstring,const mxArray* dataref){{{1*/
void FetchMatlabData(char** pstring,const mxArray* dataref){

	char* outstring=NULL;


	/*Ok, the string should be coming directly from the matlab workspace: */
	if (!mxIsClass(dataref,"char")){
		_error_("input data_type is not a string!");
	}
	else{
		/*Recover the string:*/
		int stringlen;
		
		stringlen = mxGetM(dataref)*mxGetN(dataref)+1;
		outstring = (char*)xmalloc(sizeof(mxChar)*stringlen);
		mxGetString(dataref,outstring,stringlen);
	}

	/*Assign output pointers:*/
	*pstring=outstring;
}
/*FUNCTION FetchMatlabData(char** pmatrix,int* pnumel,int* pndims,int** psize,const mxArray* dataref){{{1*/
void FetchMatlabData(char** pmatrix,int* pnumel,int* pndims,int** psize,const mxArray* dataref){

	int      outmatrix_numel,outmatrix_ndims;
	int*     outmatrix_size=NULL;
	char*    outmatrix=NULL;

	if(mxIsEmpty(dataref) ){
		/*Nothing to pick up. Just initialize matrix pointer to NULL: */
		outmatrix_numel=0;
		outmatrix_ndims=0;
		outmatrix_size =NULL;
		outmatrix=NULL;
	}
	else if (mxIsClass(dataref,"char") ){

		/*Check dataref is not pointing to NaN: */
		if ( mxIsNaN(*(mxGetPr(dataref))) && (mxGetNumberOfElements(dataref)==1) ){
			outmatrix_numel=0;
			outmatrix_ndims=0;
			outmatrix_size =NULL;
			outmatrix=NULL;
		}
		else{

			/*Convert matlab n-dim array to char* matrix: */
			MatlabNArrayToNArray(&outmatrix,&outmatrix_numel,&outmatrix_ndims,&outmatrix_size,dataref);
		}
	}
	else{
		/*This is an error: we don't have the correct input!: */
		_error_("Input parameter of class %s not supported yet",mxGetClassName(dataref));
	}
			
	/*Assign output pointers:*/
	*pmatrix=outmatrix;
	if (pnumel)*pnumel=outmatrix_numel;
	if (pndims)*pndims=outmatrix_ndims;
	if (psize )*psize =outmatrix_size;
	else xfree((void**)&outmatrix_size);

}
/*}}}*/
/*FUNCTION FetchMatlabData(double* pscalar,const mxArray* dataref){{{1*/
void FetchMatlabData(double* pscalar,const mxArray* dataref){

	double scalar;

	if (!mxIsClass(dataref,"double")){
		_error_("input data_type is not a double!");
	}
	else{
		/*Recover the double: */
		scalar=mxGetScalar(dataref);
	}

	/*Assign output pointers:*/
	*pscalar=scalar;
}
/*}}}*/
/*FUNCTION FetchMatlabData(int* pinteger,const mxArray* dataref){{{1*/
void FetchMatlabData(int* pinteger,const mxArray* dataref){

	int integer;

	if (!mxIsClass(dataref,"double")){
		_error_("input data_type is not a scalar!");
	}
	else{
		/*Recover the double: */
		integer=(int)mxGetScalar(dataref);
	}

	/*Assign output pointers:*/
	*pinteger=integer;
}
/*}}}*/
/*FUNCTION FetchMatlabData(bool* pboolean,const mxArray* dataref){{{1*/
void FetchMatlabData(bool* pboolean,const mxArray* dataref){

	bool* mxbool_ptr=NULL;

	if (mxIsClass(dataref,"logical")){
		if(mxGetM(dataref)!=1) _error_("input data is not of size 1x1");
		if(mxGetN(dataref)!=1) _error_("input data is not of size 1x1");
		mxbool_ptr=mxGetLogicals(dataref);
	}
	else{
		_error_("input data_type is not a bool!");
	}

	*pboolean=*mxbool_ptr;
}
/*}}}*/
/*FUNCTION FetchMatlabData(Parameters** pparameters, DataHandle dataref){{{1*/
void FetchMatlabData(Parameters** pparameters, DataHandle dataref){

	int i,j;
	int count;

	/*output: */
	Param* param=NULL;
	Parameters* parameters=NULL;

	/*intermediary: */
	int M,N;
	double* tmatrix=NULL;
	double* matrix=NULL;
	char**  stringarray=NULL;
	double** array=NULL;
	int*     mdims_array=NULL;
	int*     ndims_array=NULL;
	int nfields;
	char* name=NULL;
	mxArray* pfield=NULL;
	mxArray* pfield2=NULL;
	int enum_type;


	/*First, create parameters : */
	parameters=new Parameters();

	/*go through matlab params structure, and create Param object for each field: */

	nfields=mxGetNumberOfFields(dataref);

	for(count=0;count<nfields;count++){

		/*Get i'th field: */
		name=(char*)mxGetFieldNameByNumber(dataref,count);
		enum_type=StringToEnumx(name);
		pfield=mxGetFieldByNumber(dataref,0,count);
		_assert_(pfield);
		
		/*Check type of field: */
		if (mxIsClass(pfield,"double")){
			
			M=mxGetM(pfield);
			N=mxGetN(pfield);

			if (M==0 | N==0){
				_error_("array in parameters structure field %s is of size 0",name);
			}
			if (M==1 && N==1){
				/*we have a simple scalar: */
				param= new DoubleParam(enum_type,*mxGetPr(pfield));
				parameters->AddObject(param);

			}
			else{
				if (N==1){
					
					/*vector: */
					param= new DoubleVecParam(enum_type,mxGetPr(pfield),M);
					parameters->AddObject(param);

				}
				else{
					/*matrix: first, transpose, then plug into Param */
					matrix=mxGetPr(pfield);
					tmatrix=(double*)xmalloc(M*N*sizeof(double));
					for (i=0;i<M;i++){
						for(j=0;j<N;j++){
							*(tmatrix+N*i+j)=*(matrix+M*j+i);
						}
					}

					param= new DoubleMatParam(enum_type,tmatrix,M,N);
					parameters->AddObject(param);
	
					/*Free ressources:*/
					xfree((void**)&tmatrix);
				}
			}

		}
		else if (mxIsClass(pfield,"logical")){

			M=mxGetM(pfield);
			N=mxGetN(pfield);

			if (M==0 | N==0){
				_error_("array in parameters structure field %s is of size 0",name);
			}
			if (M==1 && N==1){
				/*we have a simple bool: */
				param= new BoolParam(enum_type,*mxGetLogicals(pfield));
				parameters->AddObject(param);

			}
			else{
				_error_("Matrices of Booleans not supported yet in parameters");
			}
		}
		else if (mxIsClass(pfield,"char")){
			/* we have a string parameter:*/
			
			int stringlen;
			char* string=NULL;
			
			stringlen = mxGetM(pfield)*mxGetN(pfield)+1;
			string = (char*)xmalloc(sizeof(mxChar)*stringlen);
			mxGetString(pfield,string,stringlen);

			param= new StringParam(enum_type,string);
			parameters->AddObject(param);

			xfree((void**)&string);
		}
		else if (mxIsClass(pfield,"cell")){

			/*This can be a string array, or a matrix array. Check the first 
			 *element type to decide: */
			pfield2=mxGetCell(pfield,0);
			if (mxIsClass(pfield2,"char")){
				
				/*string array: */
				M=mxGetM(pfield);
				stringarray=(char**)xmalloc(M*sizeof(char*));

				for(i=0;i<M;i++){
					char* descriptor=NULL;
					pfield2=mxGetCell(pfield,i);
					FetchMatlabData(&descriptor,pfield2);
					stringarray[i]=descriptor;
				}

				param= new StringArrayParam(enum_type,stringarray,M);
				parameters->AddObject(param);

				/*Free ressources:*/
				for(i=0;i<M;i++){
					char* descriptor=stringarray[i];
					xfree((void**)&descriptor);
				}
				xfree((void**)&stringarray);

			}
			else{
				
				/*matrix array: */
				M=mxGetM(pfield);
				array=(double**)xmalloc(M*sizeof(double*));
				mdims_array=(int*)xmalloc(M*sizeof(int));
				ndims_array=(int*)xmalloc(M*sizeof(int));

				for(i=0;i<M;i++){
					double* matrix=NULL;
					int     m,n;
					pfield2=mxGetCell(pfield,i);
					FetchMatlabData(&matrix,&m,&n,pfield2);
					array[i]=matrix;
					mdims_array[i]=m;
					ndims_array[i]=n;
				}

				param= new DoubleMatArrayParam(enum_type,array,M,mdims_array,ndims_array);
				parameters->AddObject(param);

				/*Free ressources:*/
				for(i=0;i<M;i++){
					double* matrix=array[i];
					xfree((void**)&matrix);
				}
				xfree((void**)&array);
				xfree((void**)&mdims_array);
				xfree((void**)&ndims_array);
			}
		}
		else _error_("Parameters structure field %s has a format that is not supported: %s",name,mxGetClassName(pfield));
	}

	/*Assign output pointers:*/
	*pparameters=parameters;
}
/*}}}*/
#endif
