/*!\file InterpFromMesh2d.c
 * \brief: data interpolation from a list of (x,y,values) into mesh grids
 
	InterpFromMesh2d.c

	usage:
	data_mesh=InterpFromMesh2d(index,x,y,data,x_mesh,y_mesh);
	
	where:

		input:
		x,y: coordinates of matrix data
		data - matrix holding the data to be interpolated onto the mesh.
		x_mesh,y_mesh: coordinates of the mesh grids onto which we interpolate.
		
		output: 
		data_mesh:  vector of mesh interpolated data.

*/
	
#include "./InterpFromMesh2d.h"

void mexFunction( int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[]) {

	/*input: */
	double* index_data=NULL;
	int     index_data_rows;

	double* x_data=NULL;
	int     x_data_rows;
	
	double* y_data=NULL;
	int     y_data_rows;

	double* data=NULL; 
	int     data_rows;
	int     data_cols;

	double* x_prime=NULL;
	double* y_prime=NULL;
	
	int     x_prime_rows;
	int     y_prime_rows;

	double default_value;

	/*Intermediary*/
	int i,j;
	int nods_data;
	int nels_data;
	int nods_prime;
	int interpolation_type;
	double area;
	double area_1,area_2,area_3;
	double data_value;
	double xmin,xmax;
	double ymin,ymax;

	/* output: */
	Vec  data_prime=NULL;

	/*Boot module: */
	MODULEBOOT();

	/*checks on arguments on the matlab side: */
	CheckNumMatlabArguments(nlhs,NLHS,nrhs,NRHS,__FUNCT__,&InterpFromMesh2dUsage);

	/*Input datasets: */
	FetchData((void**)&index_data,&index_data_rows,NULL,INDEXHANDLE,"Matrix","Mat");
	FetchData((void**)&x_data,&x_data_rows,NULL,XHANDLE,"Matrix","Mat");
	FetchData((void**)&y_data,&y_data_rows,NULL,YHANDLE,"Matrix","Mat");
	FetchData((void**)&data,&data_rows,&data_cols,DATAHANDLE,"Matrix","Mat");
	FetchData((void**)&x_prime,&x_prime_rows,NULL,XPRIMEHANDLE,"Matrix","Mat");
	FetchData((void**)&y_prime,&y_prime_rows,NULL,YPRIMEHANDLE,"Matrix","Mat");
	FetchData((void**)&default_value,NULL,NULL,DEFAULTHANDLE,"Scalar",NULL);

	/* Run core computations: */
	//InterpFromMesh2dx( &data_mesh, index , index_rows, x, y, x_rows, data, data_rows, x_mesh, y_mesh, x_mesh_rows);

	/*some checks*/
	if (index_data_rows<1 || x_data_rows<3 || y_data_rows<3){
		throw ErrorException(__FUNCT__,"nothing to be done according to the mesh given in input");
	}
	if (x_data_rows!=y_data_rows){
		throw ErrorException(__FUNCT__,"vectors x and y should have the same length!");
	}
	if (x_prime_rows!=y_prime_rows){
		throw ErrorException(__FUNCT__,"vectors x_prime and y_prime should have the same length!");
	}
	
	/*get number of elements and number of nodes in the data*/
	nels_data=index_data_rows;
	nods_data=x_data_rows;
	nods_prime=x_prime_rows;

	/*figure out what kind of interpolation is needed*/
	if (data_rows==nods_data){
		interpolation_type=1;
	}
	else if (data_rows==nels_data){
		interpolation_type=2;
	}
	else{
		throw ErrorException(__FUNCT__,"length of vector data not supported yet. It should be of length (number of nodes) or (number of elements)!");
	}

	/*Get prime mesh extrema coordinates*/
	xmin=x_prime[0]; xmax=x_prime[0];ymin=y_prime[0]; ymax=y_prime[0];
	for (i=1;i<nods_prime;i++){
		if (x_prime[i]<xmin) xmin=x_prime[i];
		if (x_prime[i]>xmax) xmax=x_prime[i];
		if (y_prime[i]<ymin) ymin=y_prime[i];
		if (y_prime[i]>ymax) ymax=y_prime[i];
	}

	/*Initialize output*/
	data_prime=NewVec(nods_prime);
	for (i=0;i<nods_prime;i++) VecSetValue(data_prime,i,default_value,INSERT_VALUES);

	/*Loop over the elements*/
	printf("\n      interpolation progress:   %5.2lf %%",0.0);
	for (i=0;i<nels_data;i++){

		/*display current iteration*/
		if (fmod(i,100)==0){
			printf("\b\b\b\b\b\b\b%5.2lf %%",(double)i/nels_data*100);
		}

		/*if there is no point inside the domain, go to next iteration*/
		if ( (x_data[(int)index_data[3*i+0]-1]<xmin) && (x_data[(int)index_data[3*i+1]-1]<xmin) && (x_data[(int)index_data[3*i+2]-1]<xmin)) continue;
		if ( (x_data[(int)index_data[3*i+0]-1]>xmax) && (x_data[(int)index_data[3*i+1]-1]>xmax) && (x_data[(int)index_data[3*i+2]-1]>xmax)) continue;
		if ( (y_data[(int)index_data[3*i+0]-1]<ymin) && (y_data[(int)index_data[3*i+1]-1]<ymin) && (y_data[(int)index_data[3*i+2]-1]<ymin)) continue;
		if ( (y_data[(int)index_data[3*i+0]-1]>ymax) && (y_data[(int)index_data[3*i+1]-1]>ymax) && (y_data[(int)index_data[3*i+2]-1]>ymax)) continue;

		/*get area of the current element (Jacobian = 2 * area)*/
		//area =x2 * y3 - y2*x3 + x1 * y2 - y1 * x2 + x3 * y1 - y3 * x1;
		area=x_data[(int)index_data[3*i+1]-1]*y_data[(int)index_data[3*i+2]-1]-y_data[(int)index_data[3*i+1]-1]*x_data[(int)index_data[3*i+2]-1]+ x_data[(int)index_data[3*i+0]-1]*y_data[(int)index_data[3*i+1]-1]-y_data[(int)index_data[3*i+0]-1]*x_data[(int)index_data[3*i+1]-1]+ x_data[(int)index_data[3*i+2]-1]*y_data[(int)index_data[3*i+0]-1]-y_data[(int)index_data[3*i+2]-1]*x_data[(int)index_data[3*i+0]-1];

		/*loop over the prime nodes*/
		for (j=0;j<nods_prime;j++){

			/*Get first area coordinate = det(x-x3  x2-x3 ; y-y3   y2-y3)/area*/
			area_1=((x_prime[j]-x_data[(int)index_data[3*i+2]-1])*(y_data[(int)index_data[3*i+1]-1]-y_data[(int)index_data[3*i+2]-1]) -  (y_prime[j]-y_data[(int)index_data[3*i+2]-1])*(x_data[(int)index_data[3*i+1]-1]-x_data[(int)index_data[3*i+2]-1]))/area;
			/*Get second area coordinate =det(x1-x3  x-x3 ; y1-y3   y-y3)/area*/
			area_2=((x_data[(int)index_data[3*i+0]-1]-x_data[(int)index_data[3*i+2]-1])*(y_prime[j]-y_data[(int)index_data[3*i+2]-1]) - (y_data[(int)index_data[3*i+0]-1]-y_data[(int)index_data[3*i+2]-1])*(x_prime[j]-x_data[(int)index_data[3*i+2]-1]))/area;
			/*Get third area coordinate = 1-area1-area2*/
			area_3=1-area_1-area_2;

			/*is the current point in the current element?*/
			if (area_1>=0 && area_2>=0 && area_3>=0){

				/*Yes ! compute the value on the point*/
				if (interpolation_type==1){
					/*nodal interpolation*/
					data_value=area_1*data[(int)index_data[3*i+0]-1]+area_2*data[(int)index_data[3*i+1]-1]+area_3*data[(int)index_data[3*i+2]-1];
				}
				else{
					/*element interpolation*/
					data_value=data[i];
				}
				if isnan(data_value) data_value=default_value;

				/*insert value and go to the next point*/
				VecSetValue(data_prime,j,data_value,INSERT_VALUES);
			}
		}
	}
	printf("\b\b\b\b\b\b\b%5.2lf %%\n",100.0);

	/*Write data: */
	WriteData(DATAPRIME,data_prime,0,0,"Vector",NULL);

	/*end module: */
	MODULEEND();
}

void InterpFromMesh2dUsage(void)
{
	_printf_("   usage:\n");
	_printf_("      data_prime=InterpFromMesh2d(index,x,y,data,x_prime,y_prime,default_value);\n\n");
	_printf_("   where:\n");
	_printf_("      x,y: coordinates of the nodes where data is defined\n");
	_printf_("      index: index of the mesh where data is defined\n");
	_printf_("      data - vector holding the data to be interpolated onto the points.\n");
	_printf_("      x_prime,y_prime: coordinates of the mesh grids onto which we interpolate.\n");
	_printf_("      default_value - default value if no interpolation is found.\n");
	_printf_("      data_prime:  vector of prime interpolated data.\n");
	_printf_("\n");
}
