function data_out=InterpFromFile(x,y,filename)
%INTERPFROMFILE - load data and interpolate on the given nodes
%
%   load a matlab file (extension .mat) which holds 3 or 4 variables
%   and interpolate the data on the mesh and plug it onto the model.
%
%   o 3 variables
%     - a vector x (if the name of the variable do not begin with "x", an error can appear)
%     - a vector y (if the name of the variable do not begin with "y", an error can appear)
%     - a vector or matrix data (if the name of the variable do not begin with the field name, an error can appear)
%   o 4 variables
%     - a vector x (if the name of the variable do not begin with "x", an error can appear)
%     - a vector y (if the name of the variable do not begin with "y", an error can appear)
%     - a matrix with 3 columns (if the name of the variable do not begin with "index" or "elements", an error can appear)
%     - a vector data (if the name of the variable do not begin with the field name, an error can appear)
%
%   Usage:
%      data=InterpFromFile(x,y,filename);
%
%   Example:
%      md.surface=InterpFromFile(md.x,md.y,'surfacefile.mat');
%
%   See also: PLUGVELOCITIES, INTERPFROMGRID, INTERPFROMMESH2D, INTERPFROMMESH3D

%some checks
if nargin~=3 | nargout~=1
	help InterpFromFile
	error('plugdata error message: bad usage');
end
if ~exist(filename)
	error(['plugdata error message: file ' filename  ' does not exist']);
end
if length(x)~=length(y),
	error('plugdata error message: x and y should have the same length');
end

%Get variables
A=whos('-file',filename);

%check the number of variables
if (length(A)~=4 & length(A)~=3)
	error(['plugdata error message: file ' filename  ' not supported yet (it should hold 3 or 4 variables)']);
end

%1: find data using their names
xenum=NaN; yenum=NaN; indexenum=NaN; dataenum=NaN;
for i=1:length(A),

	%x -> name begins with "x" and is a vector
	if (strcmpi(A(i).name(1),'x') & min(A(i).size)==1),
		xenum=i;

	%y -> name begins with "y" and is a vector
	elseif (strcmpi(A(i).name(1),'y') & min(A(i).size)==1),
		yenum=i;

	%index -> name begins with "index" or "elements" and 3 columns
	elseif length(A)==4
		if (( (length(A(i).name)>=5 & strcmpi(A(i).name(1:5),'index')) ...
				| (length(A(i).name)>=8 & strcmpi(A(i).name(1:8),'elements')) )...
				& A(i).size(2)==3),
			indexenum=i;
		end

	%data -> name begins with "field" and is a vector or a matrix
	%elseif (strcmpi(A(i).name,field) & max(A(i).size)>1),
	%	dataenum=i;

	end
end

%2: if only one item is missing, find it by elimination
if length(A)==4
	pos=find(isnan([xenum yenum indexenum dataenum]));
	if length(pos)==1,
		list=[xenum yenum indexenum dataenum]; list(pos)=[];
		if pos==1,
			xenum=setdiff(1:4,list);
		elseif pos==2,
			yenum=setdiff(1:4,list);
		elseif pos==3,
			indexenum=setdiff(1:4,list);
		elseif pos==4,
			dataenum=setdiff(1:4,list);
		end
	end
else
	pos=find(isnan([xenum yenum dataenum]));
	if length(pos)==1,
		list=[xenum yenum indexenum dataenum]; list(pos)=[];
		if pos==1,
			xenum=setdiff(1:3,list);
		elseif pos==2,
			yenum=setdiff(1:3,list);
		elseif pos==3,
			dataenum=setdiff(1:3,list);
		end
	end
end

%3: if one or several variables are missing, use sizes instead of names (works only if data is a matrix...)
if (isnan(xenum) | isnan(yenum) |  isnan(dataenum) | (length(A)==4 & isnan(indexenum)))

	if (length(A)==4 & isnan(indexenum)),
		for i=1:4
			if A(i).size(2)==3,
				indexenum=i;
			end
		end
		if isnan(indexenum)
			error(['InterpFromFile error message: file ' filename  ' not supported yet (index not found)']);
		end
	end

	if isnan(dataenum),
		maxsize1=1;
		maxsize2=1;

		%find dataenum
		for i=1:length(A)
			sizei=A(i).size;
			if (sizei(1)>=maxsize1-1 & sizei(2)>=maxsize2-1),
				dataenum=i
				maxsize1=sizei(1);
				maxsize2=sizei(2);
			end
		end
		if isnan(dataenum)
			error(['InterpFromFile error message: file ' filename  ' not supported yet (data not found)']);
		end
		if (A(dataenum).size(1)==A(dataenum).size(2) & isnan(xenum) & isnan(yenum)),
			error(['InterpFromFile error message: file ' filename  ' not supported (the data is a square matrix, save x and y with another name)']);
		end
	end

	%find xenum and yenum
	if isnan(xenum) | isnan(yenum)
		for i=1:length(A),
			lengthi=max(A(i).size);
			if isnan(yenum) & ((i~=dataenum) & (lengthi==A(dataenum).size(1) | lengthi==A(dataenum).size(1)+1)),
				yenum=i;
			elseif isnan(xenum) & ((i~=dataenum) & (lengthi==A(dataenum).size(2) | lengthi==A(dataenum).size(2)+1)),
				xenum=i;
			end
		end
	end

	%last chance: if only one item is missing, find it by elimination
	if length(A)==4
		pos=find(isnan([xenum yenum indexenum dataenum]));
		if length(pos)==1,
			list=[xenum yenum indexenum dataenum]; list(pos)=[];
			if pos==1,
				xenum=setdiff(1:4,list);
			elseif pos==2,
				yenum=setdiff(1:4,list);
			elseif pos==3,
				indexenum=setdiff(1:4,list);
			elseif pos==4,
				dataenum=setdiff(1:4,list);
			end
		end
	else
		pos=find(isnan([xenum yenum dataenum]));
		if length(pos)==1,
			list=[xenum yenum indexenum dataenum]; list(pos)=[];
			if pos==1,
				xenum=setdiff(1:3,list);
			elseif pos==2,
				yenum=setdiff(1:3,list);
			elseif pos==3,
				dataenum=setdiff(1:3,list);
			end
		end
	end

	%last check
	if (isnan(xenum) | isnan(yenum) | isnan(dataenum) | (length(A)==4 & isnan(indexenum)))
		error(['InterpFromFile error message: file ' filename  ' not supported yet (not recognzed variable names or sizes)']);
	end
end

%OK, interpolate
if length(A)==4,

	%create names:
	xname=A(xenum).name;
	yname=A(yenum).name;
	indexname=A(indexenum).name;
	dataname=A(dataenum).name;

	%load data
	B=load(filename);

	%get x y and data
	eval(['x_data=B.' xname ';'])
	eval(['y_data=B.' yname ';'])
	eval(['index_data=B.' indexname ';'])
	eval(['data=B.' dataname ';'])

	%interpolate
	data_out=InterpFromMesh2d(index_data,x_data,y_data,data,x,y);

else

	%create names:
	xname=A(xenum).name;
	yname=A(yenum).name;
	dataname=A(dataenum).name;

	%load data
	B=load(filename);

	%get x y and data
	eval(['x_data=B.' xname ';'])
	eval(['y_data=B.' yname ';'])
	eval(['data=B.' dataname ';'])

	%interpolate
	data_out=InterpFromGrid(x_data,y_data,data,x,y);
end
