/*!\file Vector.cpp
 * \brief: implementation of the Vector object
 */

/*Headers:*/
/*{{{1*/
#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <stdio.h>
#include <string.h>
#include "../objects.h"
#include "../../shared/shared.h"
#include "../../Container/Container.h"
#include "../../include/include.h"
#include "../../toolkits/toolkits.h"
#include "./Vector.h"
/*}}}*/

/*Vector constructors and destructor*/
/*FUNCTION Vector::Vector(){{{1*/
Vector::Vector(){

	this->M=0;
	#ifdef _HAVE_PETSC_
	this->vector=NULL;
	#else
	this->vector=NULL;
	#endif
	#ifdef _HAVE_ADOLC_
	this->avector=NULL;
	#endif
}
/*}}}*/
/*FUNCTION Vector::Vector(int M,bool fromlocalsize){{{1*/
Vector::Vector(int pM,bool fromlocalsize){

	#ifdef _HAVE_PETSC_
	this->vector=NewVec(pM,fromlocalsize);
	#else
	this->M=pM;
	this->vector=(double*)xcalloc(pM,sizeof(double));
	#endif
	#ifdef _HAVE_ADOLC_
	this->avector=(adouble*)xmalloc(pM*sizeof(adouble));
	#endif
}
/*}}}*/
/*FUNCTION Vector::Vector(double* serial_vec,int M){{{1*/
Vector::Vector(double* serial_vec,int pM){

	int i,j;

	#ifdef _HAVE_PETSC_
		int* idxm=NULL;

		this->vector=NewVec(pM);
		
		idxm=(int*)xmalloc(M*sizeof(int));
		for(i=0;i<M;i++)idxm[i]=i;
		VecSetValues(this->vector,M,idxm,serial_vec,INSERT_VALUES);

		VecAssemblyBegin(this->vector);
		VecAssemblyEnd(this->vector);

		xfree((void**)&idxm);

	#else
		this->M=pM;
		this->vector=(double*)xcalloc(pM,sizeof(double));
	#endif
	#ifdef _HAVE_ADOLC_
		this->avector=(adouble*)xmalloc(pM*sizeof(adouble));
	#endif
}
/*}}}*/
#ifdef _HAVE_PETSC_
/*FUNCTION Vector::Vector(Vec petsc_vec){{{1*/
Vector::Vector(Vec petsc_vec){

	/*Get Vector size*/
	VecGetSize(petsc_vec,&this->M);

	/*copy vector*/
	VecDuplicate(petsc_vec,&this->vector);
	VecCopy(petsc_vec,this->vector);

}
/*}}}*/
#endif
		/*FUNCTION Vector::~Vector(){{{1*/
Vector::~Vector(){

 	#ifdef _HAVE_PETSC_
	VecFree(&this->vector);
	#else
	xfree((void**)&this->vector);
	#endif
	#ifdef _HAVE_ADOLC_
	xfree((void**)&this->avector);
	#endif
}
/*}}}*/

/*Vector specific routines: */
/*FUNCTION Vector::Echo{{{1*/
void Vector::Echo(void){

	int i;

	#ifdef _HAVE_PETSC_
	VecView(this->vector,PETSC_VIEWER_STDOUT_WORLD);
	#else
	printf("Vector size: %i\n",M);
	for(i=0;i<M;i++){
		printf("%g\n ",*(vector+i));
	}
	#endif

	#ifdef _HAVE_ADOLC_
	/*Not sure about that one. Should we use the overloaded operator >>?*/
	printf("ADOLC Vector equivalent:" );
	for(i=0;i<M;i++){
		printf("%g\n ",*(avector+i));
	}
	#endif
}
/*}}}*/

#ifdef _SERIAL_
/*FUNCTION Vector::ToMatlabVector{{{1*/
mxArray* Vector::ToMatlabVector(void){

	mxArray* dataref=NULL;
	#ifdef _HAVE_PETSC_
	PetscVectorToMatlabVector(&dataref,this->vector);
	#else
	_error_("not implemented yet!");
	#endif
	return dataref;

}
/*}}}*/
#endif
/*FUNCTION Vector::Assemble{{{1*/
void Vector::Assemble(void){
		
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecAssemblyBegin(this->vector); 
		VecAssemblyEnd(this->vector);
	#else
		/*do nothing*/
	#endif

}
/*}}}*/
/*FUNCTION Vector::SetValues{{{1*/
void Vector::SetValues(int ssize, int* list, double* values, InsMode mode){
		
		
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecSetValues(this->vector,ssize,list,values,ISSMToPetscInsertMode(mode));
	#else
		_error_("not implemented yet!");
	#endif

}
/*}}}*/
/*FUNCTION Vector::SetValue{{{1*/
void Vector::SetValue(int dof, double value, InsMode mode){
		
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecSetValues(this->vector,1,&dof,&value,ISSMToPetscInsertMode(mode));
	#else
		_error_("not implemented yet!");
	#endif

}
/*}}}*/
/*FUNCTION Vector::GetValue{{{1*/
void Vector::GetValue(double* pvalue,int dof){
		
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecGetValues(this->vector,1,&dof,pvalue);
	#else
		_error_("not implemented yet!");
	#endif
}
/*}}}*/
/*FUNCTION Vector::GetSize{{{1*/
void Vector::GetSize(int* pM){
		
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecGetSize(this->vector,pM);
	#else
		_error_("not implemented yet!");
	#endif

}
/*}}}*/
/*FUNCTION Vector::GetLocalSize{{{1*/
void Vector::GetLocalSize(int* pM){
		
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecGetLocalSize(this->vector,pM);
	#else
		_error_("not implemented yet!");
	#endif

}
/*}}}*/
/*FUNCTION Vector::Duplicate{{{1*/
Vector* Vector::Duplicate(void){
	
	Vector* output=NULL;
	output=new Vector();
	#ifdef _HAVE_PETSC_
		Vec vec_output=NULL;
		_assert_(this->vector);
		VecDuplicate(this->vector,&vec_output);
		output->vector=vec_output;
		VecGetSize(output->vector,&output->M);
	#else
		_error_("not implemented yet!");
	#endif
	return output;

}
/*}}}*/
/*FUNCTION Vector::Set{{{1*/
void Vector::Set(double value){
	
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecSet(this->vector,value);
	#else
		_error_("not implemented yet!");
	#endif

}
/*}}}*/
/*FUNCTION Vector::AXPY{{{1*/
void Vector::AXPY(Vector* X, double a){
	
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecAXPY(this->vector,a,X->vector);
	#else
		_error_("not implemented yet!");
	#endif
}
/*}}}*/
/*FUNCTION Vector::AYPX{{{1*/
void Vector::AYPX(Vector* X, double a){
	
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecAYPX(this->vector,a,X->vector);
	#else
		_error_("not implemented yet!");
	#endif

}
/*}}}*/
/*FUNCTION Vector::ToMPISerial{{{1*/
double* Vector::ToMPISerial(void){

	double* vec_serial=NULL;

	#ifdef _HAVE_PETSC_
		VecToMPISerial(&vec_serial, this->vector);
	#else
		_error_("not implemented yet!");
	#endif

	return vec_serial;

}
/*}}}*/
/*FUNCTION Vector::Copy{{{1*/
void Vector::Copy(Vector* to){

	#ifdef _HAVE_PETSC_
		VecCopy(this->vector,to->vector);
	#else
		_error_("not implemented yet!");
	#endif

}
/*}}}*/
/*FUNCTION Vector::Norm{{{1*/
double Vector::Norm(NormMode norm_type){
	
	double norm=0;
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecNorm(this->vector,ISSMToPetscNormMode(norm_type),&norm);
	#else
		_error_("not implemented yet!");
	#endif
	return norm;
}
/*}}}*/
/*FUNCTION Vector::Scale{{{1*/
void Vector::Scale(double scale_factor){
	
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecScale(this->vector,scale_factor); 
	#else
		_error_("not implemented yet!");
	#endif
}
/*}}}*/
/*FUNCTION Vector::Dot{{{1*/
double Vector::Dot(Vector* vector){

	double dot;
	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecDot(this->vector,vector->vector,&dot);
	#else
		_error_("not implemented yet!");
	#endif
	return dot;
}
/*}}}*/
/*FUNCTION Vector::PointwiseDivide{{{1*/
void Vector::PointwiseDivide(Vector* x,Vector* y){

	#ifdef _HAVE_PETSC_
		_assert_(this->vector);
		VecPointwiseDivide(this->vector,x->vector,y->vector);
	#else
		_error_("not implemented yet!");
	#endif
}
/*}}}*/
