/*\file SystemMatrices.c
 *\brief: build system matrices (stiffness matrix, loads vector)
 */

#include "./SystemMatrices.h"

void mexFunction( int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[]){

	/*input datasets: */
	Elements   *elements   = NULL;
	Nodes      *nodes      = NULL;
	Vertices   *vertices   = NULL;
	Loads      *loads      = NULL;
	Materials  *materials  = NULL;
	Parameters *parameters = NULL;
	bool        kflag,pflag,penalty_kflag,penalty_pflag;
	
	/* output datasets: */
	Matrix*    Kff  = NULL;
	Matrix*    Kfs  = NULL;
	Vector*    pf   = NULL;
	Vector*    df   = NULL;

	double kmax;

	/*Boot module: */
	MODULEBOOT();

	/*checks on arguments on the matlab side: */
	if((nlhs!=NLHS) || (nrhs!=6 && nrhs!=10)){
		SystemMatricesUsage();
		_error_(" usage. See above");
	}

	/*Input datasets: */
	FetchMatlabData((DataSet**)&elements,ELEMENTS);
	FetchMatlabData((DataSet**)&nodes,NODES);
	FetchMatlabData((DataSet**)&vertices,VERTICES);
	FetchMatlabData((DataSet**)&loads,LOADS);
	FetchMatlabData((DataSet**)&materials,MATERIALS);
	FetchMatlabData(&parameters,PARAMETERS);

	/*configure: */
	elements->  Configure(elements,loads, nodes,vertices, materials,parameters);
	nodes->     Configure(elements,loads, nodes,vertices, materials,parameters);
	loads->     Configure(elements, loads, nodes,vertices, materials,parameters);
	materials-> Configure(elements, loads, nodes,vertices, materials,parameters);

	/*!Generate internal degree of freedom numbers: */
	if(nrhs==10){
		FetchMatlabData(&kflag,KFLAG);
		FetchMatlabData(&pflag,PFLAG);
		FetchMatlabData(&penalty_kflag,PENALTYKFLAG);
		FetchMatlabData(&penalty_pflag,PENALTYPFLAG);
		SystemMatricesx(&Kff,&Kfs,&pf,&df,&kmax,elements,nodes,vertices,loads,materials,parameters,kflag,pflag,penalty_kflag,penalty_pflag);
	}
	else
	 SystemMatricesx(&Kff,&Kfs,&pf,&df,&kmax,elements,nodes,vertices,loads,materials,parameters);

	/*write output datasets: */
	WriteMatlabData(KFF,Kff);
	WriteMatlabData(KFS,Kfs);
	WriteMatlabData(PF,pf);
	WriteMatlabData(DF,df);
	WriteMatlabData(KMAX,kmax);
	
	/*Free ressources: */
	delete elements;
	delete nodes;
	delete vertices;
	delete loads;
	delete materials;
	delete parameters;
	xdelete(&Kff);
	xdelete(&Kfs);
	xdelete(&pf);
	xdelete(&df);

	/*end module: */
	MODULEEND();
}

void SystemMatricesUsage(void)
{
	_printf_(true,"\n");
	_printf_(true,"   usage: [Kff,Kfs,pf,df,kmax] = %s(elements,nodes,vertices,loads,materials,parameters);\n",__FUNCT__);
	_printf_(true,"   usage: [Kff,Kfs,pf,df,kmax] = %s(elements,nodes,vertices,loads,materials,parameters,kflag,pflag,penalty_kflag,penalty_pflag);\n",__FUNCT__);
	_printf_(true,"\n");
}
