/*!\file InterpFromMeshToMesh3d.c
 * \brief: data interpolation from a list of (x,y,values) into mesh vertices
 
	InterpFromMeshToMesh3d.c

	usage:
	data_mesh=InterpFromMeshToMesh3d(index,x,y,z,data,x_mesh,y_mesh,z_mesh);
	
	where:

		input:
		x,y,z: coordinates of matrix data
		data - matrix holding the data to be interpolated onto the mesh.
		x_mesh,y_mesh,z_mesh: coordinates of the mesh vertices onto which we interpolate.
		
		output: 
		data_mesh:  vector of mesh interpolated data.

*/
	
#include "./InterpFromMeshToMesh3d.h"

void mexFunction( int nlhs, mxArray* plhs[], int nrhs, const mxArray* prhs[]) {

	/*input: */
	double* index_data=NULL;
	int     index_data_rows;

	double* x_data=NULL;
	double* y_data=NULL;
	double* z_data=NULL;

	int     x_data_rows;
	int     y_data_rows;
	int     z_data_rows;

	double* data=NULL; 
	int     data_rows;
	int     data_cols;

	double* x_prime=NULL;
	double* y_prime=NULL;
	double* z_prime=NULL;
	
	int     x_prime_rows;
	int     y_prime_rows;
	int     z_prime_rows;

	double  default_value;

	/*Intermediary*/
	int nods_data;
	int nels_data;
	int nods_prime;

	/* output: */
	Vector*  data_prime=NULL;

	/*Boot module: */
	MODULEBOOT();

	/*checks on arguments on the matlab side: */
	CheckNumMatlabArguments(nlhs,NLHS,nrhs,NRHS,__FUNCT__,&InterpFromMeshToMesh3dUsage);

	/*Input datasets: */
	FetchMatlabData(&index_data,&index_data_rows,NULL,INDEXHANDLE);
	FetchMatlabData(&x_data,&x_data_rows,NULL,XHANDLE);
	FetchMatlabData(&y_data,&y_data_rows,NULL,YHANDLE);
	FetchMatlabData(&z_data,&z_data_rows,NULL,ZHANDLE);
	FetchMatlabData(&data,&data_rows,&data_cols,DATAHANDLE);
	FetchMatlabData(&x_prime,&x_prime_rows,NULL,XPRIMEHANDLE);
	FetchMatlabData(&y_prime,&y_prime_rows,NULL,YPRIMEHANDLE);
	FetchMatlabData(&z_prime,&z_prime_rows,NULL,ZPRIMEHANDLE);
	FetchMatlabData(&default_value,DEFAULTHANDLE);

	/*some checks*/
	if (x_data_rows!=y_data_rows || x_data_rows!=z_data_rows){
		_error_("vectors x, y and z should have the same length!");
	}
	if (x_prime_rows!=y_prime_rows || x_prime_rows!=z_prime_rows){
		_error_("vectors x_prime, y_prime and z_prime should have the same length!");
	}
	/*get number of elements and number of nodes in the data*/
	nels_data=index_data_rows;
	nods_data=x_data_rows;
	nods_prime=x_prime_rows;

	/* Run core computations: */
	InterpFromMeshToMesh3dx(&data_prime,index_data,x_data,y_data,z_data,nods_data,nels_data,data,data_rows,x_prime,y_prime,z_prime,nods_prime,default_value);

	/*Write data: */
	WriteMatlabData(DATAPRIME,data_prime);

	/*end module: */
	MODULEEND();
}

void InterpFromMeshToMesh3dUsage(void)
{
	_printf_(true,"INTERPFROMMESHTOMESH3D - interpolation from a 3d hexahedron mesh onto a list of point\n");
	_printf_(true,"\n");
	_printf_(true,"   This function is a multi-threaded mex file that interpolates a field\n");
	_printf_(true,"   defined on a triangular mesh onto a list of point\n");
	_printf_(true,"\n");
	_printf_(true,"   Usage:\n");
	_printf_(true,"      data_prime=InterpFromMeshToMesh3d(index,x,y,z,data,x_prime,y_prime,z_prime,default_value);\n");
	_printf_(true,"\n");
	_printf_(true,"      index: index of the mesh where data is defined\n");
	_printf_(true,"      x,y,z: coordinates of the nodes where data is defined\n");
	_printf_(true,"      data: matrix holding the data to be interpolated onto the mesh.\n");
	_printf_(true,"      x_prime,y_prime,z_prime: coordinates of the points onto which we interpolate.\n");
	_printf_(true,"      default_value: default value if no data is found (holes).\n");
	_printf_(true,"      data_prime: vector of mesh interpolated data.\n");
	_printf_(true,"\n");
	_printf_(true,"   Example:\n");
	_printf_(true,"      load('temperature.mat');\n");
	_printf_(true,"      md.initialization.temperature=InterpFromMeshToMesh3d(index,x,y,z,temperature,md.mesh.x,md.mesh.y,md.mesh.z,253);\n");
	_printf_(true,"\n");
}
