/*!\file Vector.cpp
 * \brief: implementation of the Vector object
 */

/*Headers:*/
/*{{{1*/
#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <stdio.h>
#include <string.h>
#include "../objects.h"
#include "../../shared/shared.h"
#include "../../Container/Container.h"
#include "../../include/include.h"
#include "./Vector.h"
/*}}}*/

/*Vector constructors and destructor*/
/*FUNCTION Vector::Vector(){{{1*/
Vector::Vector(){

	this->M=0;
	#ifdef _HAVE_PETSC_
	this->vector=NULL;
	#else
	this->vector=NULL;
	#endif
	#ifdef _HAVE_ADOLC_
	this->avector=NULL;
	#endif
}
/*}}}*/
/*FUNCTION Vector::Vector(int M,bool fromlocalsize){{{1*/
Vector::Vector(int pM,bool fromlocalsize){

	#ifdef _HAVE_PETSC_
	this->vector=NewVec(pM,fromlocalsize);
	#else
	this->M=pM;
	this->vector=(double*)xcalloc(pM,sizeof(double));
	#endif
	#ifdef _HAVE_ADOLC_
	this->avector=(adouble*)xmalloc(pM*sizeof(adouble));
	#endif
}
/*}}}*/
/*FUNCTION Vector::~Vector(){{{1*/
Vector::~Vector(){

 	#ifdef _HAVE_PETSC_
	VecFree(&this->vector);
	#else
	xfree((void**)&this->vector);
	#endif
	#ifdef _HAVE_ADOLC_
	xfree((void**)&this->avector);
	#endif
}
/*}}}*/

/*Vector specific routines: */
/*FUNCTION Vector::Echo{{{1*/
void Vector::Echo(void){

	int i;

	#ifdef _HAVE_PETSC_
	VecView(this->vector,PETSC_VIEWER_STDOUT_WORLD);
	#else
	printf("Vector size: %i\n",M);
	for(i=0;i<M;i++){
		printf("%g\n ",*(vector+i));
	}
	#endif

	#ifdef _HAVE_ADOLC_
	/*Not sure about that one. Should we use the overloaded operator >>?*/
	printf("ADOLC Vector equivalent:" );
	for(i=0;i<M;i++){
		printf("%g\n ",*(avector+i));
	}
	#endif
}
/*}}}*/

#ifdef _SERIAL_
/*FUNCTION Vector::ToMatlabVector{{{1*/
mxArray* Vector::ToMatlabVector(void){

	mxArray* dataref=NULL;
	#ifdef _HAVE_PETSC_
	PetscVectorToMatlabVector(&dataref,this->vector);
	#else
	_error_("not implemented yet!");
	#endif
	return dataref;

}
/*}}}*/
#endif
/*FUNCTION Vector::Assemble{{{1*/
void Vector::Assemble(void){
		
	#ifdef _HAVE_PETSC_
		VecAssemblyBegin(this->vector); 
		VecAssemblyEnd(this->vector);
	#else
		/*do nothing*/
	#endif

}
/*}}}*/
/*FUNCTION Vector::SetValues{{{1*/
void Vector::SetValues(int ssize, int* list, double* values, int mode){
		
		
	#ifdef _HAVE_PETSC_
	VecSetValues(this->vector,ssize,list,values,(InsertMode)mode);
	#else
		_error_("not implemented yet!");
	#endif

}
/*}}}*/
/*FUNCTION Vector::GetSize{{{1*/
void Vector::GetSize(int* pM){
		
	#ifdef _HAVE_PETSC_
		VecGetSize(this->vector,pM);
	#else
		_error_("not implemented yet!");
	#endif

}
/*}}}*/
/*FUNCTION Vector::GetLocalSize{{{1*/
void Vector::GetLocalSize(int* pM){
		
	#ifdef _HAVE_PETSC_
		VecGetLocalSize(this->vector,pM);
	#else
		_error_("not implemented yet!");
	#endif

}
/*}}}*/
/*FUNCTION Vector::Duplicate{{{1*/
Vector* Vector::Duplicate(void){
	
	Vector* output=NULL;
	output=new Vector();
	#ifdef _HAVE_PETSC_
		Vec vec_output=NULL;
		VecDuplicate(this->vector,&vec_output);
		output->vector=vec_output;
		VecGetSize(output->vector,&output->M);
	#else
		_error_("not implemented yet!");
	#endif
	return output;

}
/*}}}*/
/*FUNCTION Vector::Set{{{1*/
void Vector::Set(double value){
	
	#ifdef _HAVE_PETSC_
		this->vector->Set(value);
	#else
		_error_("not implemented yet!");
	#endif

}
/*}}}*/
/*FUNCTION Vector::AXPY{{{1*/
void Vector::AXPY(Vector* X, double a){
	
	#ifdef _HAVE_PETSC_
		VecAXPY(this->vector,a,X->vector);
	#else
		_error_("not implemented yet!");
	#endif

}
/*}}}*/
/*FUNCTION Vector::AYPX{{{1*/
void Vector::AYPX(Vector* X, double a){
	
	#ifdef _HAVE_PETSC_
		VecAYPX(this->vector,a,X->vector);
	#else
		_error_("not implemented yet!");
	#endif

}
/*}}}*/
/*FUNCTION Vector::ToMPISerial{{{1*/
double* Vector::ToMPISerial(void){

	double* vec_serial=NULL;

	#ifdef _HAVE_PETSC_
		VecToMPISerial(&vec_serial, this->vector);
	#else
		_error_("not implemented yet!");
	#endif

	return vec_serial;

}
/*}}}*/
/*FUNCTION Vector::Copy{{{1*/
void Vector::Copy(Vector* to){

	#ifdef _HAVE_PETSC_
		VecCopy(this->vector,to->vector);
	#else
		_error_("not implemented yet!");
	#endif

}
/*}}}*/
/*FUNCTION Vector::Norm{{{1*/
double Vector::Norm(int norm_type){
	
	double norm=0;
	#ifdef _HAVE_PETSC_
	    VecNorm(this->vector,(NormType)norm_type,&norm);
	#else
		_error_("not implemented yet!");
	#endif
	return norm;
}
/*}}}*/
/*FUNCTION Vector::Scale{{{1*/
void Vector::Scale(double scale_factor){
	
	#ifdef _HAVE_PETSC_
	    VecScale(this->vector,scale_factor); 
	#else
		_error_("not implemented yet!");
	#endif
}
/*}}}*/
