/*!\file Matrix.cpp
 * \brief: implementation of the Matrix object
 */

/*Headers:*/
/*{{{1*/
#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <stdio.h>
#include <string.h>
#include "../objects.h"
#include "../../shared/shared.h"
#include "../../Container/Container.h"
#include "../../include/include.h"
#include "./Matrix.h"
/*}}}*/

/*Matrix constructors and destructor*/
/*FUNCTION Matrix::Matrix(){{{1*/
Matrix::Matrix(){

	this->M=0;
	this->N=0;

	#ifdef _HAVE_PETSC_
	this->matrix=NULL;
	#else
	this->matrix=NULL;
	#endif
	#ifdef _HAVE_ADOLC_
	this->amatrix=NULL;
	#endif
}
/*}}}*/
/*FUNCTION Matrix::Matrix(int M,int N){{{1*/
Matrix::Matrix(int pM,int pN){

	this->M=pM;
	this->N=pN;

	#ifdef _HAVE_PETSC_
	this->matrix=NewMat(pM,pN);
	#else
	this->matrix=(double*)xcalloc(pM*pN,sizeof(double));
	#endif
	#ifdef _HAVE_ADOLC_
	this->amatrix=(adouble*)xmalloc(pM*pN*sizeof(adouble));
	#endif
}
/*}}}*/
/*FUNCTION Matrix::Matrix(int M,int N,double sparsity){{{1*/
Matrix::Matrix(int pM,int pN,double sparsity){

	this->M=pM;
	this->N=pN;

	#ifdef _HAVE_PETSC_
	this->matrix=NewMat(pM,pN,sparsity);
	#else
	this->matrix=(double*)xcalloc(pM*pN,sizeof(double));
	#endif
	#ifdef _HAVE_ADOLC_
	this->amatrix=(adouble*)xmalloc(pM*pN*sizeof(adouble));
	#endif
}
/*}}}*/
/*FUNCTION Matrix::Matrix(int M,int N,int connectivity,int numberofdofspernode){{{1*/
Matrix::Matrix(int pM,int pN,int connectivity,int numberofdofspernode){
	
	this->M=pM;
	this->N=pN;

	#ifdef _HAVE_PETSC_
	this->matrix=NewMat(pM,pN,connectivity,numberofdofspernode);
	#else
	this->matrix=(double*)xcalloc(pM*pN,sizeof(double));
	#endif
	#ifdef _HAVE_ADOLC_
	this->amatrix=(adouble*)xmalloc(pM*pN*sizeof(adouble));
	#endif
}
/*}}}*/
/*FUNCTION Matrix::~Matrix(){{{1*/
Matrix::~Matrix(){

 	#ifdef _HAVE_PETSC_
	MatFree(&this->matrix);
	#else
	xfree((void**)&this->matrix);
	#endif
	#ifdef _HAVE_ADOLC_
	xfree((void**)&this->amatrix);
	#endif
}
/*}}}*/

/*Matrix specific routines: */
/*FUNCTION Matrix::Echo{{{1*/
void Matrix::Echo(void){

	int i,j;

	#ifdef _HAVE_PETSC_
	MatView(this->matrix,PETSC_VIEWER_STDOUT_WORLD);
	#else
	printf("Matrix size: %i-%i\n",M,N);
	for(i=0;i<M;i++){
		for(j=0;j<N;j++){
			printf("%g ",*(matrix+N*i+j));
		}
		printf("\n");
	}
	#endif

	#ifdef _HAVE_ADOLC_
	/*Not sure about that one. Should we use the overloaded operator >>?*/
	printf("ADOLC Matrix equivalent:" );
	for(i=0;i<M;i++){
		for(j=0;j<N;j++){
			printf("%g ",*(amatrix+N*i+j));
		}
		printf("\n");
	}
	#endif
}
/*}}}*/

#ifdef _SERIAL_
/*FUNCTION Matrix::ToMatlabMatrix{{{1*/
mxArray* Matrix::ToMatlabMatrix(void){

	mxArray* dataref=NULL;
	#ifdef _HAVE_PETSC_
	PetscMatrixToMatlabMatrix(&dataref,this->matrix);
	#else
	_error_("not implemented yet!");
	#endif
	return dataref;

}
/*}}}*/
#endif
/*FUNCTION Matrix::Assemble{{{1*/
void Matrix::Assemble(void){
	#ifdef _HAVE_PETSC_
		MatAssemblyBegin(this->matrix,MAT_FINAL_ASSEMBLY);
		MatAssemblyEnd(this->matrix,MAT_FINAL_ASSEMBLY);
		#if _PETSC_MAJOR_ == 2 
			MatCompress(this->matrix);
		#endif
	#else
		/*do nothing:*/
	#endif

}
/*}}}*/
/*FUNCTION Matrix::Norm{{{1*/
double Matrix::Norm(int norm_type){
	
	double norm=0;
	#ifdef _HAVE_PETSC_
	    MatNorm(this->matrix,(NormType)norm_type,&norm);
	#else
		_error_("not implemented yet!");
	#endif
	return norm;
}
/*}}}*/
/*FUNCTION Matrix::GetSize{{{1*/
void Matrix::GetSize(int* pM,int* pN){
	
	#ifdef _HAVE_PETSC_
	    MatGetSize(this->matrix,pM,pN);
	#else
		_error_("not implemented yet!");
	#endif
}
/*}}}*/
/*FUNCTION Matrix::GetLocalSize{{{1*/
void Matrix::GetLocalSize(int* pM,int* pN){
	
	#ifdef _HAVE_PETSC_
	    MatGetLocalSize(this->matrix,pM,pN);
	#else
		_error_("not implemented yet!");
	#endif
}
/*}}}*/
/*FUNCTION Matrix::MatMult{{{1*/
void Matrix::MatMult(Vector* X,Vector* AX){

	#ifdef _HAVE_PETSC_
	    MatMultPatch(this->matrix,X->vector,AX->vector);
	#else
		_error_("not implemented yet!");
	#endif
}
/*}}}*/
