%GENERIC cluster class definition
%
%   Usage:
%      cluster=generic('name','astrid',);
%      cluster=generic('name','astrid','np',3);
%      cluster=generic('name',oshostname(),'np',3,'login','username');

classdef generic
    properties (SetAccess=public) 
		 % {{{1
		 name='';
		 login='';
		 np=1;
		 port=0;
		 interactive=1;
		 codepath=[issmtier() '/bin'];
		 executionpath=[issmtier() '/execution'];
		 valgrind=[issmtier() '/externalpackages/valgrind/install/bin/valgrind'];
		 valgrindlib=[issmtier() '/externalpackages/valgrind/install/lib/libmpidebug.so'];
		 valgrindsup=[issmtier() '/externalpackages/valgrind/issm.supp'];
		 %}}}
	 end
	 methods
		 function cluster=generic(varargin) % {{{1

			 %use provided options to change fields
			 options=pairoptions(varargin{:});

			 %get name
			 if ~exist(options,'name'), error('option ''name'' has not been provided'); end
			 cluster.name=getfieldvalue(options,'name');

			 %initialize cluster using user settings if provided
			 if (exist([cluster.name '_settings'])==2), eval([cluster.name '_settings']); end

			 %OK get other fields
			 for i=1:size(options.list,1),
				 fieldname=options.list{i,1};
				 fieldvalue=options.list{i,2};
				 if ismember(fieldname,properties('generic')),
					 cluster.(fieldname)=fieldvalue;
				 else
					 disp(['''' fieldname ''' is not a property of cluster generic']);
				 end
			 end
		 end
		 %}}}
		 function disp(cluster) % {{{1
			 %  display the object
			 disp(sprintf('class ''%s'' object ''%s'' = ',class(cluster),inputname(1)));
			 disp(sprintf('    name: %s',cluster.name));
			 disp(sprintf('    login: %s',cluster.login));
			 disp(sprintf('    np: %i',cluster.np));
			 disp(sprintf('    port: %i',cluster.port));
			 disp(sprintf('    codepath: %s',cluster.codepath));
			 disp(sprintf('    executionpath: %s',cluster.executionpath));
			 disp(sprintf('    valgrind: %s',cluster.valgrind));
			 disp(sprintf('    valgrindlib: %s',cluster.valgrindlib));
			 disp(sprintf('    valgrindsup: %s',cluster.valgrindsup));
		 end
		 %}}}
		 function checkconsistency(cluster,md,solution,analyses) % {{{1
			 if cluster.np<1
				 checkmessage(['number of processors should be at least 1']);
			 end
			 if isnan(cluster.np),
				 checkessage('number of processors should not be NaN!');
			 end
		 end
		 %}}}
		 function BuildQueueScript(cluster,md) % {{{1
		 
			 %retrieve parameters
			 modelname=md.miscellaneous.name;
			 solution=md.private.solution;
			 isvalgrind=md.debug.valgrind;
			 isgprof=md.debug.gprof;

			 %open file for writing: 
			 if ~ispc,
				 fid=fopen([modelname '.queue'],'w');
			 else
				 fid=fopen([modelname '.bat'],'w');
			 end

			 %write instructions for launching a job on the cluster
			 if ~ispc,
				 fprintf(fid,'#!/bin/sh\n');
			 else
				 fprintf(fid,'@echo off\n');
			 end
			 
			 if ~isvalgrind,
				 if cluster.interactive
					 if ~ispc,
						 fprintf(fid,'mpiexec -np %i %s/issm.exe %s %s %s ',cluster.np,cluster.codepath,EnumToString(solution),cluster.executionpath,modelname);
					 else
						 fprintf(fid,'"%s/issm.exe" %s "%s" %s ',cluster.codepath,EnumToString(solution),cluster.executionpath,modelname);
					 end
				 else
					 if ~ispc,
						 fprintf(fid,'mpiexec -np %i %s/issm.exe %s %s %s 2> %s.errlog >%s.outlog ',cluster.np,cluster.codepath,EnumToString(solution),cluster.executionpath,modelname,modelname,modelname);
					 else
						 fprintf(fid,'"%s/issm.exe" %s "%s" %s 2> %s.errlog >%s.outlog ',cluster.codepath,EnumToString(solution),cluster.executionpath,modelname,modelname,modelname);
					 end
				 end
			 else
				 if ~ispc,
					 %Add --gen-suppressions=all to get suppression lines
					 fprintf(fid,'LD_PRELOAD=%s \\\n',cluster.valgrindlib);
					 fprintf(fid,'mpiexec -np %i %s --leak-check=full --suppressions=%s %s/issm.exe %s %s %s 2> %s.errlog >%s.outlog ',...
					 cluster.np,cluster.valgrind,cluster.valgrindsup, cluster.codepath,EnumToString(solution),cluster.executionpath,modelname,modelname,modelname);
				 else
					 error('valgrind not supported on windows platforms');
				 end
			 end

			 if isgprof,
				 if ~ispc,
					 fprintf(fid,'\n gprof %s/issm.exe gmon.out > %s.performance',cluster.codepath,modelname);
				 else
					 error('gprof not supported on windows platforms');
				 end

			 end

			 if ~md.settings.io_gather,
				 if ~ispc,
					 %concatenate the output files:
					 fprintf(fid,'\ncat %s.outbin.* > %s.outbin',modelname,modelname);
				 else
					 error('iogather not supported on windows platforms');
				 end

			 end
			 
			 %close file: 
			 fclose(fid);

			 %in interactive mode, create a run file, and errlog and outlog file
			 if cluster.interactive,
				 fid=fopen([modelname '.errlog'],'w'); fclose(fid);
				 fid=fopen([modelname '.outlog'],'w'); fclose(fid);
			 end


		 end
		 %}}}
		 function LaunchQueueJob(cluster,md,options)% {{{1
			 
			 if ~ispc,
					 %lauch command, to be executed via ssh
					 launchcommand=['cd ' cluster.executionpath ' && rm -rf ./' md.private.runtimename ' && mkdir ' md.private.runtimename ...
					 ' && cd ' md.private.runtimename ' && mv ../' md.private.runtimename '.tar.gz ./ && tar -zxf ' md.private.runtimename '.tar.gz  && source  ' md.miscellaneous.name '.queue '];

					 if ~strcmpi(options.batch,'yes'),

						 %compress the files into one zip.
						 compressstring=['tar -zcf ' md.private.runtimename '.tar.gz ' md.miscellaneous.name '.bin ' md.miscellaneous.name '.queue '  md.miscellaneous.name '.petsc '];
						 if md.qmu.isdakota,
							 compressstring=[compressstring md.miscellaneous.name '.qmu.in'];
					end
					if cluster.interactive,
						compressstring=[compressstring ' ' md.miscellaneous.name '.errlog ' md.miscellaneous.name '.outlog '];
					end
					system(compressstring);

					disp('uploading input file and queueing script');
					issmscpout(cluster.name,cluster.executionpath,cluster.login,cluster.port,{[md.private.runtimename '.tar.gz']});

					disp('launching solution sequence on remote cluster');
					issmssh(cluster.name,cluster.login,cluster.port,launchcommand);
				else
					disp('batch mode requested: not launching job interactively');
					disp('launch solution sequence on remote cluster by hand');
				end
			else
				%launch right here, do not compress or archive.
				system([md.miscellaneous.name '.bat']);
			end

		end %}}}
		 function Download(cluster,md)% {{{1

			if ~ispc,
				%some check
				if isempty(md.private.runtimename),
					error('supply runtime name for results to be loaded!');
				end

				%Figure out the  directory where all the files are in: 
				directory=[cluster.executionpath '/' md.private.runtimename '/'];

				%What packages are we picking up from remote cluster
				packages={[md.miscellaneous.name '.outlog'],[md.miscellaneous.name '.errlog']};
				if md.qmu.isdakota,
					packages{end+1}=[md.miscellaneous.name '.qmu.err'];
					packages{end+1}=[md.miscellaneous.name '.qmu.out'];
					if isfield(md.qmu.params,'tabular_graphics_data'),
						if md.qmu.params.tabular_graphics_data==true,
							packages{end+1}='dakota_tabular.dat'; 
						end
					end
				else
					packages{end+1}=[md.miscellaneous.name '.outbin'];
				end

				%copy files from cluster to present directory
				issmscpin(cluster.name, cluster.login, cluster.port, directory, packages);
			else
				%do nothing!
			end
		end %}}}
	end
end
