/*!\file Matice.c
 * \brief: implementation of the Matice object
 */

#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <stdio.h>
#include <string.h>
#include "../objects.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/shared.h"
#include "../../include/include.h"
		
/*Matice constructors and destructor*/
/*FUNCTION Matice::Matice(){{{1*/
Matice::Matice(){
	this->inputs=NULL;
	this->helement=NULL;
	return;
}
/*}}}*/
/*FUNCTION Matice::Matice(int id, int index, IoModel* iomodel, int num_vertices){{{1*/
Matice::Matice(int matice_mid,int index, IoModel* iomodel){

	/*Intermediaries:*/
	int    i;
	int    matice_eid;

	/*Initialize id*/
	this->mid=matice_mid;

	/*Initialize inputs*/
	this->inputs=new Inputs();

	/*Initialize inputs from IoModel*/
	this->InputUpdateFromIoModel(index,iomodel);

	/*Hooks: */
	matice_eid=index+1;
	this->helement=new Hook(&matice_eid,1);

	return;

}
/*}}}*/
/*FUNCTION Matice::~Matice(){{{1*/
Matice::~Matice(){
	delete helement;
	delete inputs;
	return;
}
/*}}}*/

/*Object virtual functions definitions:*/
/*FUNCTION Matice::Echo {{{1*/
void Matice::Echo(void){

	printf("Matice:\n");
	printf("   mid: %i\n",mid);
	printf("   inputs:\n");
	inputs->Echo();
	printf("   element:\n");
	helement->Echo();
}
/*}}}*/
/*FUNCTION Matice::DeepEcho {{{1*/
void Matice::DeepEcho(void){

	printf("Matice:\n");
	printf("   mid: %i\n",mid);
	printf("   inputs:\n");
	inputs->DeepEcho();
	printf("   element:\n");
	helement->Echo();
}		
/*}}}*/
/*FUNCTION Matice::Id {{{1*/
int    Matice::Id(void){ return mid; }
/*}}}*/
/*FUNCTION Matice::MyRank {{{1*/
int    Matice::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/
#ifdef _SERIAL_
/*FUNCTION Matice::Marshall {{{1*/
void  Matice::Marshall(char** pmarshalled_dataset){

	/*Intermediaries*/
	char* marshalled_dataset=NULL;
	int   enum_type=0;
	char* marshalled_inputs=NULL;
	int   marshalled_inputs_size;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Matice: */
	enum_type=MaticeEnum;
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	
	/*marshall Matice data: */
	memcpy(marshalled_dataset,&mid,sizeof(mid));marshalled_dataset+=sizeof(mid);

	/*Marshall hooks: */
	helement->Marshall(&marshalled_dataset);

	/*Marshall inputs: */
	marshalled_inputs_size=inputs->MarshallSize();
	marshalled_inputs=inputs->Marshall();
	memcpy(marshalled_dataset,marshalled_inputs,marshalled_inputs_size*sizeof(char));
	marshalled_dataset+=marshalled_inputs_size;

	*pmarshalled_dataset=marshalled_dataset;

	/*clean up and return*/
	xfree((void**)&marshalled_inputs);
}
/*}}}*/
/*FUNCTION Matice::MarshallSize{{{1*/
int   Matice::MarshallSize(){

	return sizeof(mid)
	  +helement->MarshallSize()
	  +inputs->MarshallSize()
	  +sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION Matice::Demarshall {{{1*/
void  Matice::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/
	memcpy(&mid,marshalled_dataset,sizeof(mid));marshalled_dataset+=sizeof(mid);

	/*demarshall hooks: */
	helement=new Hook(); helement->Demarshall(&marshalled_dataset);

	/*demarshall inputs: */
	inputs=(Inputs*)DataSetDemarshallRaw(&marshalled_dataset); 

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
#endif
/*FUNCTION Matice::ObjectEnum{{{1*/
int Matice::ObjectEnum(void){

	return MaticeEnum;

}
/*}}}*/
/*FUNCTION Matice::copy {{{1*/
Object* Matice::copy() {

	/*Output*/
	Matice* matice=NULL;

	/*Initialize output*/
	matice=new Matice();

	/*copy fields: */
	matice->mid=this->mid;
	matice->helement=(Hook*)this->helement->copy();
	if(this->inputs) matice->inputs=(Inputs*)this->inputs->Copy();
	else  matice->inputs=new Inputs();

	return matice;
}
/*}}}*/

/*Matice management*/
/*FUNCTION Matice::Configure {{{1*/
void  Matice::Configure(Elements* elementsin){

	/*Take care of hooking up all objects for this element, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	helement->configure(elementsin);
}
/*}}}*/
/*FUNCTION Matice::SetCurrentConfiguration {{{1*/
void  Matice::SetCurrentConfiguration(Elements* elementsin,Loads* loadsin,Nodes* nodesin,Vertices* verticesin,Materials* materialsin,Parameters* parametersin){

}
/*}}}*/
/*FUNCTION Matice::GetB {{{1*/
double Matice::GetB(){

	/*Output*/
	double B;

	inputs->GetInputAverage(&B,MaterialsRheologyBEnum);
	return B;
}
/*}}}*/
/*FUNCTION Matice::GetBbar {{{1*/
double Matice::GetBbar(){

	/*Output*/
	double Bbar;

	inputs->GetInputAverage(&Bbar,MaterialsRheologyBbarEnum);
	return Bbar;
}
/*}}}*/
/*FUNCTION Matice::GetN {{{1*/
double Matice::GetN(){

	/*Output*/
	double n;

	inputs->GetInputAverage(&n,MaterialsRheologyNEnum);
	return n;
}
/*}}}*/
/*FUNCTION Matice::GetZ {{{1*/
double Matice::GetZ(){

	/*Output*/
	double Z;

	inputs->GetInputAverage(&Z,MaterialsRheologyZEnum);
	return Z;
}
/*}}}*/
/*FUNCTION Matice::GetZbar {{{1*/
double Matice::GetZbar(){

	/*Output*/
	double Zbar;

	inputs->GetInputAverage(&Zbar,MaterialsRheologyZbarEnum);
	return Zbar;
}
/*}}}*/
/*FUNCTION Matice::GetVectorFromInputs{{{1*/
void  Matice::GetVectorFromInputs(Vec vector,int input_enum){

	/*Intermediaries*/
	Element *element= NULL;

	/*Recover element*/
	element=(Element*)helement->delivers();

	/*Check that input_enum is a material input*/
	if (!IsInput(input_enum)) return;

	switch(element->ObjectEnum()){

		case TriaEnum:{

			/*Prepare index list*/
			int doflist1[3];
			for(int i=0;i<3;i++) doflist1[i]=((Tria*)element)->nodes[i]->GetVertexDof();

			/*Get input (either in element or material)*/
			Input* input=inputs->GetInput(input_enum);
			if(!input) _error_("Input %s not found in material",EnumToStringx(input_enum));

			/*We found the enum.  Use its values to fill into the vector, using the vertices ids: */
			input->GetVectorFromInputs(vector,&doflist1[0]);}
			break;

		default: _error_("element %s not implemented yet",EnumToStringx(element->ObjectEnum()));
	}
}
/*}}}*/
/*FUNCTION Matice::GetViscosity2d {{{1*/
void  Matice::GetViscosity2d(double* pviscosity, double* epsilon){
	/*From a string tensor and a material object, return viscosity, using Glen's flow law.
												    B
	  viscosity= -------------------------------------------------------------------
						  2[ exx^2+eyy^2+exx*eyy+exy^2+exz^2+eyz^2 ]^[(n-1)/2n]

	  where viscosity is the viscotiy, B the flow law parameter , (u,v) the velocity 
	  vector, and n the flow law exponent.

	  If epsilon is NULL, it means this is the first time SystemMatrices is being run, and we 
	  return 10^14, initial viscosity.
	  */

	/*output: */
	double viscosity;

	/*input strain rate: */
	double exx,eyy,exy;

	/*Intermediary: */
	double A,e;
	double Btmp,B,n,Z;

	/*Get B and n*/
	Btmp=GetBbar();
	Z=GetZbar();
	n=GetN();
	B=Z*Btmp;

	if (n==1){
		/*Viscous behaviour! viscosity=B: */
		viscosity=B/2;
	}
	else{
		if((epsilon[0]==0) && (epsilon[1]==0) && (epsilon[2]==0)){
			viscosity=0.5*pow((double)10,(double)14);
		}
		else{
			/*Retrive strain rate components: */
			exx=*(epsilon+0);
			eyy=*(epsilon+1);
			exy=*(epsilon+2);

			/*Build viscosity: viscosity=B/(2*A^e) */
			A=pow(exx,2)+pow(eyy,2)+pow(exy,2)+exx*eyy;
			if(A==0){
				/*Maxiviscositym viscosity for 0 shear areas: */
				viscosity=2.5*pow((double)10,(double)17);
			}
			else{
				e=(n-1)/(2*n);
				viscosity=B/(2*pow(A,e));
			}
		}
	}

	/*Checks in debugging mode*/
	if(viscosity<=0) _error_("Negative viscosity");
	_assert_(B>0);
	_assert_(Z>0);
	_assert_(n>0);

	/*Return: */
	*pviscosity=viscosity;
}
/*}}}*/
/*FUNCTION Matice::GetViscosity3d {{{1*/
void  Matice::GetViscosity3d(double* pviscosity3d, double* epsilon){

	/*Return viscosity accounting for steady state power law creep [Thomas and MacAyeal, 1982]: 
	 *
	 *                                               B
	 * viscosity3d= -------------------------------------------------------------------
	 *                      2[ exx^2+eyy^2+exx*eyy+exy^2+exz^2+eyz^2 ]^[(n-1)/2n]
	 *
	 *     where mu is the viscotiy, B the flow law parameter , (u,v) the velocity 
	 *     vector, and n the flow law exponent.
	 *
	 * If epsilon is NULL, it means this is the first time Emg is being run, and we 
	 * return g, initial viscosity.
	 */
	
	/*output: */
	double viscosity3d;

	/*input strain rate: */
	double exx,eyy,exy,exz,eyz;

	/*Intermediaries: */
	double A,e;
	double Btmp,B,n,Z;

	/*Get B and n*/
	Btmp=GetB();
	Z=GetZ();
	n=GetN();
	B=Z*Btmp;

	if (n==1){
		/*Viscous behaviour! viscosity3d=B: */
		viscosity3d=B/2;
	}
	else{
		if((epsilon[0]==0) && (epsilon[1]==0) && (epsilon[2]==0) && 
				(epsilon[3]==0) && (epsilon[4]==0)){
			viscosity3d=0.5*pow((double)10,(double)14);
		}
		else{

			/*Retrive strain rate components: */
			exx=*(epsilon+0);
			eyy=*(epsilon+1);
			exy=*(epsilon+2);
			exz=*(epsilon+3);
			eyz=*(epsilon+4);

			/*Build viscosity: viscosity3d=2*B/(2*A^e) */
			A=pow(exx,2)+pow(eyy,2)+pow(exy,2)+pow(exz,2)+pow(eyz,2)+exx*eyy;
			if(A==0){
				/*Maxiviscosity3dm viscosity for 0 shear areas: */
				viscosity3d=2.25*pow((double)10,(double)17);
			}
			else{
				e=(n-1)/2/n;
			
				viscosity3d=B/(2*pow(A,e));
			}
		}
	}

	/*Checks in debugging mode*/
	if(viscosity3d<=0) _error_("Negative viscosity");
	_assert_(B>0);
	_assert_(Z>0);
	_assert_(n>0);

	/*Assign output pointers:*/
	*pviscosity3d=viscosity3d;
}
/*}}}*/
/*FUNCTION Matice::GetViscosity3dStokes {{{1*/
void  Matice::GetViscosity3dStokes(double* pviscosity3d, double* epsilon){
	/*Return viscosity accounting for steady state power law creep [Thomas and MacAyeal, 1982]: 
	 *
	 *                                          B
	 * viscosity3d= -------------------------------------------------------------------
	 *                   2[ exx^2+eyy^2+exx*eyy+exy^2+exz^2+eyz^2 ]^[(n-1)/2n]
	 *
	 *     where mu is the viscotiy, B the flow law parameter , (u,v) the velocity 
	 *     vector, and n the flow law exponent.
	 *
	 * If epsilon is NULL, it means this is the first time Emg is being run, and we 
	 * return g, initial viscosity.
	 */
	
	/*output: */
	double viscosity3d;

	/*input strain rate: */
	double exx,eyy,exy,exz,eyz,ezz;

	/*Intermediaries: */
	double A,e;
	double Btmp,B,n,Z;
	double eps0;

	/*Get B and n*/
	eps0=pow((double)10,(double)-27);
	Btmp=GetB();
	Z=GetZ();
	n=GetN();
	B=Z*Btmp;
	
	if (n==1){
		/*Viscous behaviour! viscosity3d=B: */
		viscosity3d=B/2;
	}
	else{
		if((epsilon[0]==0) && (epsilon[1]==0) && (epsilon[2]==0) && 
				(epsilon[3]==0) && (epsilon[4]==0) && (epsilon[5]==0)){
			viscosity3d=0.5*pow((double)10,(double)14);
		}
		else{

			/*Retrive strain rate components: */
			exx=*(epsilon+0);
			eyy=*(epsilon+1);
			ezz=*(epsilon+2); //not used
			exy=*(epsilon+3);
			exz=*(epsilon+4);
			eyz=*(epsilon+5);

			/*Build viscosity: viscosity3d=B/(2*A^e) */
			A=pow(exx,2)+pow(eyy,2)+pow(exy,2)+pow(exz,2)+pow(eyz,2)+exx*eyy+pow(eps0,2);
			if(A==0){
				/*Maxiviscosity3dm viscosity for 0 shear areas: */
				viscosity3d=2.25*pow((double)10,(double)17);
			}
			else{
				e=(n-1)/2/n;
				viscosity3d=B/(2*pow(A,e));
			}
		}
	}

	/*Checks in debugging mode*/
	if(viscosity3d<=0) _error_("Negative viscosity");
	_assert_(B>0);
	_assert_(n>0);

	/*Assign output pointers:*/
	*pviscosity3d=viscosity3d;
}
/*}}}*/
/*FUNCTION Matice::GetViscosityComplement {{{1*/
void  Matice::GetViscosityComplement(double* pviscosity_complement, double* epsilon){
	/*Return viscosity accounting for steady state power law creep [Thomas and MacAyeal, 1982]: 
	 *
	 *  										                1
	 * viscosity= -------------------------------------------------------------------
	 *  				  2[ exx^2+eyy^2+exx*eyy+exy^2+exz^2+eyz^2 ]^[(n-1)/2n]
	 *
	 * If epsilon is NULL, it means this is the first time Gradjb is being run, and we 
	 * return mu20, initial viscosity.
	 */
	
	/*output: */
	double viscosity_complement;

	/*input strain rate: */
	double exx,eyy,exy;

	/*Intermediary value A and exponent e: */
	double A,e;
	double B,n;

	/*Get B and n*/
	B=GetBbar(); /* why is this needed in this function? */
	n=GetN();

	if(epsilon){
		exx=*(epsilon+0);
		eyy=*(epsilon+1);
		exy=*(epsilon+2);

		/*Build viscosity: mu2=B/(2*A^e) */
		A=pow(exx,2)+pow(eyy,2)+pow(exy,2)+exx*eyy;
		if(A==0){
			/*Maximum viscosity_complement for 0 shear areas: */
			viscosity_complement=2.25*pow((double)10,(double)17);
		}
		else{
			e=(n-1)/(2*n);
		
			viscosity_complement=1/(2*pow(A,e));
		}
	}
	else{
		viscosity_complement=4.5*pow((double)10,(double)17);
	}

	/*Checks in debugging mode*/
	_assert_(B>0);
	_assert_(n>0);
	_assert_(viscosity_complement>0);
		
	/*Return: */
	*pviscosity_complement=viscosity_complement;
}
/*}}}*/
/*FUNCTION Matice::GetViscosityZComplement {{{1*/
void  Matice::GetViscosityZComplement(double* pviscosity_complement, double* epsilon){
	/*Return viscosity derivative for control method d(mu)/dZ: 
	 *
	 *  										               B 
	 * dviscosity= -------------------------------------------------------------------
	 *  				  2[ exx^2+eyy^2+exx*eyy+exy^2+exz^2+eyz^2 ]^[(n-1)/2n]
	 *
	 * If epsilon is NULL, it means this is the first time Gradjb is being run, and we 
	 * return mu20, initial viscosity.
	 */
	
	/*output: */
	double viscosity_complement;

	/*input strain rate: */
	double exx,eyy,exy;

	/*Intermediary value A and exponent e: */
	double A,e;
	double B,n;

	/*Get B and n*/
	B=GetBbar();
	n=GetN();

	if(epsilon){
		exx=*(epsilon+0);
		eyy=*(epsilon+1);
		exy=*(epsilon+2);

		/*Build viscosity: mu2=B/(2*A^e) */
		A=pow(exx,2)+pow(eyy,2)+pow(exy,2)+exx*eyy;
		if(A==0){
			/*Maximum viscosity_complement for 0 shear areas: */
			viscosity_complement=2.25*pow((double)10,(double)17);
		}
		else{
			e=(n-1)/(2*n);
		
			viscosity_complement=B/(2*pow(A,e));
		}
	}
	else{
		viscosity_complement=4.5*pow((double)10,(double)17);
	}

	/*Checks in debugging mode*/
	_assert_(B>0);
	_assert_(n>0);
	_assert_(viscosity_complement>0);
		
	/*Return: */
	*pviscosity_complement=viscosity_complement;
}
/*}}}*/
/*FUNCTION Matice::GetViscosityDerivativeEpsSquare{{{1*/
void  Matice::GetViscosityDerivativeEpsSquare(double* pmu_prime, double* epsilon){

	/*output: */
	double mu_prime;
	double mu,n,eff2;

	/*input strain rate: */
	double exx,eyy,exy,exz,eyz;

	/*Get visocisty and n*/
	GetViscosity3d(&mu,epsilon);
	n=GetN();

	if((epsilon[0]==0) && (epsilon[1]==0) && (epsilon[2]==0) && 
				(epsilon[3]==0) && (epsilon[4]==0)){
		mu_prime=0.5*pow((double)10,(double)14);
	}
	else{
		/*Retrive strain rate components: */
		exx=epsilon[0];
		eyy=epsilon[1];
		exy=epsilon[2];
		exz=epsilon[3];
		eyz=epsilon[4];
		eff2 = exx*exx + eyy*eyy + exx*eyy + exy*exy + exz*exz + eyz*eyz;

		mu_prime=(1-n)/(2*n) * mu/eff2;
	}

	/*Assign output pointers:*/
	*pmu_prime=mu_prime;
}
/*}}}*/
/*FUNCTION Matice::GetViscosity2dDerivativeEpsSquare{{{1*/
void  Matice::GetViscosity2dDerivativeEpsSquare(double* pmu_prime, double* epsilon){

	/*output: */
	double mu_prime;
	double mu,n,eff2;

	/*input strain rate: */
	double exx,eyy,exy,exz;

	/*Get visocisty and n*/
	GetViscosity2d(&mu,epsilon);
	n=GetN();

	if((epsilon[0]==0) && (epsilon[1]==0) && (epsilon[2]==0)){
		mu_prime=0.5*pow((double)10,(double)14);
	}
	else{
		/*Retrive strain rate components: */
		exx=epsilon[0];
		eyy=epsilon[1];
		exy=epsilon[2];
		eff2 = exx*exx + eyy*eyy + exx*eyy + exy*exy ;

		mu_prime=(1-n)/(2*n) * mu/eff2;
	}

	/*Assign output pointers:*/
	*pmu_prime=mu_prime;
}
/*}}}*/
/*FUNCTION Matice::InputDuplicate{{{1*/
void  Matice::InputDuplicate(int original_enum,int new_enum){

	/*Call inputs method*/
	if (IsInput(original_enum)) inputs->DuplicateInput(original_enum,new_enum);

}
/*}}}*/
/*FUNCTION Matice::InputUpdateFromVector(double* vector, int name, int type) {{{1*/
void  Matice::InputUpdateFromVector(double* vector, int name, int type){

	/*Intermediaries*/
	Element *element      = NULL;

	/*Recover element*/
	element=(Element*)helement->delivers();

	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	switch(type){

		case VertexEnum:

			switch(element->ObjectEnum()){

				case TriaEnum:
					double values[3];
					for (int i=0;i<3;i++) values[i]=vector[((Tria*)element)->nodes[i]->GetVertexDof()];
					this->inputs->AddInput(new TriaP1Input(name,values));
					return;

				default: _error_("element %s not implemented yet",EnumToStringx(element->ObjectEnum()));
			}
		default: _error_("type %i (%s) not implemented yet",type,EnumToStringx(type));
	}
}
/*}}}*/
/*FUNCTION Matice::InputUpdateFromVector(int* vector, int name, int type) {{{1*/
void  Matice::InputUpdateFromVector(int* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matice::InputUpdateFromVector(bool* vector, int name, int type) {{{1*/
void  Matice::InputUpdateFromVector(bool* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matice::InputUpdateFromVectorDakota(double* vector, int name, int type) {{{1*/
void  Matice::InputUpdateFromVectorDakota(double* vector, int name, int type){

	/*Intermediaries*/
	Element *element      = NULL;
	Parameters* parameters= NULL;
	int         dim;

	/*Recover element*/
	element=(Element*)helement->delivers();

	/*Check that name is an element input*/
	if (!IsInput(name)) return;

	switch(type){

		case VertexEnum:

			switch(element->ObjectEnum()){

				case TriaEnum:
					double values[3];
					for (int i=0;i<3;i++) values[i]=vector[((Tria*)element)->nodes[i]->GetSidList()]; //use sid list, to index into serial oriented vector 
					this->inputs->AddInput(new TriaP1Input(name,values));
					/*Special case for rheology B in 2D: Pourave land for this solution{{{2*/
					if(name==MaterialsRheologyBEnum){
						/*Are we in 2D?:*/
						if(element->ObjectEnum()==TriaEnum){
							parameters=((Tria*)(element))->parameters;
						}
						else{
							parameters=((Penta*)(element))->parameters;
						}
						parameters->FindParam(&dim,MeshDimensionEnum);
						if(dim==2){
							/*Dupliacte rheology input: */
							this->inputs->AddInput(new TriaP1Input(MaterialsRheologyBbarEnum,values));
						}
					}
					/*}}}*/
					return;

				default: _error_("element %s not implemented yet",EnumToStringx(element->ObjectEnum()));
			}
		default: _error_("type %i (%s) not implemented yet",type,EnumToStringx(type));
	}



}
/*}}}*/
/*FUNCTION Matice::InputUpdateFromMatrixDakota(int* vector, int name, int type) {{{1*/
void  Matice::InputUpdateFromMatrixDakota(double* matrix, int nrows, int ncols,int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matice::InputUpdateFromVectorDakota(int* vector, int name, int type) {{{1*/
void  Matice::InputUpdateFromVectorDakota(int* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matice::InputUpdateFromVectorDakota(bool* vector, int name, int type) {{{1*/
void  Matice::InputUpdateFromVectorDakota(bool* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matice::InputUpdateFromConstant(double constant, int name) {{{1*/
void  Matice::InputUpdateFromConstant(double constant, int name){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matice::InputUpdateFromConstant(int constant, int name) {{{1*/
void  Matice::InputUpdateFromConstant(int constant, int name){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matice::InputUpdateFromConstant(bool constant, int name) {{{1*/
void  Matice::InputUpdateFromConstant(bool constant, int name){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matice::InputUpdateFromSolution{{{1*/
void  Matice::InputUpdateFromSolution(double* solution){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Matice::InputUpdateFromIoModel{{{1*/
void Matice::InputUpdateFromIoModel(int index, IoModel* iomodel){

	int i,j;

	int    dim;
	bool   control_analysis;
	int    num_control_type;

	/*Fetch parameters: */
	iomodel->Constant(&dim,MeshDimensionEnum);
	iomodel->Constant(&control_analysis,InversionIscontrolEnum);
	if(control_analysis) iomodel->Constant(&num_control_type,InversionNumControlParametersEnum);

	/*if 2d*/
	if(dim==2){

		/*Intermediaries*/
		const int num_vertices = 3; //Tria has 3 vertices
		double    nodeinputs[num_vertices];
		double    cmmininputs[num_vertices];
		double    cmmaxinputs[num_vertices];

		/*Get B*/
		if (iomodel->Data(MaterialsRheologyBEnum)) {
			for(i=0;i<num_vertices;i++) nodeinputs[i]=iomodel->Data(MaterialsRheologyBEnum)[int(iomodel->Data(MeshElementsEnum)[num_vertices*index+i]-1)];
			this->inputs->AddInput(new TriaP1Input(MaterialsRheologyBbarEnum,nodeinputs));
		}

		/*Get n*/
		if (iomodel->Data(MaterialsRheologyNEnum)) {
			for(i=0;i<num_vertices;i++) nodeinputs[i]=iomodel->Data(MaterialsRheologyNEnum)[index];
			this->inputs->AddInput(new TriaP1Input(MaterialsRheologyNEnum,nodeinputs));
		}
		
		/*Get Z*/
		if (iomodel->Data(MaterialsRheologyZEnum)) {
			for(i=0;i<num_vertices;i++) nodeinputs[i]=iomodel->Data(MaterialsRheologyZEnum)[int(iomodel->Data(MeshElementsEnum)[num_vertices*index+i]-1)];
			this->inputs->AddInput(new TriaP1Input(MaterialsRheologyZbarEnum,nodeinputs));
		}

		/*Control Inputs*/
		#ifdef _HAVE_CONTROL_
		if (control_analysis && iomodel->Data(InversionControlParametersEnum)){
			for(i=0;i<num_control_type;i++){
				switch((int)iomodel->Data(InversionControlParametersEnum)[i]){
					case MaterialsRheologyBbarEnum:
						if (iomodel->Data(MaterialsRheologyBEnum)){
							_assert_(iomodel->Data(MaterialsRheologyBEnum));_assert_(iomodel->Data(InversionMinParametersEnum)); _assert_(iomodel->Data(InversionMaxParametersEnum)); 
							for(j=0;j<num_vertices;j++)nodeinputs[j]=iomodel->Data(MaterialsRheologyBEnum)[int(iomodel->Data(MeshElementsEnum)[num_vertices*index+j]-1)];
							for(j=0;j<num_vertices;j++)cmmininputs[j]=iomodel->Data(InversionMinParametersEnum)[int(iomodel->Data(MeshElementsEnum)[num_vertices*index+j]-1)*num_control_type+i];
							for(j=0;j<num_vertices;j++)cmmaxinputs[j]=iomodel->Data(InversionMaxParametersEnum)[int(iomodel->Data(MeshElementsEnum)[num_vertices*index+j]-1)*num_control_type+i];
							this->inputs->AddInput(new ControlInput(MaterialsRheologyBbarEnum,TriaP1InputEnum,nodeinputs,cmmininputs,cmmaxinputs,i+1));
						}
						break;
					case MaterialsRheologyZbarEnum:
						if (iomodel->Data(MaterialsRheologyZEnum)){
							_assert_(iomodel->Data(MaterialsRheologyZEnum));_assert_(iomodel->Data(InversionMinParametersEnum)); _assert_(iomodel->Data(InversionMaxParametersEnum)); 
							for(j=0;j<num_vertices;j++)nodeinputs[j]=iomodel->Data(MaterialsRheologyZEnum)[int(iomodel->Data(MeshElementsEnum)[num_vertices*index+j]-1)];
							for(j=0;j<num_vertices;j++)cmmininputs[j]=iomodel->Data(InversionMinParametersEnum)[int(iomodel->Data(MeshElementsEnum)[num_vertices*index+j]-1)*num_control_type+i];
							for(j=0;j<num_vertices;j++)cmmaxinputs[j]=iomodel->Data(InversionMaxParametersEnum)[int(iomodel->Data(MeshElementsEnum)[num_vertices*index+j]-1)*num_control_type+i];
							this->inputs->AddInput(new ControlInput(MaterialsRheologyZbarEnum,TriaP1InputEnum,nodeinputs,cmmininputs,cmmaxinputs,i+1));
						}
						break;
				}
			}
		}
		#endif
	}

	/*if 3d*/
	#ifdef _HAVE_THREED_
	else if(dim==3){

		/*Intermediaries*/
		const int num_vertices = 6; //Penta has 6 vertices
		double    nodeinputs[num_vertices];
		double    cmmininputs[num_vertices];
		double    cmmaxinputs[num_vertices];

		/*Get B*/
		if (iomodel->Data(MaterialsRheologyBEnum)) {
			for(i=0;i<num_vertices;i++) nodeinputs[i]=iomodel->Data(MaterialsRheologyBEnum)[int(iomodel->Data(MeshElementsEnum)[num_vertices*index+i]-1)];
			this->inputs->AddInput(new PentaP1Input(MaterialsRheologyBEnum,nodeinputs));
		}

		/*Get n*/
		if (iomodel->Data(MaterialsRheologyNEnum)) {
			for(i=0;i<num_vertices;i++) nodeinputs[i]=iomodel->Data(MaterialsRheologyNEnum)[index];
			this->inputs->AddInput(new PentaP1Input(MaterialsRheologyNEnum,nodeinputs));
		}
		
		/*Get Z*/
		if (iomodel->Data(MaterialsRheologyZEnum)) {
			for(i=0;i<num_vertices;i++) nodeinputs[i]=iomodel->Data(MaterialsRheologyZEnum)[int(iomodel->Data(MeshElementsEnum)[num_vertices*index+i]-1)];
			this->inputs->AddInput(new PentaP1Input(MaterialsRheologyZEnum,nodeinputs));
		}

		/*Control Inputs*/
		#ifdef _HAVE_CONTROL_
		if (control_analysis && iomodel->Data(InversionControlParametersEnum)){
			for(i=0;i<num_control_type;i++){
				switch((int)iomodel->Data(InversionControlParametersEnum)[i]){
					case MaterialsRheologyBbarEnum:
						if (iomodel->Data(MaterialsRheologyBEnum)){
							_assert_(iomodel->Data(MaterialsRheologyBEnum));_assert_(iomodel->Data(InversionMinParametersEnum)); _assert_(iomodel->Data(InversionMaxParametersEnum)); 
							for(j=0;j<num_vertices;j++)nodeinputs[j]=iomodel->Data(MaterialsRheologyBEnum)[int(iomodel->Data(MeshElementsEnum)[num_vertices*index+j]-1)];
							for(j=0;j<num_vertices;j++)cmmininputs[j]=iomodel->Data(InversionMinParametersEnum)[int(iomodel->Data(MeshElementsEnum)[num_vertices*index+j]-1)*num_control_type+i];
							for(j=0;j<num_vertices;j++)cmmaxinputs[j]=iomodel->Data(InversionMaxParametersEnum)[int(iomodel->Data(MeshElementsEnum)[num_vertices*index+j]-1)*num_control_type+i];
							this->inputs->AddInput(new ControlInput(MaterialsRheologyBEnum,PentaP1InputEnum,nodeinputs,cmmininputs,cmmaxinputs,i+1));
						}
						break;
					case MaterialsRheologyZbarEnum:
						if (iomodel->Data(MaterialsRheologyZEnum)){
							_assert_(iomodel->Data(MaterialsRheologyZEnum));_assert_(iomodel->Data(InversionMinParametersEnum)); _assert_(iomodel->Data(InversionMaxParametersEnum)); 
							for(j=0;j<num_vertices;j++)nodeinputs[j]=iomodel->Data(MaterialsRheologyZEnum)[int(iomodel->Data(MeshElementsEnum)[num_vertices*index+j]-1)];
							for(j=0;j<num_vertices;j++)cmmininputs[j]=iomodel->Data(InversionMinParametersEnum)[int(iomodel->Data(MeshElementsEnum)[num_vertices*index+j]-1)*num_control_type+i];
							for(j=0;j<num_vertices;j++)cmmaxinputs[j]=iomodel->Data(InversionMaxParametersEnum)[int(iomodel->Data(MeshElementsEnum)[num_vertices*index+j]-1)*num_control_type+i];
							this->inputs->AddInput(new ControlInput(MaterialsRheologyZEnum,PentaP1InputEnum,nodeinputs,cmmininputs,cmmaxinputs,i+1));
						}
						break;
				}
			}
		}
		#endif
	}
	#endif
	else{
		_error_(" Mesh type not supported yet!");
	}

	return;
}
/*}}}*/
/*FUNCTION Matice::IsInput{{{1*/
bool Matice::IsInput(int name){
	if (
				name==MaterialsRheologyBEnum ||
				name==MaterialsRheologyBbarEnum ||
				name==MaterialsRheologyNEnum ||
				name==MaterialsRheologyZEnum ||
				name==MaterialsRheologyZbarEnum 
		){
		return true;
	}
	else return false;
}
/*}}}*/
